﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/util/util_BinTypes.h>

#include <nn/gfxTool/gfxTool_Util.h>

#include <gfxTool_VariationManager.h>
#include <gfxTool_ShaderSourceManager.h>
#include <gfxTool_CompileOptionManager.h>
#include <gfxTool_CompilerManager.h>
#include <gfxTool_ShaderCompilerContext.h>
#include <gfxTool_CompileOutput.h>

namespace nn {
namespace gfxTool {

ShaderCompilerContext::ShaderCompilerContext()
    : m_State( static_cast< int >( State::Created ) )
{
}

ShaderCompilerContext::~ShaderCompilerContext()
{
}

nngfxToolResultCode ShaderCompilerContext::Reset()
{
    m_State = StaticCastAuto( State::Created );

    // ひとまず安全重視
    decltype( m_pShaderSourceManager )().swap( m_pShaderSourceManager );
    decltype( m_pCompileOptionManager )().swap( m_pCompileOptionManager );
    decltype( m_pVariationManager )().swap( m_pVariationManager );
    decltype( m_pCompilerManager )().swap( m_pCompilerManager );
    decltype( m_pCompileOutput )().swap( m_pCompileOutput );

    return nngfxToolResultCode_Succeeded;
}

nngfxToolResultCode ShaderCompilerContext::Compile( nngfxToolShaderCompilerCompileOutput** ppOutput,
    const nngfxToolShaderCompilerCompileArg* pArg )
{
    try
    {
        if( !CheckState( State::Created ) || CheckState( State::Compiled ) )
        {
            NN_GFXTOOL_THROW( nngfxToolResultCode_InvalidApiCall );
        }

        ValidateCompileArg( pArg );

        SetupOptions( pArg );

        NN_GFXTOOL_PRINT( "Variation Count: %u\n", pArg->variationCount );

        SetupSources( pArg );

        SetupVariations( pArg );

        CompileImpl( ppOutput, pArg );

        AddState( State::Compiled );
    }
    catch( ... )
    {
        return ReportException();
    }

    return nngfxToolResultCode_Succeeded;
}

void ShaderCompilerContext::SetupSources( const nngfxToolShaderCompilerCompileArg* pArg )
{
    if( pArg->shaderSourceFormat != nngfxToolShaderCompilerShaderSourceFormat_SpirV )
    {
        m_pShaderSourceManager.reset( new ShaderSourceManager() );
        m_pShaderSourceManager->Initialize( pArg );
    }
}

void ShaderCompilerContext::SetupOptions( const nngfxToolShaderCompilerCompileArg* pArg )
{
    m_pCompileOptionManager.reset( new CompileOptionManager() );
    m_pCompileOptionManager->Initialize( pArg );
}

void ShaderCompilerContext::SetupVariations( const nngfxToolShaderCompilerCompileArg* pArg )
{
    m_pVariationManager.reset( new VariationManager() );
    m_pVariationManager->Initialize( pArg );
}

void ShaderCompilerContext::CompileImpl( nngfxToolShaderCompilerCompileOutput** ppOutput,
    const nngfxToolShaderCompilerCompileArg* pArg )
{
    auto pOutput = CompileOutput::Create();
    pOutput->Initialize( pArg );
    pOutput->variationCount = pArg->variationCount;
    nn::util::BinVersion version = { MajorVersion, MinorVersion, MicroVersion };
    pOutput->compilerVersion = version.GetPacked();
    *ppOutput = static_cast< nngfxToolShaderCompilerCompileOutput* >( pOutput );

    m_pCompilerManager.reset( new CompilerManager() );
    m_pCompilerManager->Compile( pOutput, this, pArg );
}

void ShaderCompilerContext::ValidateCompileArg( const nngfxToolShaderCompilerCompileArg* pArg )
{
    if( pArg == nullptr )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "CompileArg" );
    }

    bool hasStage = false;
    for( int idxStage = 0; idxStage < static_cast<int>( ShaderStage::End ); ++idxStage )
    {
        if( GetStageSource( pArg, StaticCastAuto( idxStage ) ).pValue )
        {
            hasStage = true;
            break;
        }
    }
    if( !hasStage )
    {
        NN_GFXTOOL_THROW_MSG( nngfxToolResultCode_InvalidArgument, "There is no shader source." );
    }

    // TODO
}

ShaderCompilerContext::HandleType ShaderCompilerContext::Create()
{
    auto pContext = new ( std::nothrow ) ShaderCompilerContext();
    if( pContext )
    {
        pContext->AddState( State::Created );
    }
    return reinterpret_cast< HandleType >( pContext );
}

nngfxToolResultCode ShaderCompilerContext::Delete( HandleType handle )
{
    auto ptr = Get( handle );
    if( ptr == nullptr )
    {
        return nngfxToolResultCode_InvalidHandle;
    }
    delete ptr;
    return nngfxToolResultCode_Succeeded;
}

}
}
