﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/gfx/gfx_Enum.h>

#include <gfxTool_Compiler-d3d-source.h>
#include <gfxTool_ShaderCompilerContext.h>
#include <gfxTool_CompileOption-glsl.h>
#include <gfxTool_ShaderSourceManager.h>
#include <gfxTool_GroupSource.h>
#include <gfxTool_VariationGrouper.h>
#include <gfxTool_CompileOptionManager.h>
#include <gfxTool_VariationManager.h>
#include <gfxTool_CompileOutput.h>
#include <gfxTool_OptionOutput.h>
#include <gfxTool_BoostPreprocessor.h>

namespace {

template< typename TSource >
void SetSource( nn::gfxTool::SourceArrayOutput* pSourceArray, const TSource* pSource )
{
    if( pSource && !pSource->empty() )
    {
        pSourceArray->codeArray.emplace_back();
        pSourceArray->codeArray.back() = pSource->data();
        pSourceArray->codeSizeArray.push_back( nn::gfxTool::NumericCastAuto( pSource->size() ) );
    }
}

}

namespace nn {
namespace gfxTool {

typedef CompilerVariation< static_cast<
    nngfxToolShaderCompilerLowLevelApiType >( nngfxToolShaderCompilerLowLevelApiType_D3d ),
    nngfxToolShaderCompilerCodeType_Source > Target;

void Compiler< Target >::PreCompile( CompileOutput* pOutput,
    const ShaderCompilerContext* pContext, const nngfxToolShaderCompilerCompileArg* pArg )
{
    auto pSourceOutput = pOutput->GetSourceOutput();

    pSourceOutput->pGlobalPreprocessorSource = pContext->GetCompileOptionManager()->GetCompileOption<
        nngfxToolShaderCompilerOptionType_Common >()->GetPreprocessorDefinitionSource();

    for( int idxStage = 0; idxStage < static_cast< int >( ShaderStage::End ); ++idxStage )
    {
        auto stage = static_cast< ShaderStage >( idxStage );
        if( GetStageSource( pArg, stage ).pValue == nullptr )
        {
            continue;
        }
        pSourceOutput->pShaderSource[ idxStage ] =
            pContext->GetShaderSourceManager()->GetShaderSource( stage );
        pSourceOutput->pVariationConstantSource[ idxStage ] =
            pContext->GetVariationManager()->GetVariationConstantSource( stage );
        pSourceOutput->pPreprocessorDefinitionSource[ idxStage ] =
            pContext->GetVariationManager()->GetPreprocessorDefinitionSource( stage );
    }
}

void Compiler< Target >::CompileGroup( CompileOutput* pOutput, const ShaderCompilerContext* pContext,
    const nngfxToolShaderCompilerCompileArg* pArg, int idxGroup )
{
    auto pCommonOption = pContext->GetCompileOptionManager()->GetCompileOption<
        nngfxToolShaderCompilerOptionType_Common >();
    auto pCommonProgramOption = pContext->GetCompileOptionManager()->GetCompileOption<
        nngfxToolShaderCompilerOptionType_Common >();
    auto idxVariation = idxGroup;
    auto pSourceOutput = pOutput->GetSourceOutput();
    auto pProgramOutput = pOutput->GetVariationOutput( idxVariation )->GetSourceOutput();
    auto& info = *pProgramOutput->GetInfo();
    info.flags.SetBit( nn::gfx::ShaderInfoData::Flag_SeparationEnable,
        pCommonProgramOption->IsSeparationEnabled() );
    info.codeType = StaticCastAuto( nn::gfx::ShaderCodeType_Source );
    info.sourceFormat = StaticCastAuto( nn::gfx::ShaderSourceFormat_Hlsl );
    info.binaryFormat = 0;
    std::shared_ptr< OptionOutputProgramCommon > pOptionOutputProgramCommon(
        new OptionOutputProgramCommon() );
    pOptionOutputProgramCommon->Initialize( pArg );
    pProgramOutput->AddOptionOutput(
        nngfxToolShaderCompilerOptionOutputType_ProgramCommon, pOptionOutputProgramCommon );
    for( int idxStage = 0; idxStage < static_cast< int >( ShaderStage::End ); ++idxStage )
    {
        auto stage = static_cast< ShaderStage >( idxStage );
        auto& stageSource = GetStageSource( pArg, stage );
        if( stageSource.pValue == nullptr )
        {
            continue;
        }

        auto variationConstantGroup = pContext->GetVariationManager(
            )->GetVariationConstantGroup( stage )->GetVariationToGroupTable().at( idxVariation );
        auto& variationConstantSource = pSourceOutput->pVariationConstantSource[
            idxStage ].get()->GetSources().at( variationConstantGroup );
        auto preprocessorDefinitionGroup = pContext->GetVariationManager(
            )->GetPreprocessorDefinitionGroup( stage )->GetVariationToGroupTable().at( idxVariation );
        auto& preprocessorDefinitionSource = pSourceOutput->pPreprocessorDefinitionSource[
            idxStage ].get()->GetSources().at( preprocessorDefinitionGroup );

        if( pCommonOption->IsPreprocessEnabled() )
        {
            Custom< std::string >::Type source;
            source.append( pCommonOption->GetPreprocessorDefinitionSource()->c_str() ).append(
                preprocessorDefinitionSource.c_str() ).append(
                variationConstantSource.c_str() ).append(
                stageSource.pValue, stageSource.length );

            BoostPreprocessor preprocessor;
            preprocessor.SetReadFileCallback( pArg->pReadIncludeFileCallback,
                pArg->pReadIncludeFileCallbackParam );
            if( pArg->pVariationDefinitionArg )
            {
                preprocessor.SetVariationDefinition( GetStageVariationDefinition(
                    pArg->pVariationDefinitionArg, stage ) );
            }
            preprocessor.SetVariationConstantEmulationEnabled( true );
            preprocessor.Preprocess( source.c_str(), source.length() );

            auto codeSize = preprocessor.GetResult().str().length() + 1;
            std::shared_ptr< void > pCode( malloc( codeSize ), free );
            memcpy( pCode.get(), preprocessor.GetResult().str().c_str(), codeSize );
            pProgramOutput->SetShaderCode( stage, pCode, codeSize );

            if( pCommonProgramOption->IsDumpEnabled() )
            {
                std::shared_ptr< Custom< std::string >::Type > pDump( new Custom< std::string >::Type() );
                *pDump = preprocessor.GetResult().str();
                pOptionOutputProgramCommon->GetOptionOutputStageCommon( stage )->SetDump( pDump );
            }
        }
        else
        {
            auto pShaderSource = pSourceOutput->pShaderSource[ idxStage ].get();
            Custom< std::string >::Type source;
            source.append( pSourceOutput->pGlobalPreprocessorSource.get()->c_str() ).append(
                preprocessorDefinitionSource.c_str() ).append(
                variationConstantSource.c_str() ).append(
                pShaderSource->beforeVariationBufferView.data(), pShaderSource->beforeVariationBufferView.size() ).append(
                pShaderSource->variationBufferSource.data(), pShaderSource->variationBufferSource.size() ).append(
                pShaderSource->afterVariationBufferView.data(), pShaderSource->afterVariationBufferView.size() );

            auto codeSize = source.length() + 1;
            std::shared_ptr< void > pCode( malloc( codeSize ), free );
            memcpy( pCode.get(), source.c_str(), codeSize );
            pProgramOutput->SetShaderCode( stage, pCode, codeSize );

            if( pCommonProgramOption->IsDumpEnabled() )
            {
                std::shared_ptr< Custom< std::string >::Type > pDump( new Custom< std::string >::Type() );
                *pDump = source;
                pOptionOutputProgramCommon->GetOptionOutputStageCommon( stage )->SetDump( pDump );
            }
        }
    }
}

void Compiler< Target >::DumpShaderSource( ProgramOutput* pOutput,
    const SourceArrayOutput& sourceArray, ShaderStage stage )
{
    if( pOutput == nullptr )
    {
        return;
    }

    std::shared_ptr< Custom< std::string >::Type > pDump( new Custom< std::string >::Type() );
    for( int idxSource = 0; idxSource < static_cast< int >( sourceArray.codeArray.size() ); ++idxSource )
    {
        pDump.get()->append( static_cast< const char* >(
            sourceArray.codeArray[ idxSource ] ), sourceArray.codeSizeArray[ idxSource ] );
    }
    static_cast< OptionOutputProgramCommon* >( pOutput->GetOptionOutput(
        nngfxToolShaderCompilerOptionOutputType_ProgramCommon )
        )->GetOptionOutputStageCommon( stage )->SetDump( pDump );
}

}
}
