﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <thread>

#ifndef WIN32_LEAN_AND_MEAN
#define WIN32_LEAN_AND_MEAN
#endif
#ifndef NOMINMAX
#define NOMINMAX
#endif
#include <nn/nn_Windows.h>

#include <nn/gfxTool/gfxTool_Util.h>

#include <gfxTool_CompileOption.h>

namespace nn {
namespace gfxTool {

namespace {

int RetrieveLogicalProcessorCount()
{
    int ret = 0;
    DWORD size = 0;
    GetLogicalProcessorInformation( nullptr, &size );
    std::shared_ptr< SYSTEM_LOGICAL_PROCESSOR_INFORMATION > pBuffer(
        static_cast< SYSTEM_LOGICAL_PROCESSOR_INFORMATION* >( malloc( size ) ), free );
    if( GetLogicalProcessorInformation( pBuffer.get(), &size ) == TRUE )
    {
        for( int index = 0; index * sizeof( SYSTEM_LOGICAL_PROCESSOR_INFORMATION ) <
            static_cast< int >( size ); ++index )
        {
            auto& info = pBuffer.get()[ index ];
            if( info.Relationship == RelationProcessorCore )
            {
                for( int idxBit = 0; idxBit < sizeof( info.ProcessorMask ) * CHAR_BIT; ++idxBit )
                {
                    if( info.ProcessorMask & static_cast< decltype( info.ProcessorMask ) >( 1 ) << idxBit )
                    {
                        ++ret;
                    }
                }
            }
        }
    }
    return ret;
}

}

typedef CompileOption< nngfxToolShaderCompilerOptionType_Common >
    CompileOptionCommonProgram;

void CompileOptionCommonProgram::Initialize( const void* pOption )
{
    auto& option = *static_cast< const nngfxToolShaderCompilerCompileOptionCommon* >( pOption );
    m_IsDumpEnabled = option.dump != 0;
    m_PrintProgressRate = option.printProgress;
    m_IsSeparationEnabled = option.isSeparationEnabled != 0;
    m_IsShaderStatisticsEnabled = option.isShaderStatisticsEnabled != 0;
    m_IsReflectionEnabled = option.isReflectionEnabled != 0;
    m_IsCompressionEnabled = option.isCompressionEnabled != 0;
    m_IsPreprocessEnabled = option.isPreprocessEnabled != 0;
    m_IsInvertZEnabled = option.isInvertYEnabled != 0;
    m_IsRemapZEnabled = option.isRemapZEnabled != 0;
    m_DebugInfoLevel = StaticCastAuto( option.debugInfoLevel );
    if( m_PrintProgressRate < FLT_EPSILON )
    {
        // 0 以下は表示なし
        m_PrintProgressRate = 1.1f;
    }
    if( option.logOutputLevel == nngfxToolShaderCompilerLogOutputLevel_Silent )
    {
        Logger::SetLogOutputLevel( Logger::LogType::Default, Logger::LogLevel::Silent );
    }

    static const int s_DefaultThreadCount = 32;
    m_MaxThreads = NumericCastAuto( option.maxThreads );
    if( m_MaxThreads <= 0 )
    {
        m_MaxThreads = std::thread::hardware_concurrency();
    }
    if( m_MaxThreads <= 0 )
    {
        m_MaxThreads = RetrieveLogicalProcessorCount();
    }
    if( m_MaxThreads <= 0 )
    {
        m_MaxThreads = s_DefaultThreadCount;
    }

    m_CodePage = StaticCastAuto( option.codePage );

    m_UniformRegisterBlockName.assign(
        option.uniformRegisterBlockName.pValue, option.uniformRegisterBlockName.length );

    auto pDefineString = "#define ";
    m_pPreprocessorDefinitionSource.reset( new Custom< std::string >::Type() );
    decltype( ArgType< 0 >( &m_pPreprocessorDefinitionSource->reserve ) )
        preprocessorDefinitionSourceLength = 1;
    std::for_each( option.pPreprocessorDefinitionArray, option.pPreprocessorDefinitionArray
        + option.preprocessorDefinitionCount, [ & ]( decltype( *option.pPreprocessorDefinitionArray )&
        preprocessorDefinition ) { preprocessorDefinitionSourceLength += preprocessorDefinition.length; } );
    preprocessorDefinitionSourceLength += ( strlen( pDefineString ) + 2 ) * option.preprocessorDefinitionCount;
    m_pPreprocessorDefinitionSource->reserve( preprocessorDefinitionSourceLength );

    std::for_each( option.pPreprocessorDefinitionArray,
        option.pPreprocessorDefinitionArray + option.preprocessorDefinitionCount,
        [ & ]( decltype( *option.pPreprocessorDefinitionArray )& preprocessorDefinition ) {
            m_pPreprocessorDefinitionSource->append( pDefineString, strlen( pDefineString ) );
            m_pPreprocessorDefinitionSource->append( preprocessorDefinition.pValue,
                preprocessorDefinition.length );
            m_pPreprocessorDefinitionSource->append( "\n" );
    } );
}

}
}
