﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/dev.h>
#include <nw/gfnd.h>
#include <nw/math.h>
#include <nw/ut.h>
#include <FrameBuffer.h>
#include <Texture.h>
#include <nw/eft/eftut2_Heap.h>
#include <nw/eft2.h>

namespace nw      {
namespace eftdemo {

class Texture;

//---------------------------------------------------------------------------
//! @brief  2Dスクリーン描画クラス
//---------------------------------------------------------------------------
class ScreenDrawer
{
public:
    enum ShaderType
    {
        SHADER_TYPE_COPY                 = 0,
        SHADER_TYPE_LUMINANCE_SELECTION  = 1,
        SHADER_TYPE_GAUSS                = 2,
        SHADER_TYPE_GAUSS_X              = 3,
        SHADER_TYPE_GAUSS_Y              = 4,
        SHADER_TYPE_REDUCE               = 5,
        SHADER_TYPE_DEPTH_COPY           = 6,
        SHADER_TYPE_MAX                  = 7
    };

    static const u32 GAUSS_LENGTH = 5;

    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    ScreenDrawer();

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    ~ScreenDrawer(){}

    //---------------------------------------------------------------------------
    //! @brief        初期化処理です。
    //---------------------------------------------------------------------------
    void Initialize( nw::ut::IAllocator* allocator );

    //---------------------------------------------------------------------------
    //! @brief        初期化処理です。
    //---------------------------------------------------------------------------
#if defined(NW_PLATFORM_WIN32)
    void Initialize( nw::ut::IAllocator* allocator,
                           const u8* screenDrawerShaderVsh,          u32 screenDrawerShaderVshSize,
                           const u8* screenDrawerShaderCopyFsh,      u32 screenDrawerShaderCopyFshSize,
                           const u8* screenDrawerShaderLuminanceFsh, u32 screenDrawerShaderLuminanceFshSize,
                           const u8* screenDrawerShaderGaussFsh,     u32 screenDrawerShaderGaussFshSize,
                           const u8* screenDrawerShaderGaussXFsh,    u32 screenDrawerShaderGaussXFshSize,
                           const u8* screenDrawerShaderGaussYFsh,    u32 screenDrawerShaderGaussYFshSize,
                           const u8* screenDrawerShaderReduceFsh,    u32 screenDrawerShaderReduceFshSize,
                           const u8* screenDrawerShaderDepthCopyFsh, u32 screenDrawerShaderDepthCopyFshSize );
#endif
#if defined(NW_PLATFORM_CAFE)
    void Initialize( nw::ut::IAllocator* allocator, void* shaderRes, u32 shaderResSize );
#endif

    //---------------------------------------------------------------------------
    //! @brief        終了処理です。
    //---------------------------------------------------------------------------
    void Finalize();

    //---------------------------------------------------------------------------
    //! @brief        画面全体に描画します。
    //---------------------------------------------------------------------------
    void DrawScreen( ShaderType type, Texture* texture, bool mirror = false );
    void DrawScreen( ShaderType type, const nw::eft2::TextureResource* texture, bool mirror = false );
    void DrawScreen( ShaderType type, FrameBuffer* frameBuffer, bool mirror = false );

    //---------------------------------------------------------------------------
    //! @brief        小窓を描画します。
    //---------------------------------------------------------------------------
    void DrawWipe( FrameBuffer* frameBuffer, float left, float bottom, float right, float top );

    //---------------------------------------------------------------------------
    //! @brief        キューブマップテクスチャを描画します。
    //---------------------------------------------------------------------------
#if defined(NW_PLATFORM_WIN32)
    void DrawCubeMapTexture( GLuint* texture, f32 left, f32 bottom, f32 right, f32 top );
#endif
#if defined(NW_PLATFORM_CAFE)
    void DrawCubeMapTexture( GX2Texture* texture, f32 left, f32 bottom, f32 right, f32 top );
#endif

    //---------------------------------------------------------------------------
    //! @brief        ガウスフィルタの重みを設定します。
    //---------------------------------------------------------------------------
    void SetWeight( const f32* weight );

private:
    //---------------------------------------------------------------------------
    //! @brief        正射影をマトリクスを生成します。
    //---------------------------------------------------------------------------
    void _setupOrthoProjectionMatrix();

    //---------------------------------------------------------------------------
    //! @brief        テクスチャをセットアップします。
    //---------------------------------------------------------------------------
#if defined(NW_PLATFORM_WIN32)
    void _setupTexture( ShaderType type, GLuint texture );
#endif
#if defined(NW_PLATFORM_CAFE)
    void _setupTexture( ShaderType type, const GX2Texture* texture );
#endif

private:
    struct Vertex
    {
        nw::math::VEC3 pos;     //!< 位置です。
        nw::math::VEC2 uv;      //!< UV 座標です。
    };

    u32                             mPositionAttr[SHADER_TYPE_MAX];     //!<
    u32                             mTexCoordAttr[SHADER_TYPE_MAX];     //!<
    u32                             mWVPOffset[SHADER_TYPE_MAX];        //!<
    u32                             mTextureLocation[SHADER_TYPE_MAX];  //!<
    u32                             mTextureSize[SHADER_TYPE_MAX];      //!<
    u32                             mWeight[SHADER_TYPE_MAX];           //!< ガウスフィルタの重み
    nw::math::Matrix44              mWvpMatrix;                         //!<
    nw::math::Matrix44              mWvpMatrixMirror;                   //!< 上下反転させたマトリクス
    nw::math::VEC3*                 mQuadPositionBuf;                   //!<
    nw::math::VEC2*                 mQuadTexCrdBuf;                     //!<
    nw::ut::IAllocator*             mAllocator;                         //!<
    f32                             mWeightValue[GAUSS_LENGTH * 4];     //!< ガウスフィルタの重み

    nw::eft2::ResTextureSampler     m_ResTextureSampler;
    nw::eft2::TextureSampler        mTextureSampler;                    //!<
    nw::eft2::VertexBuffer          mPositionVB;                        //!< 位置座標VBO
    nw::eft2::VertexBuffer          mTexCoordVB;                        //!< テクスチャ座標VBO
    nw::eft2::ShaderResourceBase    mShader[SHADER_TYPE_MAX];           //!<
    nw::eftut2::NwHeap              mEffectHeap;                        //!<
};


} // namespace eftdemo
} // namespace nw
