﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Runtime.InteropServices;
using System.ComponentModel;
using System.Diagnostics;
using System.IO;

namespace Nintendo.Win32api
{
    /// <summary>
    /// Win32の関数と構造体、定数値の定義
    /// ・現時点では必要なものだけ定義してある。
    /// ・マーシャリングで不可解なところもあるので、その都度その箇所にコメントを入れている。
    /// ・関数の戻り値が BOOL の場合は、OSInfoのサンプル見る限り bool で記述してよさそうだ。
    /// </summary>
    public class Win32
    {

    // WinNT.h

        public const uint   GENERIC_READ                = 0x80000000;
        public const uint   GENERIC_WRITE               = 0x40000000;
        public const uint   GENERIC_EXECUTE             = 0x20000000;
        public const uint   GENERIC_ALL                 = 0x10000000;

    // WinDef.h

        [StructLayout(LayoutKind.Sequential)]
        public struct RECT
        {
            public int      left;
            public int      top;
            public int      right;
            public int      bottom;
        }

    // ntstatus.h

        public const uint   STATUS_BUFFER_OVERFLOW      = 0x80000005;

    // Error.h

        public const int    ERROR_BROKEN_PIPE           = 109;
        public const int    ERROR_PIPE_NOT_CONNECTED    = 233;
        public const int    ERROR_MORE_DATA             = 234;
        public const int    ERROR_PIPE_CONNECTED        = 535;
        public const int    ERROR_OPERATION_ABORTED     = 995;
        public const int    ERROR_IO_PENDING            = 997;

// kernel
    // WinBase.h

        // 定数定義

        public static readonly IntPtr INVALID_HANDLE_VALUE	= new IntPtr(-1);

        public const uint   FILE_FLAG_OVERLAPPED		= 0x40000000;

        public const uint   CREATE_NEW                  = 1;
        public const uint   CREATE_ALWAYS               = 2;
        public const uint   OPEN_EXISTING               = 3;
        public const uint   OPEN_ALWAYS                 = 4;
        public const uint   TRUNCATE_EXISTING           = 5;

        public const uint   PIPE_ACCESS_DUPLEX		    = 0x00000003;

        public const uint   PIPE_READMODE_MESSAGE       = 0x00000002;
        public const uint   PIPE_TYPE_MESSAGE           = 0x00000004;

        public const uint   PIPE_UNLIMITED_INSTANCES	= 255;

        // 構造体の定義

        /*
            null値を渡せるようにするため、classで定義

                pSecurityDescriptor にセットするポインタは
                    ConvertStringSecurityDescriptorToSecurityDescriptor関数で得られるようだ。
                    このポインタは不要になったらLocalFree()で解放する必要があるようだ。

                struct SECURITY_ATTRIBUTES
                {
                    DWORD	nLength;
                    LPVOID	lpSecurityDescriptor;
                    BOOL	bInheritHandle;
                };
             */
        [StructLayout(LayoutKind.Sequential)]
        public class SECURITY_ATTRIBUTES
        {
            public			SECURITY_ATTRIBUTES()
            {
                nLength = (uint)Marshal.SizeOf(this);
                lpSecurityDescriptor = IntPtr.Zero;
                bInheritHandle = 0/*false*/;
            }

            private uint	nLength;
            public IntPtr   lpSecurityDescriptor;
            public int      bInheritHandle;
        }

        // デリゲート定義

        // VOID CALLBACK FileIOCompletionRoutine(DWORD dwErrorCode, DWORD dwNumberOfBytesTransfered, LPOVERLAPPED lpOverlapped);
        public delegate void FileIOCompletionRoutine(uint dwErrorCode, uint dwNumberOfBytesTransfered, IntPtr lpOverlapped);

        // 関数定義

        /*
            HANDLE WINAPI CreateFileW(IN LPCWSTR lpFileName, IN DWORD dwDesiredAccess, IN DWORD dwShareMode, IN LPSECURITY_ATTRIBUTES lpSecurityAttributes,
                IN DWORD dwCreationDisposition, IN DWORD dwFlagsAndAttributes, IN HANDLE hTemplateFile);
         */
        [DllImport("Kernel32.dll", SetLastError = true, CharSet = CharSet.Unicode, ExactSpelling = true, EntryPoint = "CreateFileW")]
        public static extern IntPtr CreateFile(string fileName, uint desiredAccess, uint shareMode, SECURITY_ATTRIBUTES securityAttributes,
            uint creationDisposition, uint flagsAndAttributes, IntPtr hTemplateFile);

        /*
            BOOL ReadFile(IN HANDLE hFile, OUT LPVOID lpBuffer, IN DWORD nNumberOfBytesToRead,
                OUT LPDWORD lpNumberOfBytesRead, IN LPOVERLAPPED lpOverlapped);
        */
        /*
            non Overlapped I/O 版

            注意
                lpOverlapped は IntPtr.Zero であること
        */
        [DllImport("Kernel32.dll", SetLastError = true)]
        public static extern bool ReadFile(IntPtr hFile, [Out] byte[] buffer, uint numberOfBytesToReading, out uint numberOfBytesRead, IntPtr lpOverlapped);

        /*
            Overlapped I/O 版

            注意
                lpBuffer は固定されたアドレス
                lpNumberOfBytesRead は、IntPtr.Zeroか固定されたアドレス
                lpOverlapped は固定されたアドレス
                であること
         */
        [DllImport("Kernel32.dll", SetLastError = true)]
        public static extern bool ReadFile(IntPtr hFile, IntPtr lpBuffer, uint numberOfBytesToReading, IntPtr lpNumberOfBytesRead, IntPtr lpOverlapped);

        /*
            BOOL WriteFile(IN HANDLE hFile, IN LPCVOID lpBuffer, IN DWORD nNumberOfBytesToWrite,
                OUT LPDWORD lpNumberOfBytesWritten, IN LPOVERLAPPED lpOverlapped);
        */

        /*
            non Overlapped I/O 版

            注意
                lpOverlapped は IntPtr.Zero であること
        */
        [DllImport("Kernel32.dll", SetLastError = true)]
        public static extern bool WriteFile(IntPtr hFile, byte[] buffer, uint numberOfBytesToWrite,
            out uint numberOfBytesWritten, IntPtr lpOverlapped);

        /*
            Overlapped I/O 版

            注意
                lpBuffer は固定されたアドレス
                lpNumberOfBytesWritten は、IntPtr.Zeroか固定されたアドレス
                lpOverlapped は固定されたアドレス
                であること
        */
        [DllImport("Kernel32.dll", SetLastError = true)]
        public static extern bool WriteFile(IntPtr hFile, IntPtr lpBuffer, uint numberOfBytesToWrite, IntPtr lpNumberOfBytesWritten, IntPtr lpOverlapped);


        /*
            HANDLE WINAPI CreateNamedPipeW(IN LPCWSTR lpName, IN DWORD dwOpenMode, IN DWORD dwPipeMode, IN DWORD nMaxInstances,
                IN DWORD nOutBufferSize, IN DWORD nInBufferSize, IN DWORD nDefaultTimeOut, IN LPSECURITY_ATTRIBUTES lpSecurityAttributes);
         */
        [DllImport("Kernel32.dll", SetLastError = true, CharSet=CharSet.Unicode, ExactSpelling=true, EntryPoint="CreateNamedPipeW")]
        public static extern IntPtr CreateNamedPipe(string name, uint openMode, uint pipeMode, uint maxInstances,
            uint outBufferSize, uint inBufferSize, uint defaultTimeOut, SECURITY_ATTRIBUTES securityAttributes);

        //  BOOL ConnectNamedPipe(HANDLE hNamedPipe, LPOVERLAPPED lpOverlapped);
        [DllImport("Kernel32.dll", SetLastError = true)]
        public static extern bool ConnectNamedPipe(IntPtr hNamedPipe, IntPtr lpOverlapped);

        // BOOL DisconnectNamedPipe(IN HANDLE hNamedPipe);
        [DllImport("Kernel32.dll", SetLastError = true)]
        public static extern bool DisconnectNamedPipe(IntPtr hNamedPipe);

        // BOOL SetNamedPipeHandleState(IN HANDLE hNamedPipe, IN LPDWORD lpMode, IN LPDWORD lpMaxCollectionCount, IN LPDWORD lpCollectDataTimeout);
        [DllImport("Kernel32.dll", SetLastError = true)]
        public static extern bool SetNamedPipeHandleState(IntPtr hNamedPipe, uint[] lpMode, uint[] lpMaxCollectionCount, uint[] lpCollectDataTimeout);

        /*
            注意
                データを読み取らないで、
                メッセージのうちの読み取られていないバイト数を取得する場合は、
                bufferにnullを渡すだけではなく、nBufferSizeを0にしないと、正しい値が得られない。

                これが、この関数の仕様なのか、マーシャリングで意図した挙動になっていないのかは不明。
         */
        /*
            BOOL PeekNamedPipe(HANDLE hNamedPipe, LPVOID lpBuffer, DWORD nBufferSize, LPDWORD lpBytesRead,
                    LPDWORD lpTotalBytesAvail, LPDWORD lpBytesLeftThisMessage)
        */
        [DllImport("Kernel32.dll", SetLastError = true)]
        public static extern bool PeekNamedPipe(IntPtr hNamedPipe, [Out] byte[] buffer, uint nBufferSize, out uint bytesRead, out uint totalBytesAvail, out uint bytesLeftThisMessage);

        // BOOL GetOverlappedResult(HANDLE hFile, LPOVERLAPPED lpOverlapped, LPDWORD lpNumberOfBytesTransferred, BOOL bWait);
        [DllImport("Kernel32.dll", SetLastError = true)]
        public static extern bool GetOverlappedResult(IntPtr hFile, IntPtr lpOverlapped, out uint numberOfBytesTransferred, bool bWait);

        /*
            Overlapped I/O 操作のキャンセル

                キャンセルされた場合でも、あるいは既に操作が完了していても CancelIoは TRUE を返すようだ。
        */
        // BOOL CancelIo(HANDLE hFile);
        [DllImport("Kernel32.dll", SetLastError = true)]
        public static extern bool CancelIo(IntPtr hFile);

        // BOOL CloseHandle(HANDLE hObject);
        [DllImport("Kernel32.dll", SetLastError = true)]
        public static extern bool CloseHandle(HandleRef hObject);

        [DllImport("Kernel32.dll", SetLastError = true)]
        public static extern bool CloseHandle(IntPtr hObject);

        // BOOL BindIoCompletionCallback (HANDLE FileHandle, LPOVERLAPPED_COMPLETION_ROUTINE Function, ULONG Flags);
        [DllImport("Kernel32.dll", SetLastError = true)]
        public static extern bool BindIoCompletionCallback(IntPtr hFile, FileIOCompletionRoutine callback, uint flags);

    // WinCon.h

        // WINBASEAPI BOOL WINAPI AllocConsole( VOID );
        [DllImport("Kernel32.dll", SetLastError = true)]
        public static extern bool AllocConsole();

// gdi

        [StructLayout(LayoutKind.Sequential, Pack = 2)]
        public struct BITMAPFILEHEADER
        {
            public ushort       bfType;
            public uint         bfSize;
            public ushort       bfReserved1;
            public ushort       bfReserved2;
            public uint         bfOffBits;

            public void FromStream(Stream stream)
            {
                BinaryReader br = new BinaryReader(stream);

                this.bfType         = br.ReadUInt16();
                this.bfSize         = br.ReadUInt32();
                this.bfReserved1    = br.ReadUInt16();
                this.bfReserved2    = br.ReadUInt16();
                this.bfOffBits      = br.ReadUInt32();
            }
        }

        [StructLayout(LayoutKind.Sequential, Pack = 2)]
        public class CBITMAPFILEHEADER
        {
            public ushort       bfType;
            public uint         bfSize;
            public ushort       bfReserved1;
            public ushort       bfReserved2;
            public uint         bfOffBits;

            public static CBITMAPFILEHEADER FromStream(Stream stream)
            {
                BinaryReader br = new BinaryReader(stream);

                CBITMAPFILEHEADER head = new CBITMAPFILEHEADER();

                head.bfType         = br.ReadUInt16();
                head.bfSize         = br.ReadUInt32();
                head.bfReserved1    = br.ReadUInt16();
                head.bfReserved2    = br.ReadUInt16();
                head.bfOffBits      = br.ReadUInt32();

                return head;
            }
        }

        [StructLayout(LayoutKind.Sequential)]
        public struct BITMAPINFOHEADER
        {
            public uint         biSize;
            public int          biWidth;
            public int          biHeight;
            public ushort       biPlanes;
            public ushort       biBitCount;
            public uint         biCompression;
            public uint         biSizeImage;
            public int          biXPelsPerMeter;
            public int          biYPelsPerMeter;
            public uint         biClrUsed;
            public uint         biClrImportant;

            public void FromStream(Stream stream)
            {
                BinaryReader br = new BinaryReader(stream);

                this.biSize             = br.ReadUInt32();
                this.biWidth            = br.ReadInt32();
                this.biHeight           = br.ReadInt32();
                this.biPlanes           = br.ReadUInt16();
                this.biBitCount         = br.ReadUInt16();
                this.biCompression      = br.ReadUInt32();
                this.biSizeImage        = br.ReadUInt32();
                this.biXPelsPerMeter    = br.ReadInt32();
                this.biYPelsPerMeter    = br.ReadInt32();
                this.biClrUsed          = br.ReadUInt32();
                this.biClrImportant     = br.ReadUInt32();
            }
        }

        [StructLayout(LayoutKind.Sequential)]
        public class CBITMAPINFOHEADER
        {
            public uint         biSize;
            public int          biWidth;
            public int          biHeight;
            public ushort       biPlanes;
            public ushort       biBitCount;
            public uint         biCompression;
            public uint         biSizeImage;
            public int          biXPelsPerMeter;
            public int          biYPelsPerMeter;
            public uint         biClrUsed;
            public uint         biClrImportant;

            public static CBITMAPINFOHEADER  FromStream(Stream stream)
            {
                BinaryReader br = new BinaryReader(stream);

                CBITMAPINFOHEADER head = new CBITMAPINFOHEADER();

                head.biSize             = br.ReadUInt32();
                head.biWidth            = br.ReadInt32();
                head.biHeight           = br.ReadInt32();
                head.biPlanes           = br.ReadUInt16();
                head.biBitCount         = br.ReadUInt16();
                head.biCompression      = br.ReadUInt32();
                head.biSizeImage        = br.ReadUInt32();
                head.biXPelsPerMeter    = br.ReadInt32();
                head.biYPelsPerMeter    = br.ReadInt32();
                head.biClrUsed          = br.ReadUInt32();
                head.biClrImportant     = br.ReadUInt32();

                return head;
            }
        }

        [StructLayout(LayoutKind.Sequential)]
        public struct RGBQUAD
        {
            public byte         rgbBlue;
            public byte         rgbGreen;
            public byte         rgbRed;
            public byte         rgbReserved;

            public static RGBQUAD  FromStream(Stream stream)
            {
                BinaryReader br = new BinaryReader(stream);

                RGBQUAD rgb = new RGBQUAD();

                rgb.rgbBlue             = br.ReadByte();
                rgb.rgbGreen            = br.ReadByte();
                rgb.rgbRed              = br.ReadByte();
                rgb.rgbReserved         = br.ReadByte();

                return rgb;
            }
        }

        /*
            このクラスは物理的なイメージを定義するものではありません。
         */
        public struct BITMAPINFO
        {
            public BITMAPINFOHEADER bmiHeader;
            public RGBQUAD[]        bmiColors;
        }

        public class CBITMAPINFO
        {
            public BITMAPINFOHEADER bmiHeader;
            public RGBQUAD[]        bmiColors;

            static public CBITMAPINFO Copy( CBITMAPINFO src )
            {
                if( src == null )
                {
                    return null;
                }

                CBITMAPINFO	copy  = new CBITMAPINFO();
                copy.bmiHeader = src.bmiHeader;

                if( src.bmiColors != null )
                {
                    copy.bmiColors = new RGBQUAD[src.bmiColors.Length];
                    src.bmiColors.CopyTo( copy.bmiColors, 0 );
                }

                return copy;
            }
        }

    // user

        /*
         * Window Styles
         */
        public const int    WS_OVERLAPPED       = 0x00000000;
        public const int    WS_POPUP            = unchecked( (int)0x80000000 );
        public const int    WS_CHILD            = 0x40000000;
        public const int    WS_MINIMIZE         = 0x20000000;
        public const int    WS_VISIBLE          = 0x10000000;
        public const int    WS_DISABLED         = 0x08000000;
        public const int    WS_CLIPSIBLINGS     = 0x04000000;
        public const int    WS_CLIPCHILDREN     = 0x02000000;
        public const int    WS_MAXIMIZE         = 0x01000000;
        public const int    WS_CAPTION          = 0x00C00000;    /* WS_BORDER | WS_DLGFRAME  */
        public const int    WS_BORDER           = 0x00800000;
        public const int    WS_DLGFRAME         = 0x00400000;
        public const int    WS_VSCROLL          = 0x00200000;
        public const int    WS_HSCROLL          = 0x00100000;
        public const int    WS_SYSMENU          = 0x00080000;
        public const int    WS_THICKFRAME       = 0x00040000;
        public const int    WS_GROUP            = 0x00020000;
        public const int    WS_TABSTOP          = 0x00010000;

        public const int    WS_MINIMIZEBOX      = 0x00020000;
        public const int    WS_MAXIMIZEBOX      = 0x00010000;


        public const int    WS_TILED            = WS_OVERLAPPED;
        public const int    WS_ICONIC           = WS_MINIMIZE;
        public const int    WS_SIZEBOX          = WS_THICKFRAME;
        public const int    WS_TILEDWINDOW      = WS_OVERLAPPEDWINDOW;

        /*
         * Common Window Styles
         */
        public const int    WS_OVERLAPPEDWINDOW =  (WS_OVERLAPPED     |
                                                    WS_CAPTION        |
                                                    WS_SYSMENU        |
                                                    WS_THICKFRAME     |
                                                    WS_MINIMIZEBOX    |
                                                    WS_MAXIMIZEBOX);

        public const int    WS_POPUPWINDOW      =  (WS_POPUP          |
                                                    WS_BORDER         |
                                                    WS_SYSMENU);

        public const int    WS_CHILDWINDOW      = WS_CHILD;

        /*
         * Extended Window Styles
         */
        public const int    WS_EX_DLGMODALFRAME     = 0x00000001;
        public const int    WS_EX_NOPARENTNOTIFY    = 0x00000004;
        public const int    WS_EX_TOPMOST           = 0x00000008;
        public const int    WS_EX_ACCEPTFILES       = 0x00000010;
        public const int    WS_EX_TRANSPARENT       = 0x00000020;
        public const int    WS_EX_MDICHILD          = 0x00000040;
        public const int    WS_EX_TOOLWINDOW        = 0x00000080;
        public const int    WS_EX_WINDOWEDGE        = 0x00000100;
        public const int    WS_EX_CLIENTEDGE        = 0x00000200;
        public const int    WS_EX_CONTEXTHELP       = 0x00000400;

        public const int    WS_EX_RIGHT             = 0x00001000;
        public const int    WS_EX_LEFT              = 0x00000000;
        public const int    WS_EX_RTLREADING        = 0x00002000;
        public const int    WS_EX_LTRREADING        = 0x00000000;
        public const int    WS_EX_LEFTSCROLLBAR     = 0x00004000;
        public const int    WS_EX_RIGHTSCROLLBAR    = 0x00000000;

        public const int    WS_EX_CONTROLPARENT     = 0x00010000;
        public const int    WS_EX_STATICEDGE        = 0x00020000;
        public const int    WS_EX_APPWINDOW         = 0x00040000;

        public const int    WS_EX_OVERLAPPEDWINDOW  = (WS_EX_WINDOWEDGE | WS_EX_CLIENTEDGE);
        public const int    WS_EX_PALETTEWINDOW     = (WS_EX_WINDOWEDGE | WS_EX_TOOLWINDOW | WS_EX_TOPMOST);

        public const int    WS_EX_LAYERED           = 0x00080000;

        public const int    WS_EX_NOINHERITLAYOUT   = 0x00100000;   // Disable inheritence of mirroring by children
        public const int    WS_EX_LAYOUTRTL         = 0x00400000;   // Right to left mirroring

        public const int    WS_EX_COMPOSITED        = 0x02000000;
        public const int    WS_EX_NOACTIVATE        = 0x08000000;


        /*
         * Class styles
         */
        public const int    CS_VREDRAW          = 0x0001;
        public const int    CS_HREDRAW          = 0x0002;
        public const int    CS_DBLCLKS          = 0x0008;
        public const int    CS_OWNDC            = 0x0020;
        public const int    CS_CLASSDC          = 0x0040;
        public const int    CS_PARENTDC         = 0x0080;
        public const int    CS_NOCLOSE          = 0x0200;
        public const int    CS_SAVEBITS         = 0x0800;
        public const int    CS_BYTEALIGNCLIENT  = 0x1000;
        public const int    CS_BYTEALIGNWINDOW  = 0x2000;
        public const int    CS_GLOBALCLASS      = 0x4000;

        public const int    CS_IME              = 0x00010000;
        public const int    CS_DROPSHADOW       = 0x00020000;

        public const int    VK_CLEAR            = 0x0C;
        public const int    VK_RETURN           = 0x0D;

        public const int    VK_SHIFT            = 0x10;
        public const int    VK_CONTROL          = 0x11;
        public const int    VK_MENU             = 0x12;
        public const int    VK_PAUSE            = 0x13;
        public const int    VK_CAPITAL          = 0x14;

        public const int    VK_KANA             = 0x15;
        public const int    VK_HANGUL           = 0x15;
        public const int    VK_JUNJA            = 0x17;
        public const int    VK_FINAL            = 0x18;
        public const int    VK_HANJA            = 0x19;
        public const int    VK_KANJI            = 0x19;

        public const int    VK_ESCAPE           = 0x1B;

        public const int    VK_CONVERT          = 0x1C;
        public const int    VK_NONCONVERT       = 0x1D;
        public const int    VK_ACCEPT           = 0x1E;
        public const int    VK_MODECHANGE       = 0x1F;

        public const int    VK_SPACE            = 0x20;
        public const int    VK_PRIOR            = 0x21;
        public const int    VK_NEXT             = 0x22;
        public const int    VK_END              = 0x23;
        public const int    VK_HOME             = 0x24;
        public const int    VK_LEFT             = 0x25;
        public const int    VK_UP               = 0x26;
        public const int    VK_RIGHT            = 0x27;
        public const int    VK_DOWN             = 0x28;
        public const int    VK_SELECT           = 0x29;
        public const int    VK_PRINT            = 0x2A;
        public const int    VK_EXECUTE          = 0x2B;
        public const int    VK_SNAPSHOT         = 0x2C;
        public const int    VK_INSERT           = 0x2D;
        public const int    VK_DELETE           = 0x2E;
        public const int    VK_HELP             = 0x2F;

        // WINUSERAPI HDC WINAPI GetDC(IN HWND hWnd);
        [DllImport("user32.dll", SetLastError = true)]
        public static extern IntPtr GetDC(IntPtr hWnd);

        // WINUSERAPI int WINAPI ReleaseDC(IN HWND hWnd, IN HDC hDC);
        [DllImport("user32.dll")]
        public static extern int ReleaseDC(IntPtr hWnd, IntPtr hDC);

        // WINUSERAPI HWND WINAPI GetDesktopWindow(VOID);
        [DllImport("user32.dll")]
        public static extern IntPtr GetDesktopWindow();

        // WINUSERAPI BOOL WINAPI GetWindowRect(IN HWND hWnd, OUT LPRECT lpRect);
        [DllImport("user32.dll", SetLastError = true)]
        public static extern bool GetWindowRect(IntPtr hWnd, out RECT rect);

        // WINUSERAPI SHORT WINAPI GetKeyState(IN int nVirtKey);
        [DllImport("user32.dll", SetLastError = true)]
        public static extern short GetKeyState(int nVirtKey);

    // OpenGL関連

        /* pixel types */
        public const byte   PFD_TYPE_RGBA           = 0;
        public const byte   PFD_TYPE_COLORINDEX     = 1;

        /* layer types */
        public const byte   PFD_MAIN_PLANE          = 0;
        public const byte   PFD_OVERLAY_PLANE       = 1;
        public const byte   PFD_UNDERLAY_PLANE      = unchecked ( (byte)-1 );


        /* PIXELFORMATDESCRIPTOR flags */
        public const uint   PFD_DOUBLEBUFFER        = 0x00000001;
        public const uint   PFD_STEREO              = 0x00000002;
        public const uint   PFD_DRAW_TO_WINDOW      = 0x00000004;
        public const uint   PFD_DRAW_TO_BITMAP      = 0x00000008;
        public const uint   PFD_SUPPORT_GDI         = 0x00000010;
        public const uint   PFD_SUPPORT_OPENGL      = 0x00000020;
        public const uint   PFD_GENERIC_FORMAT      = 0x00000040;
        public const uint   PFD_NEED_PALETTE        = 0x00000080;
        public const uint   PFD_NEED_SYSTEM_PALETTE = 0x00000100;
        public const uint   PFD_SWAP_EXCHANGE       = 0x00000200;
        public const uint   PFD_SWAP_COPY           = 0x00000400;
        public const uint   PFD_SWAP_LAYER_BUFFERS  = 0x00000800;
        public const uint   PFD_GENERIC_ACCELERATED = 0x00001000;
        public const uint   PFD_SUPPORT_DIRECTDRAW  = 0x00002000;

        /* Pixel format descriptor */
        [StructLayout(LayoutKind.Sequential)]
        public class PIXELFORMATDESCRIPTOR
        {
            private ushort  nSize;
            public ushort  nVersion;
            public uint dwFlags;
            public byte  iPixelType;
            public byte  cColorBits;
            public byte  cRedBits;
            public byte  cRedShift;
            public byte  cGreenBits;
            public byte  cGreenShift;
            public byte  cBlueBits;
            public byte  cBlueShift;
            public byte  cAlphaBits;
            public byte  cAlphaShift;
            public byte  cAccumBits;
            public byte  cAccumRedBits;
            public byte  cAccumGreenBits;
            public byte  cAccumBlueBits;
            public byte  cAccumAlphaBits;
            public byte  cDepthBits;
            public byte  cStencilBits;
            public byte  cAuxBuffers;
            public byte  iLayerType;
            protected byte  bReserved = 0;
            public uint dwLayerMask;
            public uint dwVisibleMask;
            public uint dwDamageMask;

            public PIXELFORMATDESCRIPTOR(
                ushort  nVersion,
                uint dwFlags,
                byte  iPixelType,
                byte  cColorBits,
                byte  cRedBits,
                byte  cRedShift,
                byte  cGreenBits,
                byte  cGreenShift,
                byte  cBlueBits,
                byte  cBlueShift,
                byte  cAlphaBits,
                byte  cAlphaShift,
                byte  cAccumBits,
                byte  cAccumRedBits,
                byte  cAccumGreenBits,
                byte  cAccumBlueBits,
                byte  cAccumAlphaBits,
                byte  cDepthBits,
                byte  cStencilBits,
                byte  cAuxBuffers,
                byte  iLayerType,
                uint dwLayerMask,
                uint dwVisibleMask,
                uint dwDamageMask
            )
            {
                this.nSize              = (ushort)Marshal.SizeOf(this);
                this.nVersion           = nVersion;
                this.dwFlags            = dwFlags;
                this.iPixelType         = iPixelType;
                this.cColorBits         = cColorBits;
                this.cRedBits           = cRedBits;
                this.cRedShift          = cRedShift;
                this.cGreenBits         = cGreenBits;
                this.cGreenShift        = cGreenShift;
                this.cBlueBits          = cBlueBits;
                this.cBlueShift         = cBlueShift;
                this.cAlphaBits         = cAlphaBits;
                this.cAlphaShift        = cAlphaShift;
                this.cAccumBits         = cAccumBits;
                this.cAccumRedBits      = cAccumRedBits;
                this.cAccumGreenBits    = cAccumGreenBits;
                this.cAccumBlueBits     = cAccumBlueBits;
                this.cAccumAlphaBits    = cAccumAlphaBits;
                this.cDepthBits         = cDepthBits;
                this.cStencilBits       = cStencilBits;
                this.cAuxBuffers        = cAuxBuffers;
                this.iLayerType         = iLayerType;
                this.dwLayerMask        = dwLayerMask;
                this.dwVisibleMask      = dwVisibleMask;
                this.dwDamageMask       = dwDamageMask;
            }

        };

        // WINGDIAPI int   WINAPI ChoosePixelFormat( IN HDC, IN CONST PIXELFORMATDESCRIPTOR *);
        [DllImport("GDI32.dll", SetLastError = true)]
        public static extern int ChoosePixelFormat(IntPtr hDC, PIXELFORMATDESCRIPTOR pixelFormatDescriptor);

        // WINGDIAPI BOOL  WINAPI SetPixelFormat(IN HDC, IN int, IN CONST PIXELFORMATDESCRIPTOR *);
        [DllImport("GDI32.dll", SetLastError = true)]
        public static extern bool SetPixelFormat(IntPtr hDC, int iPixelFormat, PIXELFORMATDESCRIPTOR pixelFormatDescriptor);

        // WINGDIAPI BOOL  WINAPI SwapBuffers(HDC);
        [DllImport("gdi32.dll", SetLastError = true)]
        public static extern bool SwapBuffers(IntPtr hDC);

    }

    public class Win32X
    {
        public static void CloseHandle(HandleRef hObject)
        {
            if (! Win32.CloseHandle(hObject))
            {
                reportError("CloseHandle");
            }
        }

        public static void CloseHandle(IntPtr hObject)
        {
            if (! Win32.CloseHandle(hObject))
            {
                reportError("CloseHandle");
            }
        }

        public static void CancelIo(IntPtr hFile)
        {
            if (! Win32.CancelIo(hFile))
            {
                reportError("CancelIo");
            }
        }

        public static void ReleaseDC(IntPtr hWnd, IntPtr hDC)
        {
            if (0 == Win32.ReleaseDC(hWnd, hDC))
            {
                Debug.WriteLine("Release DC fail. - window = " + hWnd + ", DC = " + hDC);
            }
        }

        private static void reportError(string funcName)
        {
            Debug.WriteLine(funcName + "() error. - " + Marshal.GetLastWin32Error());
        }
    }
}

