﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.IO;
using System.Xml.Serialization;
using System.Diagnostics;
using System.Windows.Forms;

namespace Nintendo.McsServer
{
    /// <summary>
    /// UserSettings の概要の説明です。
    /// XMLシリアライズ機能を利用するため、難読化対象から除外されています。
    /// ( プロジェクトフォルダのdotfuscator.xmlで設定しています。)
    /// </summary>
    public class UserSettings
    {
        [Flags]
        public enum CapturePosition
        {
            None = 0x0,
            UpperLeft = 0x1,
            UpperRight = 0x2,
            DownLeft = 0x4,
            DownRight = 0x8,
            Top = 0x10,
            Bottom = 0x20,
            Left = 0x40,
            Right = 0x80,
            Default = 0xF
        }

        public const uint       DefaultPollingInterval  = 1;
        public const ushort     DefaultPortNo           = 5598;

        const string SettingsFileName       = "user.config";

        bool _appSettingsChanged = false;
        ushort _portNo = DefaultPortNo;
        uint _pollingInterval = DefaultPollingInterval;

        bool _bDisableDeviceUSBAdapter = false;
        bool _bDisableDeviceNDEV = false;
        bool _bInTaskTray = true;

        bool _bCaptureOnKeyShortcut = true;   // キーの2回押下げによるマウスキャプチャ起動有効
        Keys _captureShortcutKey = Keys.Scroll;
        int _captureOnKeyShortcutInterval = 400;

        bool _bCaptureOnMousePosition = false; // マウスカーソルの画面端移動によるマウスキャプチャ起動有効
        CapturePosition _captureOnMousePosition = CapturePosition.Default;
        int _captureOnMousePositionCount = 100;

        bool _bCheckAlive = true;   // ターゲットプログラムの生存確認を行うか。
        uint _checkAliveTimeout = 6; // 生存確認の応答待ち時間の上限[秒]。

        /// <summary>
        /// コンストラクタです。
        /// </summary>
        public UserSettings()
        {
            //
            // TODO: コンストラクタ ロジックをここに追加してください。
            //
        }

        /// <summary>
        /// 待ち受け用のポート番号設定です。
        /// </summary>
        public ushort PortNo
        {
            get
            {
                return _portNo;
            }
            set
            {
                if (_portNo != value)
                {
                    _portNo = value;
                    _appSettingsChanged = true;
                }
            }
        }

        /// <summary>
        /// ポーリング間隔です。
        /// </summary>
        public uint PollingInterval
        {
            get
            {
                return _pollingInterval;
            }
            set
            {
                if (_pollingInterval != value)
                {
                    _pollingInterval = value;
                    _appSettingsChanged = true;
                }
            }
        }

        public bool DisableDeviceUSBAdapter
        {
            get
            {
                return _bDisableDeviceUSBAdapter;
            }
            set
            {
                if (_bDisableDeviceUSBAdapter != value)
                {
                    _bDisableDeviceUSBAdapter = value;
                    _appSettingsChanged = true;
                }
            }
        }

        public bool DisableDeviceCATDEV
        {
            get
            {
                return _bDisableDeviceNDEV;
            }
            set
            {
                if (_bDisableDeviceNDEV != value)
                {
                    _bDisableDeviceNDEV = value;
                    _appSettingsChanged = true;
                }
            }
        }

        public bool IsInTaskTray
        {
            get
            {
                return _bInTaskTray;
            }
            set
            {
                if (_bInTaskTray != value)
                {
                    _bInTaskTray = value;
                    _appSettingsChanged = true;
                }
            }
        }

        /// <summary>
        /// キーの2回押下げによるマウスキャプチャ起動有効設定
        /// </summary>
        public bool IsCaptureOnKeyShortcut
        {
            get
            {
                return _bCaptureOnKeyShortcut;
            }
            set
            {
                if (_bCaptureOnKeyShortcut != value)
                {
                    _bCaptureOnKeyShortcut = value;
                    _appSettingsChanged = true;
                }
            }
        }

        /// <summary>
        /// マウスカーソルの画面端移動によるマウスキャプチャ起動有効設定
        /// </summary>
        public bool IsCaptureOnMousePosition
        {
            get
            {
                return _bCaptureOnMousePosition;
            }
            set
            {
                if (_bCaptureOnMousePosition != value)
                {
                    _bCaptureOnMousePosition = value;
                    _appSettingsChanged = true;
                }
            }
        }

        /// <summary>
        /// マウスキーボードキャプチャのショートカットキー設定
        /// </summary>
        public Keys CaptureShortcutKey
        {
            get
            {
                return _captureShortcutKey;
            }
            set
            {
                if (_captureShortcutKey != value)
                {
                    _captureShortcutKey = value;
                    _appSettingsChanged = true;
                }
            }
        }

        /// <summary>
        /// マウスキーボードキャプチャのマウスショートカットの場所
        /// </summary>
        public CapturePosition CaptureMousePosition
        {
            get
            {
                return _captureOnMousePosition;
            }
            set
            {
                if (_captureOnMousePosition != value)
                {
                    _captureOnMousePosition = value;
                    _appSettingsChanged = true;
                }
            }
        }

        /// <summary>
        /// マウスキーボードキャプチャのマウスショートカットの滞在時間
        /// </summary>
        public int CaptureOnMousePositionCount
        {
            get
            {
                return _captureOnMousePositionCount;
            }
            set
            {
                if (_captureOnMousePositionCount != value)
                {
                    _captureOnMousePositionCount = value;
                    _appSettingsChanged = true;
                }
            }
        }

        /// <summary>
        /// キーボードの二回押しの間隔
        /// </summary>
        public int CaptureOnKeyShortcutInterval
        {
            get
            {
                return _captureOnKeyShortcutInterval;
            }
            set
            {
                if (_captureOnKeyShortcutInterval != value)
                {
                    _captureOnKeyShortcutInterval = value;
                    _appSettingsChanged = true;
                }
            }
        }

        /// <summary>
        /// ターゲットプログラムの生存確認を行うか。
        /// </summary>
        public bool IsCheckAlive
        {
            get
            {
                return _bCheckAlive;
            }
            set
            {
                if (_bCheckAlive != value)
                {
                    _bCheckAlive = value;
                    _appSettingsChanged = true;
                }
            }
        }

        /// <summary>
        /// 生存確認の応答待ち時間の上限[秒]。
        /// </summary>
        public uint CheckAliveTimeout
        {
            get
            {
                return _checkAliveTimeout;
            }
            set
            {
                if (_checkAliveTimeout != value)
                {
                    _checkAliveTimeout = value;
                }
            }
        }

        /// <summary>
        /// 構成ファイルからクラスをシリアル化解除します。
        /// </summary>
        /// <returns></returns>
        public static UserSettings LoadAppSettings()
        {
            try
            {
                // 構成ファイルが存在する場合は、それを開きます。
                FileInfo fi = new FileInfo(GetAppDataPath() + @"\" + SettingsFileName);
                if (fi.Exists)
                {
                    using (FileStream fs = fi.OpenRead())
                    {
                        XmlSerializer xmlSerializer = new XmlSerializer(typeof(UserSettings));
                        return (UserSettings)xmlSerializer.Deserialize(fs);
                    }
                }
            }
            catch (IOException ex)
            {
                Debug.WriteLine(ex.Message);
            }
            catch (InvalidOperationException ex)    // 構成ファイルの内容が正しくない場合
            {
                Debug.WriteLine(ex.Message);
            }
            return new UserSettings();
        }

        /// <summary>
        /// 変更されている設定がある場合、クラスを構成ファイルにシリアル化します。
        /// </summary>
        /// <returns></returns>
        public bool SaveAppSettings()
        {
            if (_appSettingsChanged)
            {
                XmlSerializer xmlSerializer = new XmlSerializer(typeof(UserSettings));

                try
                {
                    DirectoryInfo saveFileDir = new DirectoryInfo(GetAppDataPath());

                    if (! saveFileDir.Exists)
                    {
                        saveFileDir.Create();
                    }

                    using (FileStream writer = new FileStream(saveFileDir.FullName + @"\" + SettingsFileName, FileMode.Create, FileAccess.Write))
                    {
                        xmlSerializer.Serialize(writer, this);
                    }
                }
                catch (IOException ex)  // サンプルコードがtry/catchで囲んでいたため残しているが、MSDNを見る限り特に例外が記述されていない。
                {
                    Debug.WriteLine(ex.Message);
                }
            }

            return _appSettingsChanged;
        }

        static string GetAppDataPath()
        {
            return Environment.GetFolderPath(Environment.SpecialFolder.ApplicationData)     // ベース・パス
                    + @"\" + Application.CompanyName                                        // 会社名
                    + @"\" + Application.ProductName;                                       // 製品名
        }
    }
}
