﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Threading;
using System.Net;
using System.Net.Sockets;
using System.Diagnostics;
using System.IO;
using System.Runtime.InteropServices;
using System.Collections.Generic;

namespace Nintendo.McsServer
{
    /// <summary>
    /// チャンネルに送信されるコマンドの種類
    /// </summary>
    enum ServerCommand : uint
    {
        Unknown         = 0,
        ChannelRequest
    }

    /// <summary>
    /// ルータ切断コールバック
    /// </summary>
    public delegate void RouterDisconnectCallback(string devName);

    /// <summary>
    /// Router
    /// 実機との通信経路と、
    /// 複数のチャンネルごとの通信を束ねるクラス。
    ///
    /// 実機            => 各チャンネルへのデータの振り分け
    /// 各チャンネル    => 実機へのデータ送信
    ///
    /// の処理を行います。
    /// </summary>
    public class Router
    {
        /// <summary>
        /// _readBufの初期サイズ。
        /// _readBufは必要に応じて拡大されます。
        /// </summary>
        const int                   InitReadBufSize     = 0x10000;
        const string CommDeviceNameNDEV                 = "NDEV";
        const string CommDeviceNamePCViewer             = "PC Viewer";
        const string CommDeviceNamePCIO2                = "PC";
        const string CommDeviceNameCATDEV               = "CAT-DEV";

        /// <summary>
        /// アイドル時のスリープ時間(ms)。
        /// </summary>
        const int IdleSleepTime = 100;

        /// <summary>
        /// Start()関数が成功していたら true。
        /// </summary>
        bool _bActive;

        /// <summary>
        /// ルータ情報
        /// (各チャンネルに参照が渡され、各チャンネルがルータの情報を利用するときに使用されます)
        /// </summary>
        readonly RouterInfo _routerInfo;

        /// <summary>
        /// ソケット接続受付を監視するクラス。
        /// </summary>
        readonly SocketListener _socketListener;

        /// <summary>
        /// ワーカースレッド。
        /// </summary>
        Thread _thread;

        readonly EventWaitHandle _exitThreadEvent = new EventWaitHandle(false, EventResetMode.ManualReset);
        /// <summary>
        /// チャンネルマネージャ
        /// </summary>
        readonly ChannelManager _channelManager  = new ChannelManager();

        /// <summary>
        /// ICommDeviceへの書き込みデータキュー
        /// </summary>
        readonly DataPacketQueue _dataPacketQueue = new DataPacketQueue();

        /// <summary>
        /// Disconnect()になったら呼び出すコールバック
        /// </summary>
        readonly RouterDisconnectCallback _routerDisconnectCallback;

        /// <summary>
        /// FileIOを処理するサーバー
        /// </summary>
        readonly FileIOServer _fileIOServer;

        /// <summary>
        /// このクラスの公開メンバをスレッドセーフにするための同期用オブジェクト
        /// </summary>
        readonly object _syncObj = new object();

        /// <summary>
        /// ワーカースレッドと同期を取るためのオブジェクト。
        /// </summary>
        readonly object _syncThread = new object();

        /// <summary>
        /// 待ち受けポート番号。
        /// </summary>
        int _portNo;

        /// <summary>
        /// 読み込みバッファ
        /// _commDevからの受信データ(実機から書き込まれたデータ)
        /// を一旦読みこむ場所です。
        /// </summary>
        byte[]                      _readBuf;
        /// <summary>
        /// 実機側との通信路を表すクラス
        /// </summary>
        ICommDevice                 _commDev;

        int _vPollingInterval;

        /// <summary>
        /// 接続中である場合は、接続しているデバイスの名前。
        /// 接続していない場合はnull。
        /// </summary>
        string _devName;

        /// <summary>
        /// デバイス名の Setter / Getter です。
        /// </summary>
        private string DevName_
        {
            get
            {
                return _devName;
            }

            set
            {
                lock (_syncObj)
                {
                    bool before = _devName != null && _bActive;

                    _devName = value;

                    bool after = _devName != null && _bActive;

                    if (before != after)
                    {
                        this.OnConnect(after);
                    }
                }
            }
        }

        /// <summary>
        /// アクティブ状態の Setter / Getter です。
        /// </summary>
        private bool IsActive_
        {
            get
            {
                return _bActive;
            }

            set
            {
                lock (_syncObj)
                {
                    bool before = _devName != null && _bActive;

                    _bActive = value;

                    bool after = _devName != null && _bActive;

                    if (before != after)
                    {
                        this.OnConnect(after);
                    }
                }
            }
        }

        /// <summary>
        /// コンストラクタ
        /// </summary>
        public Router(RouterDisconnectCallback routerDisconnectCallback, System.Windows.Forms.Form form)
        {
            _routerInfo = new RouterInfo(_channelManager, _dataPacketQueue);
            _socketListener = new SocketListener(DoChanelRegist);
            _routerDisconnectCallback = routerDisconnectCallback;
            _fileIOServer = new FileIOServer(form);
        }

        /// <summary>
        /// 動作開始
        /// </summary>
        public void Start(int portNo)
        {
            lock (_syncObj)
            {
                if (this.IsActive_)       // 既にConnect()が成功していたら何もしない
                {
                    return;
                }

                _portNo = portNo;
                _socketListener.Start(_portNo);     // socketのaccept開始。
                _fileIOServer.Start(_portNo);
                PCIO2CommDevice.PCIO2CommDevice.Start(4000);

                CafeHioCommDevice.CommDevice.Start();

                _exitThreadEvent.Reset();
                _thread = new Thread(ThreadStart);
                _thread.Start();

                this.IsActive_ = true;
            }
        }

        /// <summary>
        /// 動作終了
        /// </summary>
        public void Stop()
        {
            lock (_syncObj)
            {
                Debug.Assert(this.DevName_ == null);

                if (! this.IsActive_)     // Start()していないか、既に停止されていたら何もしない
                {
                    return ;
                }

                PCIO2CommDevice.PCIO2CommDevice.Stop();
                _fileIOServer.Stop();
                _socketListener.Stop();

                _exitThreadEvent.Set();     // スレッドを終了するように指示
                _thread.Join();             // スレッドの終了を待つ
                _thread = null;

                this.IsActive_ = false;
            }
        }

        /// <summary>
        /// アクティブな状態か取得します。
        /// </summary>
        public bool Active
        {
            get
            {
                lock (_syncObj)
                {
                    return this.IsActive_;
                }
            }
        }

        public delegate void ConnectedEvent(bool connect);

        /// <summary>
        /// 接続、切断時のイベントです。
        /// </summary>
        public ConnectedEvent OnConnect;

        /// <summary>
        /// ターゲットとの接続状態の変化を通知するイベント引数です。
        /// </summary>
        public class TargetConnectionChangedEventArgs : EventArgs
        {
            /// <summary>
            /// ターゲットとの接続状態です。
            /// </summary>
            public bool IsConnected { get; private set; }

            /// <summary>
            /// 生存確認によってターゲットとの接続が切断された場合は true となります。
            /// </summary>
            public bool IsDisconnectedByCheckAlive { get; private set; }

            public TargetConnectionChangedEventArgs(bool connect, bool disconnectByCheckAlive)
            {
                this.IsConnected = connect;
                this.IsDisconnectedByCheckAlive = disconnectByCheckAlive;
            }
        }

        /// <summary>
        /// ターゲットプログラムとの接続状態の変化を通知する delegate です。
        /// </summary>
        /// <param name="connect">ターゲットプログラムとの接続状態です。</param>
        public delegate void TargetConnectionChangedEventHandler(TargetConnectionChangedEventArgs args);

        /// <summary>
        /// ターゲットプログラムとの接続、切断時のイベントです。
        /// </summary>
        public TargetConnectionChangedEventHandler TargetConnectionChanged;

        public delegate int MultiDeviceDelegate(object[] devices);

        /// <summary>
        /// Router接続パラメータ格納クラス
        /// </summary>
        public class ConnectParam
        {
            public uint PollingInterval = UserSettings.DefaultPollingInterval;
            public bool DisableDeviceUSBAdapter;
            public bool DisableDeviceNDEV;
            public MultiDeviceDelegate MultiDeviceDelegate;
            public string explicitConnectDevice;
            public uint CheckAliveTimeout;
        }

        /// <summary>
        /// 実機側との接続を確立します。
        /// </summary>
        public string Connect(ConnectParam connectParam)
        {
            lock (_syncObj)
            {
                Debug.Assert(this.IsActive_);

                if (this.DevName_ != null)    // 既にConnect()が成功していたら何もしない
                {
                    return this.DevName_;
                }

                try
                {
                    ICommDevice commDev = null;

                    if (connectParam.explicitConnectDevice != null) // 明示的に接続デバイスが指定されているとき
                    {
                        switch (connectParam.explicitConnectDevice)
                        {
// 実機側との通信部分を除去
#if false
                        case CommDeviceNameNDEV:
                            {
                                // HIO2デバイスの検索
                                string devicePathName = string.Empty;
                                if (HIO2CommDevice.SearchDevice(out devicePathName))
                                {
                                    commDev = new HIO2CommDevice(devicePathName);
                                }
                                else
                                {
                                    throw new RouterException(Properties.Resources.ErrorDeviceNotFoundNDEV);
                                }
                            }
                            break;
#endif

// PCIOCommDeviceを除去
#if false
                        case CommDeviceNamePCViewer:
                            if (PCIOCommDevice.SearchDevice())
                            {
                                commDev = new PCIOCommDevice();
                            }
                            else
                            {
                                throw new RouterException(Properties.Resources.ErrorDeviceNotFoundPCViewer);
                            }
                            break;
#endif

                        case CommDeviceNamePCIO2:
                            commDev = PCIO2CommDevice.PCIO2CommDevice.GetDevice();
                            break;

                        case CommDeviceNameCATDEV:
                            commDev = CafeHioCommDevice.CommDevice.GetDevice();
                            break;

                        default:
                            throw new RouterException(string.Format(Properties.Resources.ErrorUnknownDevice, connectParam.explicitConnectDevice));
                        }
                    }
                    else   // 明示的に接続デバイスを指定していないとき
                    {
                        var deviceInfos = new List<object>();

// 実機側との通信部分を除去
#if false
                        // HIO2デバイスの検索
                        HIO2DllNotFoundException hio2DllNotFoundException = null;
                        bool bFindHIO2Device = false;
                        string devicePathName = string.Empty;
                        if (! connectParam.DisableDeviceNDEV)
                        {
                            try
                            {
                                bFindHIO2Device = HIO2CommDevice.SearchDevice(out devicePathName);
                            }
                            catch (HIO2DllNotFoundException ex)
                            {
                                // 例外を一旦補足しておく
                                hio2DllNotFoundException = ex;
                            }
                        }

                        if (bFindHIO2Device)
                        {
                            deviceInfos.Add(CommDeviceNameNDEV);
                        }
#endif

// PCIOCommDeviceを除去
#if false
                        // PC Viewerの検索
                        bool bFindPCViewer = !connectParam.DisableDeviceUSBAdapter && PCIOCommDevice.SearchDevice();

                        if (bFindPCViewer)
                        {
                            deviceInfos.Add(CommDeviceNamePCViewer);
                        }
#endif

                        // PC版実機アプリとの通信デバイスの検索。
                        // TODO: PC版実機アプリの無効化
                        PCIO2CommDevice.PCIO2CommDevice.EnumDevices(
                            (object id) =>
                            {
                                deviceInfos.Add(id);
                                return false;
                            });

                        // 実機アプリとの通信デバイスの検索。
                        // TODO: 実機アプリの無効化
                        CafeHioCommDevice.CommDevice.EnumDevices(
                            (object id) =>
                            {
                                deviceInfos.Add(id);
                                return false;
                            });

                        if (deviceInfos.Count > 0)
                        {
                            object deviceInfo = null;

                            if (deviceInfos.Count == 1)
                            {
                                deviceInfo = deviceInfos[0];
                            }
                            else
                            {
                                // 複数のデバイスが見つかったときは指定されたデリゲートを呼び出す
                                int devIdx = connectParam.MultiDeviceDelegate(deviceInfos.ToArray());
                                if (devIdx == -1)
                                {
                                    return null;    // ユーザが接続をキャンセルした
                                }

                                deviceInfo = deviceInfos[devIdx];
                            }

// 実機側との通信部分を除去
#if false
                            if (commDev == null && CommDeviceNameNDEV.Equals(deviceInfo))
                            {
                                commDev = new HIO2CommDevice(devicePathName);
                            }
#endif

// PCIOCommDeviceを除去
#if false
                            if (commDev == null && CommDeviceNamePCViewer.Equals(deviceInfo))
                            {
                                commDev = new PCIOCommDevice();
                            }
#endif

                            if (commDev == null)
                            {
                                commDev = PCIO2CommDevice.PCIO2CommDevice.GetDevice(deviceInfo);
                            }

                            if (commDev == null)
                            {
                                commDev = CafeHioCommDevice.CommDevice.GetDevice(deviceInfo);
                            }
                        }

                        if (commDev == null)
                        {
// 実機側との通信部分を除去
#if false
                            if (hio2DllNotFoundException != null)
                            {
                                // 補足していた例外を再throw
                                throw hio2DllNotFoundException;
                            }
#endif

                            throw new RouterException(Properties.Resources.ErrorDeviceNotFound);
                        }

                        if (commDev is ICheckAlive)
                        {
                            (commDev as ICheckAlive).CheckAliveTimeout = connectParam.CheckAliveTimeout;
                        }
                    }

                    lock (_syncThread)
                    {
                        _commDev = commDev;
                        this.DevName_ = _commDev.Name;
                        _readBuf = new byte[InitReadBufSize];

                        _vPollingInterval = (int)connectParam.PollingInterval;

                        return this.DevName_;
                    }
                }
// 実機側との通信部分を除去
#if false
                catch (HIO2DllNotFoundException)
                {
                    throw new RouterException(Properties.Resources.ErrorHIO2DllNotFound);
                }
#endif
                catch (CommDeviceException e)
                {
                    throw new RouterException(e.Message, e);
                }
                catch (SocketException ex)
                {
                    throw new RouterException("SocketException", ex);
                }
            }
        }

        /// <summary>
        /// 実機側との接続を切断します。
        /// </summary>
        public void Disconnect()
        {
            string devName = string.Empty;

            lock (_syncObj)
            {
                if (this.DevName_ == null)   // Connect()していないか、既に切断されていたら何もしない
                {
                    return ;
                }

                ICommDevice commDev = _commDev;

                lock (_syncThread)
                {
                    _commDev = null;
                }

                RouterLog.ServerReport("[{0}] is disconnected.", commDev.Name);

                _channelManager.SetTargetConnect(false);

                commDev.Dispose();

                devName = this.DevName_;
                this.DevName_ = null;
            }

            _routerDisconnectCallback(devName);
        }

        /// <summary>
        /// 実機側と接続されているか取得します。
        /// </summary>
        public bool IsConnect
        {
            get
            {
                lock (_syncObj)
                {
                    return this.DevName_ != null;
                }
            }
        }

        /// <summary>
        /// 実機のターゲットプログラムとの通信が確立しているか取得します。
        /// </summary>
        public bool IsTargetConnect
        {
            get
            {
                lock (_syncObj)
                {
                    if (_commDev != null)
                    {
                        return _commDev.IsTargetConnect;
                    }
                    else
                    {
                        return false;
                    }
                }
            }
        }

        /// <summary>
        /// デバイス名を取得します。
        /// </summary>
        public string DeviceName
        {
            get
            {
                lock (_syncObj)
                {
                    return this.DevName_;
                }
            }
        }

        /// <summary>
        /// チャンネル登録処理を行います。
        ///
        /// Router.Startで新規接続が開始されたあと、
        /// ソケットの接続が完了(EndAccept)するたびに呼び出されます。
        /// </summary>
        internal void DoChanelRegist(SocketListenerResult result)
        {
            if (result.Exception == null)
            {
                // チャンネル情報を生成し、登録します。
                // result.Socketはチャンネル毎に用意されたアプリとの通信
                // のためのソケットです。
                ChannelInfo channelInfo = new ChannelInfo(result.Socket, _routerInfo);
                if (channelInfo.RegistChannel())
                {
                    _routerInfo.ChannelManager.Add(channelInfo);
                }
            }
        }

        /// <summary>
        /// スレッド関数
        /// </summary>
        void ThreadStart()
        {
            Stopwatch stopWatch = new Stopwatch();

            WaitHandle[] waitHandles = new WaitHandle[] { _exitThreadEvent };
            for (int sleepTime = 0; WaitHandle.WaitTimeout == WaitHandle.WaitAny(waitHandles, sleepTime, false); )
            {
                stopWatch.Start();
                bool bPollingDevice = false;

                try
                {
                    lock (_syncThread)
                    {
                        if (_commDev != null)
                        {
                            PollingDevice();
                            bPollingDevice = true;
                        }
                    }
                }
                catch (CommDeviceException ex)
                {
                    RouterLog.ServerReport(ServerReportType.Error, "HIO error occured. Disconnected. {0}", ex.Message);
                    Disconnect();
                }

                // 経過時間によりスレッドのスリープ時間を決める
                stopWatch.Stop();
                long elapsedMSec = stopWatch.ElapsedMilliseconds;
                stopWatch.Reset();

                int maxTime = bPollingDevice ? _vPollingInterval : IdleSleepTime;
                sleepTime = elapsedMSec < maxTime ? (int)(maxTime - elapsedMSec) : 0;
            }

            Debug.WriteLine("thread exit");
        }

        /// <summary>
        /// ICommDev(実機との通信路)の状態更新を監視して、
        /// 書き込みがあれば、書き込まれたデータを各チャンネルに振り分ける。
        ///
        /// PC側チャンネルから実機側への書き込み要求があれば、
        /// ICommDev(実機との通信路)を通じてデータを送信する。
        /// </summary>
        void PollingDevice()
        {
            bool bOldTargetConnect = _commDev.IsTargetConnect;

            _commDev.Negotiate();

            if (_commDev.IsTargetConnect)
            {
                if (!bOldTargetConnect)    // 切断状態から接続状態になった場合
                {
                    RouterLog.ServerReport("[{0}] is connected.", _commDev.Name);
                    _channelManager.SetTargetConnect(true);
                    this.OnTargetConnectionChanged(true, false);
                }

                if (_channelManager.IsRegistFileIOServer)    // File I/O サーバが登録されるまでデバイスからの読み込みを待つようにする
                {
                    //----------------------------------------------------------------------------
                    // ICommDev(実機との通信路)に実機側から書き込みがあったかどうかを状態を更新して
                    // 判断します。
                    if (_commDev.Read())
                    {
                        MessageData messageData;

                        // 実機側との共有メモリからメッセージ(書き込み)を読み出します。
                        // メッセージには、データのサイズとチャンネル情報が格納されています。
                        // チャンネル情報を元に、該当するソケットにデータを送信しています。
                        while ((messageData = _commDev.GetMessage()) != null)
                        {
                            if (messageData.Length > 0)
                            {
                                ChannelInfo channelInfo = _channelManager.GetChannelInfo((ushort)messageData.Channel);
                                if (channelInfo != null)
                                {
                                    channelInfo.WriteData(messageData.Data, messageData.Offset, messageData.Length);
                                }
                            }
                        }
                    }
                }

                Thread.Sleep(0);

                //----------------------------------------------------------------------------
                // ICommDev(実機との通信路)に対して、転送すべきデータがキューにあるか調べ、
                // データがあれば転送します。
                if (_dataPacketQueue.NewItemEvent.WaitOne(0, false))
                {
                    int packetNum = _dataPacketQueue.Count;    // 処理するChannelPacketを決定

                    // すべてのパケットについて...
                    for (; packetNum > 0; packetNum--)
                    {
                        DataPacket dataPacket = _dataPacketQueue.Dequeue();

                        for (int offset = 0; offset < dataPacket.Data.Length; )
                        {
                            int restBytes = dataPacket.Data.Length - offset;
                            int writeBytes;

                            // データの受信 . USB2EXIに送る。
                            DateTime stTime = DateTime.UtcNow;
                            // 更新なしで書き込み可能バイト数を取得
                            writeBytes = _commDev.GetWritableBytes(false);
                            if (writeBytes < restBytes)
                            {
                                int SendDataMinSize = (int)(_commDev.GetWriteBufferSize() / 2);

                                while ((writeBytes < SendDataMinSize) && (writeBytes < restBytes))
                                {
                                    // 更新ありで書き込み可能バイト数を取得
                                    writeBytes = _commDev.GetWritableBytes(true);
                                    if ((stTime - DateTime.UtcNow).TotalMilliseconds > 5000)    // あるms時間経過したら
                                    {
                                        RouterLog.ServerReport(ServerReportType.InternalError,
                                            "MccBufWriter can't alloc enough size. datasize:{0:d}", dataPacket.Data.Length);
                                        return;
                                    }
                                }
                            }

                            writeBytes = Math.Min(restBytes, writeBytes);
                            // 通信路への書き込み
                            _commDev.Write(dataPacket.Channel, dataPacket.Data, offset, writeBytes);
                            offset += writeBytes;
                        }
                        // Debug.WriteLine(string.Format("Received Data Size: {0:d}", dataPacket.Data.Length));
                    }
                }
            }
            else
            {
                if (bOldTargetConnect)  // 接続状態から切断状態になった場合
                {
                    if (_commDev is ICheckAlive && (_commDev as ICheckAlive).IsDisconnectByCheckAlive)
                    {
                        RouterLog.ServerReport("[{0}] is disconnected by timeout.", _commDev.Name);
                        this.OnTargetConnectionChanged(false, true);
                    }
                    else
                    {
                        RouterLog.ServerReport("[{0}] is disconnected.", _commDev.Name);
                        this.OnTargetConnectionChanged(false, false);
                    }
                    _channelManager.SetTargetConnect(false);
                    _dataPacketQueue.Clear();
                }
            }
        }

        /// <summary>
        /// ターゲットとの接続状態の変化を通知します。
        /// </summary>
        /// <param name="connect">ターゲットとの接続状態です。</param>
        /// <param name="disconnectByCheckAlive">生存確認によってターゲットとの接続が切断された場合に true を指定されます。</param>
        protected virtual void OnTargetConnectionChanged(bool connect, bool disconnectByCheckAlive)
        {
            if (this.TargetConnectionChanged != null)
            {
                this.TargetConnectionChanged(new TargetConnectionChangedEventArgs(connect, disconnectByCheckAlive));
            }
        }
    }

    /// <summary>
    /// Router例外クラス
    /// </summary>
    public class RouterException : MCSException
    {
        public RouterException(string message)
            : base(message)
        {
        }

        public RouterException(string message, Exception innerException)
            : base(message, innerException)
        {
        }
    }
}
