﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

// #define JHIREPORT

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Diagnostics;
using System.IO;
using System.Net;
using System.Net.Sockets;
using System.Reflection;
using System.Threading;

namespace Nintendo.McsServer
{
    public partial class MainForm : Form
    {
        public const string     DefaultApplicationNamespace = "Nintendo.McsServer";

        string                  _verString;

        Router                  _router;
        UserSettings            _userSettings   = new UserSettings();

        List<string>            _modifiedEnvVar = new List<string>();


        /// <summary>
        /// アプリケーションが終了中かどうかを示します。
        /// タスクトレイ格納中に、フォームの閉じるかどうかを判断するのに利用されます。
        /// </summary>
        bool                    _bAppExiting = false;

        CaptureForm _captureForm;
        Point _captureFormPosition = new Point(100, 100);

        KeyboardHook _keyboardHook = new KeyboardHook();
        MouseHook _mouseHook = new MouseHook();

        enum KeyboardHookState : uint
        {
            Idle = 0,
            Down = 1,
            Up   = 2,
            SequentialDown = 3
        }

        uint _keyboardHookState = 0;
        int _keyboardHookTime = 0;
        bool _mouseHookCheck = false;

        Icon _connectIcon = null;
        Icon _disconnectIcon = null;
        Icon _warningIcon = null;

        // ・・・・＋－－－－＋・・・・
        // ・　　　｜　　　　｜　　　・
        // ＋－－－　　　　　　－－－＋
        // ｜　　　　　　　　　　　　｜
        // ｜　　　　　　　　　　　　｜
        // ＋－－－　　　　　　－－－＋
        // ・　　　｜　　　　｜　　　・
        // ・・・・＋－－－－＋・・・・

        // 全スクリーン矩形（マルチスクリーン全体を包含）
        Rectangle _screenRect = new Rectangle(0, 0, 0, 0);
        // 全スクリーン矩形の上辺に位置する左右点
        Point _screenUpperLeft = new Point(0, 0);
        Point _screenUpperRight = new Point(0, 0);
        // 全スクリーン矩形の下辺に位置する左右点
        Point _screenDownLeft = new Point(0, 0);
        Point _screenDownRight = new Point(0, 0);
        // 全スクリーン矩形の左辺に位置する上下点
        Point _screenLeftUpper = new Point(0, 0);
        Point _screenLeftDown = new Point(0, 0);
        // 全スクリーン矩形の右辺に位置する上下点
        Point _screenRightUpper = new Point(0, 0);
        Point _screenRightDown = new Point(0, 0);

        EnvVarForm _envVarForm;

        public MainForm()
        {
            InitializeComponent();

            System.Reflection.Assembly myAssembly =
                System.Reflection.Assembly.GetExecutingAssembly();

            foreach (string resource in myAssembly.GetManifestResourceNames())
            {
                Debug.WriteLine(resource);
            }

            //指定されたマニフェストリソースを読み込む
            _connectIcon = new Icon(
                myAssembly.GetManifestResourceStream("Nintendo.McsServer.res.McsServer.ico"), 16, 16);

            _disconnectIcon = new Icon(
                myAssembly.GetManifestResourceStream("Nintendo.McsServer.res.McsServer_disconnected.ico"), 16, 16);

            _warningIcon = new Icon(
                myAssembly.GetManifestResourceStream("Nintendo.McsServer.res.McsServer_warning.ico"), 16, 16);

            Assembly mainAssembly = Assembly.GetEntryAssembly();
            _verString = string.Format("Ver. {0}", GetVersionString(mainAssembly));     // バージョン
            Icon icon = GetApplicationIcon();
            this.Icon = icon;                               // フォームのアイコン設定
            notifyIcon1.Icon = _disconnectIcon;            // 通知アイコンのアイコン設定
            this.Icon = _disconnectIcon;
            notifyIcon1.Text = Application.ProductName + " " + _verString;
            UpdateTitle(null);

            _userSettings = UserSettings.LoadAppSettings(); // アプリケーション設定の読み込み
            SetInTaskTray(_userSettings.IsInTaskTray);

            // 前回タスクトレイに入れて終了している場合は、
            // フォームを非表示で起動し、バルーンで起動を通知します。
            if (_userSettings.IsInTaskTray)
            {
                DisappearForm();

                notifyIcon1.ShowBalloonTip(
                    3000,
                    Application.ProductName,
                    Properties.Resources.RegidentMessage,
                    ToolTipIcon.Info);
            }
            else
            {
                AppearForm();
            }

            this.CalcScreenSize();

            _keyboardHook.KeyDown += OnHookKeyDown;
            _keyboardHook.KeyUp += OnHookKeyUp;
            _mouseHook.MouseMove += OnHookMouseMove;

            _keyboardHook.Start();
            _mouseHook.Start();

            _router = new Router(Callback_RouterDisconnect, this);
            _router.OnConnect = this.Router_OnConnect;

            _router.TargetConnectionChanged = delegate(Router.TargetConnectionChangedEventArgs args)
            {
                if (this.InvokeRequired)
                {
                    this.Invoke(new Action(() =>
                        {
                            this.Router_TargetConnectionChanged(args);
                        }));
                }
                else
                {
                    this.Router_TargetConnectionChanged(args);
                }
            };

            // 初期化時にコントロールにアクセスするため、
            // ここで必ずウィンドウハンドルが生成された状態にしますす。
            if (!this.IsHandleCreated)
            {
                this.CreateHandle();
            }

            this.Initialize();
        }

        public static Stream GetResourceStream(Assembly assembly, string resName)
        {
            return assembly.GetManifestResourceStream(DefaultApplicationNamespace + "." + resName);
        }

        public static Icon GetApplicationIcon()
        {
            return new Icon(GetResourceStream(Assembly.GetEntryAssembly(), "res.McsServer.ico"));
        }

        /// <summary>
        /// バージョン名（AssemblyVersion属性）を取得
        /// </summary>
        /// <param name="mainAssembly"></param>
        /// <returns></returns>
        public static string GetVersionString(Assembly assembly)
        {
            Version appVersion = assembly.GetName().Version;
            return string.Format("{0}.{1}.{2}", appVersion.Major, appVersion.Minor, appVersion.Build);
        }

        static ActionParams AnalysisCommandLine(RuntimeSettings runtimeSettings, string[] args)
        {
            ActionParams actionParams = new ActionParams();
            char[] switchStartChars = { '/', '-' };
            const string longSwitchStartChars = "--";

            for (int j = 0; j < 2; ++j)
            {
                bool bExecMode = j == 1;

                for (int i = 0; i < args.Length;)
                {
                    string arg = args[i++];

                    // 長いスイッチを先に判定する
                    if (arg.StartsWith(longSwitchStartChars, StringComparison.InvariantCulture))    // '--'スイッチ
                    {
                        string swtStr = null;
                        string swtParam = null;
                        int sepIdx = arg.IndexOf('=', longSwitchStartChars.Length);
                        if (sepIdx != -1)
                        {
                            swtParam = arg.Substring(sepIdx + 1);
                            swtStr = arg.Substring(longSwitchStartChars.Length, sepIdx - longSwitchStartChars.Length);
                        }
                        else
                        {
                            swtStr = arg.Substring(longSwitchStartChars.Length);
                        }

                        swtStr = swtStr.ToLowerInvariant();

                        switch (swtStr)
                        {
                        case "connect":
                            if (bExecMode)
                            {
                                actionParams.IsBootConnect = true;
                                actionParams.ExplicitConnectDeviceName = swtParam;
                            }
                            break;
                        default:
                            throw new MCSException(string.Format(Properties.Resources.ErrorInvalidOption, swtStr));
                        }
                    }
                    else if (arg[0] == switchStartChars[0] || arg[0] == switchStartChars[1])        // '-'スイッチ
                    {
                        string swtStr = arg.Substring(1).ToLowerInvariant();

                        switch (swtStr)
                        {
                        case "c":
                            if (bExecMode)
                            {
                                actionParams.IsBootConnect = true;
                                actionParams.ExplicitConnectDeviceName = null;
                            }
                            break;
                        default:
                            throw new MCSException(string.Format(Properties.Resources.ErrorInvalidOption, swtStr));
                        }
                    }
                    else                // パラメータ
                    {
                        throw new MCSException(string.Format(Properties.Resources.ErrorParameter, arg));
                    }
                }
            }

            return actionParams;
        }

        private void Initialize()
        {
            _bAppExiting = false;

            RouterLog.SetRichTextBox(this.rtbConsole);
            RouterLog.ServerReport(Assembly.GetEntryAssembly().GetName().Name + " " + _verString);

#if JHIREPORT
        {
            // コンソールに出そうと試みたが、デバッガの出力ウィンドウに出てしまう。
            Win32.AllocConsole();
            Debug.Listeners.Add(new TextWriterTraceListener(Console.Out));
        }
#endif

            // 最初の引数(実行ファイル名)を取り除く
            string[] orgArgs = System.Environment.GetCommandLineArgs();
            string[] args = new string[orgArgs.Length - 1];
            Array.Copy(orgArgs, 1, args, 0, args.Length);

            // コマンドラインオプションの解析(コマンドラインオプションは既定の設定を上書きする)
            ActionParams actionParams = new ActionParams();
            try
            {
                actionParams = AnalysisCommandLine(Program.RuntimeSettings, args);
                // 起動時には必ず接続するようにする。
                actionParams.IsBootConnect = true;
            }
            catch (ApplicationException ex)
            {
                ShowMessageBox(ex);
            }

            try
            {
                _router.Start(_userSettings.PortNo);
            }
            catch (SocketException ex)
            {
                if (ex.SocketErrorCode == SocketError.AddressAlreadyInUse)
                {
                    // 何らかのソフトが同じポートNo.を使用している
                    ShowMessageBox(Properties.Resources.errorAddressInUse);
                }
                else
                {
                    // その他のSocket例外
                    ShowMessageBox(ex);
                }

                // Routerをスターとさせないで起動。ポート番号を変更してもらう
                return;
            }

            if (actionParams.IsBootConnect)
            {
                try
                {
                    Connect(actionParams.ExplicitConnectDeviceName);
                }
                catch (ApplicationException ex)
                {
                    ShowMessageBox(ex);
                }
            }
        }

        private void menuItemExit_Click(object sender, EventArgs e)
        {
            _bAppExiting = true;

            Close();
            Application.Exit();
        }

        private void Form1_FormClosing(object sender, FormClosingEventArgs e)
        {
            Debug.WriteLine("Form1_FormClosing");

            if (_userSettings.IsInTaskTray && !_bAppExiting)
            {
                if (notifyIcon1.Visible)
                {
                    e.Cancel = true;
                }

                this.Visible = false;
            }
        }

        private void MainForm_FormClosed(object sender, FormClosedEventArgs e)
        {
            Debug.WriteLine("Form1_FormClosed");

            if (_captureForm != null)
            {
                _captureForm.Close();
            }

            Debug.WriteLine("CaptureForm close after.");

            try
            {
                Disconnect();
            }
            catch (ApplicationException ex)
            {
                Debug.WriteLine(ex.Message);
            }

            _router.Stop();
            _router = null;

            RouterLog.SetRichTextBox(null);
            _userSettings.SaveAppSettings();

            if (!_userSettings.IsInTaskTray || _bAppExiting)
            {
                Application.Exit();
            }
        }

        private void menuItemFile_DropDownOpening(object sender, EventArgs e)
        {
            menuItemTaskTray.Checked = _userSettings.IsInTaskTray;
        }

        private void contextMenu1_Opening(object sender, CancelEventArgs e)
        {
            conMenuItemTaskTray.Checked = _userSettings.IsInTaskTray;

            conMenuItemConnect.Enabled = _router.Active;
            conMenuItemDisconnect.Enabled = _router.Active && _router.IsConnect;

            if (_router.Active && _router.IsConnect)
            {
                conMenuItemConnect.Text = Properties.Resources.MenuReconnect + "(&C)";
            }
            else
            {
                conMenuItemConnect.Text = Properties.Resources.MenuConnect + "(&C)";
            }

        }

        void SetInTaskTray(bool bInTaskTray)
        {
            _userSettings.IsInTaskTray = bInTaskTray;
            notifyIcon1.Visible = _userSettings.IsInTaskTray;
        }

        /// <summary>
        /// McsServer のウィンドウを表示します。
        /// </summary>
        void AppearForm()
        {
            this.ShowInTaskbar = true;
            this.Visible = true;
            if (WindowState == FormWindowState.Minimized)
            {
                WindowState = FormWindowState.Normal; // 最小化をやめる
            }

            Activate(); // フォームをアクティブにする
        }

        /// <summary>
        /// McsServer のウィンドウを非表示にします。
        /// </summary>
        void DisappearForm()
        {
            this.Visible = false;
        }

        private void menuItemTaskTray_Click(object sender, EventArgs e)
        {
            ToolStripMenuItem mi = (ToolStripMenuItem)sender;
            bool bInTaskTray = ! mi.Checked;
            SetInTaskTray(bInTaskTray);

            // 通知アイコンのコンテキストメニューから選択された場合で、
            // タスクトレイから出る場合は、フォームを出現させておく。
            if (! bInTaskTray)
            {
                AppearForm();
            }
        }

        /// <summary>
        /// タスクトレーアイコンのクリック時の動作。
        /// </summary>
        private void notifyIcon1_MouseDown(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Left)
            {
                if (this.Visible && this.WindowState != FormWindowState.Minimized)
                {
                    this.DisappearForm();
                    return;
                }

                this.AppearForm();
            }
        }

        private void menuItemAbout_Click(object sender, EventArgs e)
        {
            AboutDialog dlg = new AboutDialog();
            dlg.ShowDialog();
        }

        private void menuItem1_DropDownOpening(object sender, EventArgs e)
        {
            menuItemConnect.Enabled = _router.Active;
            menuItemDisconnect.Enabled = _router.Active && _router.IsConnect;

            if (_router.Active && _router.IsConnect)
            {
                menuItemConnect.Text = Properties.Resources.MenuReconnect + "(&C)";
            }
            else
            {
                menuItemConnect.Text = Properties.Resources.MenuConnect + "(&C)";
            }
        }

        void Connect(string explicitConnectDevice)
        {
            Thread.Sleep(500);

            Router.ConnectParam param = new Router.ConnectParam();
            param.PollingInterval = _userSettings.PollingInterval;
            param.DisableDeviceUSBAdapter = _userSettings.DisableDeviceUSBAdapter;
            param.DisableDeviceNDEV = _userSettings.DisableDeviceCATDEV;
            param.MultiDeviceDelegate = new Router.MultiDeviceDelegate(ShowDeviceSelect);
            param.explicitConnectDevice = explicitConnectDevice;
            param.CheckAliveTimeout = _userSettings.IsCheckAlive? _userSettings.CheckAliveTimeout : 0;
            string deviceInfo = _router.Connect(param);
            if (deviceInfo != null)
            {
                Text = Application.ProductName + " - [" + deviceInfo + "]";
                notifyIcon1.Text = Application.ProductName + " " + _verString + " - [" + deviceInfo + "]";
            }
        }

        void Disconnect()
        {
            _router.Disconnect();
            UpdateTitle(null);
        }

        void UpdateTitle(string deviceName)
        {
            if (deviceName != null)
            {
                Text = Application.ProductName + " - [" + deviceName + "]";
                notifyIcon1.Text = Application.ProductName + " " + _verString + " - [" + deviceName + "]";
            }
            else
            {
                Text = Application.ProductName;
                notifyIcon1.Text = Application.ProductName + " " + _verString;
            }
        }

        private void menuItemOptionDialog_Click(object sender, EventArgs e)
        {
            using (OptionDialog dlg = new OptionDialog(_userSettings))
            {
                if (DialogResult.OK == dlg.ShowDialog())
                {
                    Debug.WriteLine("option dialog result ok.");
                }
                else
                {
                    Debug.WriteLine("option dialog result cancel.");
                }
            }
        }

        void ShowMessageBox(Exception e)
        {
            ShowMessageBox(e.Message);
        }

        void ShowMessageBox(string message)
        {
            MessageBox.Show(this, message, Application.ProductName, MessageBoxButtons.OK, MessageBoxIcon.Error);
        }

        int ShowDeviceSelect(object[] devices)
        {
            using (DeviceSelectDialog dlg = new DeviceSelectDialog())
            {
                dlg.DeviceList = devices;
                dlg.SelectedIndex = 0;  // デバイスが複数あることが前提なので、1つ目をあらかじめ選択しておく。
                if (DialogResult.OK == dlg.ShowDialog(this))
                {
                    return dlg.SelectedIndex;
                }
                else
                {
                    return -1;
                }
            }
        }

        private void menuItemLaunchHelp_Click(object sender, EventArgs e)
        {
            Help.ShowHelp(this, "docs/McsServer.chm");
        }

        public void DoSendAppArgs(string[] args)
        {
            // コマンドラインオプションの解析(コマンドラインオプションは既定の設定を上書きする)
            ActionParams actionParams = new ActionParams();
            try
            {
                actionParams = AnalysisCommandLine(Program.RuntimeSettings, args);
            }
            catch (ApplicationException ex)
            {
                Activate();
                ShowMessageBox(ex);
                return;
            }

            if (actionParams.IsBootConnect)
            {
                try
                {
                    if (_router.IsConnect &&
                        (string.IsNullOrEmpty(actionParams.ExplicitConnectDeviceName) || actionParams.ExplicitConnectDeviceName == _router.DeviceName))
                    {
                        return;
                    }

                    Disconnect();
                    Connect(actionParams.ExplicitConnectDeviceName);
                }
                catch (ApplicationException ex)
                {
                    Activate();
                    ShowMessageBox(ex);
                }
            }
        }

        private void conMenuItemOpen_Click(object sender, EventArgs e)
        {
            AppearForm();
        }

        protected override void WndProc(ref Message m)
        {
            // シャットダウンなどのメッセージを捕捉する。
            // FormClosingの前に処理する必要があるため、
            // SystemEvents.SessionEnding イベントではなく、直接 WM_QUERYENDSESSION を処理する。
            const int WM_QUERYENDSESSION = 0x0011;

            if (m.Msg == WM_QUERYENDSESSION)
            {
                _bAppExiting = true;
            }
            base.WndProc(ref m);
        }

        void Callback_RouterDisconnect(string devName)
        {
            if (InvokeRequired)
            {
                // Invokeが必要な場合はこの呼び出しをBeginInvoke()によりFormのスレッドから呼んでもらうようにする。
                BeginInvoke(new RouterDisconnectCallback(Callback_RouterDisconnect), devName);
            }
            else
            {
                Debug.WriteLine(string.Format("Router disconnected. - [{0}]", devName));

                // このコールバックはBeginInvoke()により呼ばれる可能性があり、
                // そうなると、新しいConnect()呼び出しが発生して別のデバイスに接続している可能性がある。
                // なので、現在のデバイス名を取得する。
                UpdateTitle(_router.DeviceName);
            }
        }

        private void menuItemInputCapture_Click(object sender, EventArgs e)
        {
            if (_captureForm == null)
            {
                this.OpenCaptureForm();
            }
        }

        /// <summary>
        /// キャプチャダイアログが閉じる直前に呼び出されるイベントです。
        /// </summary>
        private void _captureForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            _captureFormPosition = _captureForm.Location;
        }

        /// <summary>
        /// キャプチャダイアログが閉じる直後に呼び出されるイベントです。
        /// </summary>
        private void _captureForm_FormClosed(object sender, FormClosedEventArgs e)
        {
            _captureForm = null;
        }

        private void menuItemTool_DropDownOpening(object sender, EventArgs e)
        {
            menuItemInputCapture.Enabled = _captureForm == null;
        }

        /// <summary>
        /// キャプチャフォームを開きます。
        /// </summary>
        private void OpenCaptureForm()
        {
            _captureForm = new CaptureForm();
            _captureForm.Port = _userSettings.PortNo;
            _captureForm.FormClosing += new FormClosingEventHandler(_captureForm_FormClosing);
            _captureForm.FormClosed += new FormClosedEventHandler(_captureForm_FormClosed);
            _captureForm.Show();
            _captureForm.Location = _captureFormPosition;
        }

        private void StartInputCapture()
        {
            if (!(_router.Active && _router.IsConnect))
            {
                return;
            }

            if (_captureForm == null)
            {
                this.OpenCaptureForm();
            }

            _captureForm.StartCapture();
        }

        /// <summary>
        /// 画面サイズを計算します。
        /// </summary>
        private void CalcScreenSize()
        {
            // 厳密にすると面倒なので、最小公倍数の Rectangle で判定している。

            int top = 0;
            int bottom = 0;
            int left = 0;
            int right = 0;

            foreach (Screen screen in Screen.AllScreens)
            {
                if (screen == Screen.AllScreens[0])
                {
                    top = screen.Bounds.Top;
                    bottom = screen.Bounds.Bottom;
                    left = screen.Bounds.Left;
                    right = screen.Bounds.Right;

                    _screenUpperLeft.X = _screenDownLeft.X = _screenLeftUpper.X = _screenLeftDown.X = left;
                    _screenUpperRight.X = _screenDownRight.X = _screenRightUpper.X = _screenRightDown.X = right;
                    _screenUpperLeft.Y = _screenUpperRight.Y = _screenLeftUpper.Y = _screenRightUpper.Y = top;
                    _screenDownLeft.Y = _screenDownRight.Y = _screenLeftDown.Y = _screenRightDown.Y = bottom;

                    continue;
                }

                // 全スクリーン矩形（マルチスクリーン全体を包含）
                top = Math.Min(top, screen.Bounds.Top);
                bottom = Math.Max(bottom, screen.Bounds.Bottom);
                left = Math.Min(left, screen.Bounds.Left);
                right = Math.Max(right, screen.Bounds.Right);

                // 全スクリーン矩形の上辺に位置する左右点
                if (_screenUpperLeft.Y > screen.Bounds.Top ||
                    (_screenUpperLeft.Y == screen.Bounds.Top && _screenUpperLeft.X > screen.Bounds.Left))
                {
                    _screenUpperLeft.X = screen.Bounds.Left;
                    _screenUpperLeft.Y = screen.Bounds.Top;
                }

                if (_screenUpperRight.Y > screen.Bounds.Top ||
                    (_screenUpperRight.Y == screen.Bounds.Top && _screenUpperRight.X < screen.Bounds.Right))
                {
                    _screenUpperRight.X = screen.Bounds.Right;
                    _screenUpperRight.Y = screen.Bounds.Top;
                }

                // 全スクリーン矩形の下辺に位置する左右点
                if (_screenDownLeft.Y < screen.Bounds.Bottom ||
                    (_screenDownLeft.Y == screen.Bounds.Bottom && _screenDownLeft.X > screen.Bounds.Left))
                {
                    _screenDownLeft.X = screen.Bounds.Left;
                    _screenDownLeft.Y = screen.Bounds.Bottom;
                }

                if (_screenDownRight.Y < screen.Bounds.Bottom ||
                    (_screenDownRight.Y == screen.Bounds.Bottom && _screenDownRight.X < screen.Bounds.Right))
                {
                    _screenDownRight.X = screen.Bounds.Right;
                    _screenDownRight.Y = screen.Bounds.Bottom;
                }

                // 全スクリーン矩形の左辺に位置する上下点
                if (_screenLeftUpper.X > screen.Bounds.Left ||
                    (_screenLeftUpper.X == screen.Bounds.Left && _screenLeftUpper.Y > screen.Bounds.Top))
                {
                    _screenLeftUpper.X = screen.Bounds.Left;
                    _screenLeftUpper.Y = screen.Bounds.Top;
                }

                if (_screenLeftDown.X > screen.Bounds.Left ||
                    (_screenLeftDown.X == screen.Bounds.Left && _screenLeftDown.Y < screen.Bounds.Bottom))
                {
                    _screenLeftDown.X = screen.Bounds.Left;
                    _screenLeftDown.Y = screen.Bounds.Bottom;
                }

                // 全スクリーン矩形の右辺に位置する上下点
                if (_screenRightUpper.X < screen.Bounds.Right ||
                    (_screenRightUpper.X == screen.Bounds.Right && _screenRightUpper.Y > screen.Bounds.Top))
                {
                    _screenRightUpper.X = screen.Bounds.Right;
                    _screenRightUpper.Y = screen.Bounds.Top;
                }

                if (_screenRightDown.X < screen.Bounds.Right ||
                    (_screenRightDown.X == screen.Bounds.Right && _screenRightDown.Y < screen.Bounds.Bottom))
                {
                    _screenRightDown.X = screen.Bounds.Right;
                    _screenRightDown.Y = screen.Bounds.Bottom;
                }
            }

            _screenRect = new Rectangle(left, top, right - left, bottom - top);
        }

        /// <summary>
        /// KeyDown のグローバルフックイベントです。
        /// </summary>
        private void OnHookKeyDown(object sender, KeyboardHookEventArgs e)
        {
            if (!_userSettings.IsCaptureOnKeyShortcut)
            {
                return;
            }

            if (CheckKeyShortcut(e.KeyCode))
            {
                int MaxKeyInterval = _userSettings.CaptureOnKeyShortcutInterval;

                int time = System.Environment.TickCount;

                if (_keyboardHookState == (uint)KeyboardHookState.Idle) // 初めて押下した場合
                {
                    _keyboardHookState = (uint)KeyboardHookState.Down;
                    _keyboardHookTime = time;
                }
                else if (_keyboardHookState == (uint)KeyboardHookState.Up) // 一度 KeyUp をしてから再度 KeyDown
                {
                    if (time - _keyboardHookTime > MaxKeyInterval)
                    {
                        _keyboardHookState = (uint)KeyboardHookState.Down;
                        _keyboardHookTime = time;

                    }
                    else
                    {
                        // KeyDown でキャプチャを開始すると実機に KeyUp イベントが送られるため、 KeyUp で開始する
                        _keyboardHookState = (uint)KeyboardHookState.SequentialDown;
                        _keyboardHookTime = time;
                    }
                }
            }
        }

        /// <summary>
        /// KeyDown のグローバルフックイベントです。
        /// </summary>
        private void OnHookKeyUp(object sender, KeyboardHookEventArgs e)
        {
            if (_keyboardHookState == (uint)KeyboardHookState.Down)
            {
                _keyboardHookState = (uint)KeyboardHookState.Up;
            }

            if (CheckKeyShortcut(e.KeyCode))
            {
                int MaxKeyInterval = _userSettings.CaptureOnKeyShortcutInterval;

                int time = System.Environment.TickCount;

                if (_keyboardHookState == (uint)KeyboardHookState.SequentialDown)
                {
                    if (time - _keyboardHookTime > MaxKeyInterval)
                    {
                        _keyboardHookState = (uint)KeyboardHookState.Up;
                        _keyboardHookTime = time;
                    }
                    else
                    {
                        _keyboardHookState = (uint)KeyboardHookState.Idle;
                        // グローバルフック中で重い処理をおこなうと、グローバルフックから外されてしまうので、
                        // フラグを立てるだけにしておく。

                        _mouseHookCheck = true;
                        mouseGestureTimer.Interval = 1;
                        mouseGestureTimer.Enabled = true;
                        mouseGestureTimer.Start();
                    }
                }
            }
        }

        /// <summary>
        /// ショートカットキーと一致するかどうかをチェックします。
        /// </summary>
        private bool CheckKeyShortcut(int keycode)
        {
            // LControlKey と RControlKey は同じものとして扱う。
            if (keycode == (int)Keys.RControlKey)
            {
                keycode = (int)Keys.LControlKey;
            }

            return (keycode == (int)_userSettings.CaptureShortcutKey);
        }

        /// <summary>
        /// マウスショートカットの位置にいるかどうかをチェックします。
        /// </summary>
        private bool CheckMouseShortcut(int x, int y)
        {
            const int Mergin = 10;
            Point cursol = new Point(x, y);

            if ((_userSettings.CaptureMousePosition & UserSettings.CapturePosition.UpperLeft) != 0)
            {
                if (PointEquals(cursol, _screenUpperLeft, Mergin))
                {
                    return true;
                }

                if (PointEquals(cursol, _screenLeftUpper, Mergin))
                {
                    return true;
                }
            }

            if ((_userSettings.CaptureMousePosition & UserSettings.CapturePosition.UpperRight) != 0)
            {
                if (PointEquals(cursol, _screenUpperRight, Mergin))
                {
                    return true;
                }

                if (PointEquals(cursol, _screenRightUpper, Mergin))
                {
                    return true;
                }
            }

            if ((_userSettings.CaptureMousePosition & UserSettings.CapturePosition.DownLeft) != 0)
            {
                if (PointEquals(cursol, _screenDownLeft, Mergin))
                {
                    return true;
                }

                if (PointEquals(cursol, _screenLeftDown, Mergin))
                {
                    return true;
                }
            }

            if ((_userSettings.CaptureMousePosition & UserSettings.CapturePosition.DownRight) != 0)
            {
                if (PointEquals(cursol, _screenDownRight, Mergin))
                {
                    return true;
                }

                if (PointEquals(cursol, _screenRightDown, Mergin))
                {
                    return true;
                }
            }

            if ((_userSettings.CaptureMousePosition & UserSettings.CapturePosition.Top) != 0)
            {
                if (cursol.Y <= _screenRect.Top + Mergin)
                {
                    return true;
                }
            }

            if ((_userSettings.CaptureMousePosition & UserSettings.CapturePosition.Bottom) != 0)
            {
                if (cursol.Y >= _screenRect.Bottom - Mergin)
                {
                    return true;
                }
            }

            if ((_userSettings.CaptureMousePosition & UserSettings.CapturePosition.Left) != 0)
            {
                if (cursol.X <= _screenRect.Left + Mergin)
                {
                    return true;
                }
            }

            if ((_userSettings.CaptureMousePosition & UserSettings.CapturePosition.Right) != 0)
            {
                if (cursol.X >= _screenRect.Right - Mergin)
                {
                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// MouseMove のグローバルフックイベントです。
        /// </summary>
        private void OnHookMouseMove(object sender, MouseHookEventArgs e)
        {
            if (!_userSettings.IsCaptureOnMousePosition)
            {
                return;
            }

            if (_router.IsTargetConnect && this.CheckMouseShortcut(e.X, e.Y))
            {
                if (!_mouseHookCheck)
                {
                    if (_userSettings.CaptureOnMousePositionCount == 0)
                    {
                        mouseGestureTimer.Interval = 1;
                    }
                    else
                    {
                        mouseGestureTimer.Interval = _userSettings.CaptureOnMousePositionCount;
                    }

                    mouseGestureTimer.Enabled = true;
                    mouseGestureTimer.Start();
                }
                _mouseHookCheck = true;
            }
            else
            {
                _mouseHookCheck = false;
            }
        }

        /// <summary>
        /// 位置が一致しているかどうかを判定します。
        /// </summary>
        /// <param name="p1">比較対象の位置情報１です。</param>
        /// <param name="p2">比較対象の位置情報２です。</param>
        /// <param name="mergin">許容誤差です。</param>
        /// <returns>位置が等しいと判定した場合に、true を返します。</returns>
        private static bool PointEquals(Point p1, Point p2, int mergin)
        {
            if ((Math.Abs(p1.X - p2.X) <= mergin) && (Math.Abs(p1.Y - p2.Y) <= mergin))
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        /// <summary>
        /// マウスジェスチャー用のタイマーイベント処理です。
        /// </summary>
        private void mouseGestureTimer_Tick(object sender, EventArgs e)
        {
            mouseGestureTimer.Stop();
            if (_mouseHookCheck)
            {
                this.StartInputCapture();
            }
        }

        /// <summary>
        /// メニューから接続を選択します。
        /// </summary>
        private void menuItemConnect_Click(object sender, EventArgs e)
        {
            try
            {
                Disconnect();
                Connect(null);
            }
            catch (ApplicationException ex)
            {
                ShowMessageBox(ex);
            }
        }

        /// <summary>
        /// メニューから切断を選択します。
        /// </summary>
        private void menuItemDisconnect_Click(object sender, EventArgs e)
        {
            try
            {
                Disconnect();
            }
            catch (ApplicationException ex)
            {
                ShowMessageBox(ex);
            }
        }

        /// <summary>
        /// コンテキストメニューの[接続] 実行時の動作です。
        /// </summary>
        private void conMenuItemConnect_Click(object sender, EventArgs e)
        {
            try
            {
                Disconnect();
                Connect(null);
            }
            catch (ApplicationException ex)
            {
                ShowMessageBox(ex);
            }
        }

        /// <summary>
        /// コンテキストメニューの [切断] 実行時の動作です。
        /// </summary>
        private void conMenuItemDisconnect_Click(object sender, EventArgs e)
        {
            try
            {
                Disconnect();
            }
            catch (ApplicationException ex)
            {
                ShowMessageBox(ex);
            }
        }

        /// <summary>
        /// コンテキストメニューのオプション選択時の動作です。
        /// </summary>
        private void conMenuItemOption_Click(object sender, EventArgs e)
        {
            using (OptionDialog dlg = new OptionDialog(_userSettings))
            {
                if (DialogResult.OK == dlg.ShowDialog())
                {
                    Debug.WriteLine("option dialog result ok.");
                }
                else
                {
                    Debug.WriteLine("option dialog result cancel.");
                }
            }
        }

        /// <summary>
        /// コンテキストメニューの入力キャプチャ選択時の動作です。
        /// </summary>
        private void conMenuItemCapture_Click(object sender, EventArgs e)
        {
            if (_captureForm == null)
            {
                this.OpenCaptureForm();
            }
        }

        /// <summary>
        /// Router の接続通知です。
        /// </summary>
        private void Router_OnConnect(bool connect)
        {
            if (connect)
            {
                notifyIcon1.Icon = _warningIcon;
            }
            else
            {
                notifyIcon1.Icon = _disconnectIcon;
            }

            this.Icon = notifyIcon1.Icon;
        }

        private void Router_TargetConnectionChanged(Router.TargetConnectionChangedEventArgs args)
        {
            if (args.IsConnected)
            {
                notifyIcon1.Icon = _connectIcon;
            }
            else
            {
                notifyIcon1.Icon = _warningIcon;
            }

            this.Icon = notifyIcon1.Icon;
        }

        /// <summary>
        /// 環境変数の一覧を表示します。
        /// </summary>
        private void ShowEnvVarForm()
        {
            if (_envVarForm == null)
            {
                _envVarForm = new EnvVarForm(_modifiedEnvVar);
            }

            if (!_envVarForm.Visible)
            {
                _envVarForm.ShowDialog();
            }
        }

        /// <summary>
        /// 環境変数の一覧表示のメニュー処理です。
        /// </summary>
        private void menuItemEnvVar_Click(object sender, EventArgs e)
        {
            this.ShowEnvVarForm();
        }

        /// <summary>
        /// 環境変数の一覧表示のメニュー処理です。
        /// </summary>
        private void conMenuItemEnvVar_Click(object sender, EventArgs e)
        {
            this.ShowEnvVarForm();
        }
    }
}
