﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.IO;
using System.Net;
using System.Net.Sockets;

namespace Nintendo.McsServer
{
    /// <summary>
    /// チャンネル登録要求コマンドの種類
    /// </summary>
    enum ChannelRequestCommand : uint
    {
        Unknown         = 0,
        ChannelRequest
    }

    /// <summary>
    /// チャンネル登録要求の結果
    /// </summary>
    public enum ChannelRequestResult : uint
    {
        Success,                // 成功
        ProtocolError,          // 通信プロトコルのエラーです。
        AlreadyRegisted,        // 既に登録されているチャンネルです。
        SystemReserved,         // システムで予約しています。
        NotConnect              // ターゲットと接続されていません。
    }

    /// <summary>
    /// チャンネル登録要求のオプションフラグ
    /// </summary>
    [Flags()]
    public enum ChannelRequestFlags : uint
    {
        NoCheckConnect  = 1 << 0    // ターゲットとの接続状態に依存しない
    }

    /// <summary>
    /// 指定されたソケット(ストリーム)に対して、チャンネル登録のプロトコルを
    /// 実行するクラスです。
    ///
    /// 送信したコマンドは、リモート側のソケットを管理するChannelInfo内部で受信され、
    /// レスポンスデータが返信されます。
    ///
    /// ChannelInfoとsocketの関連付けは、SocketListenerクラスが行っています。
    /// </summary>
    public class ChannelRequest
    {
        /// <summary>
        /// チャンネル登録のプロトコルを実行し、登録結果を返します。
        /// </summary>
        public static ChannelRequestResult Request( Socket socket, uint channel )
        {
            return Request(socket, channel, 0);
        }

        /// <summary>
        /// チャンネル登録のプロトコルを実行し、登録結果を返します。
        /// </summary>
        public static ChannelRequestResult Request( Socket socket, uint channel, uint flag )
        {
            return Request(new NetworkStream(socket, FileAccess.ReadWrite, false), channel, flag);
        }

        /// <summary>
        /// チャンネル登録のプロトコルを実行し、登録結果を返します。
        /// </summary>
        public static ChannelRequestResult Request( Stream stream, uint channel )
        {
            return Request(stream, channel, 0);
        }

        /// <summary>
        /// チャンネル登録のプロトコルを実行し、登録結果を返します。
        /// </summary>
        public static ChannelRequestResult Request( Stream stream, uint channel, uint flag )
        {
            // 接続の確立のプロトコルを送信
            using (MemoryStream ms = new MemoryStream())
            {
                BinaryWriter bw = new BinaryWriter(ms);
                // ヘッダ
                uint val = (uint)ChannelRequestCommand.ChannelRequest;
                bw.Write(HostToNet(val));       // コマンド(ターゲットデバイスとの接続)
                val = 4 + 4;
                bw.Write(HostToNet(val));       // データ長

                // データ
                bw.Write(HostToNet(channel));   // 希望するチャンネル
                bw.Write(HostToNet(flag));      // フラグ
                bw.Flush();

                stream.Write(ms.GetBuffer(), 0, (int)ms.Length);    // メッセージ長
            }

            // レスポンスをチェック
            BinaryReader br = new BinaryReader(stream);
            uint chunkID = NetToHost(br.ReadUInt32());
            uint msgSize = NetToHost(br.ReadUInt32());
            if (msgSize > 0x10000)      // メッセージ長が大きすぎる
            {
                return ChannelRequestResult.ProtocolError;
            }
            if (msgSize != 4) // メッセージ長が期待している長さではない
            {
                for (int i = 0; i < msgSize; ++i)   // メッセージを空読み
                {
                    br.ReadByte();
                }
                return ChannelRequestResult.ProtocolError;
            }

            return (ChannelRequestResult)NetToHost(br.ReadUInt32());
        }

        /// <summary>
        ///
        /// </summary>
        static short HostToNet(short val)
        {
            return IPAddress.HostToNetworkOrder(val);
        }

        /// <summary>
        ///
        /// </summary>
        static ushort HostToNet( ushort val )
        {
            return (ushort)IPAddress.HostToNetworkOrder((short)val);
        }

        /// <summary>
        ///
        /// </summary>
        static int HostToNet( int val )
        {
            return IPAddress.HostToNetworkOrder(val);
        }

        static uint HostToNet(uint val)
        {
            return (uint)IPAddress.HostToNetworkOrder((int)val);
        }

        /// <summary>
        ///
        /// </summary>
        static short NetToHost( short val )
        {
            return IPAddress.NetworkToHostOrder(val);
        }

        /// <summary>
        ///
        /// </summary>
        static ushort NetToHost( ushort val )
        {
            return (ushort)IPAddress.NetworkToHostOrder((short)val);
        }

        /// <summary>
        ///
        /// </summary>
        static int NetToHost( int val )
        {
            return IPAddress.NetworkToHostOrder(val);
        }

        /// <summary>
        ///
        /// </summary>
        static uint NetToHost( uint val )
        {
            return (uint)IPAddress.NetworkToHostOrder((int)val);
        }

    }
}
