﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;

namespace Nintendo.McsServer
{
    /// <summary>
    /// CommDevice の概要の説明です。
    /// </summary>
    public interface ICommDevice : IDisposable
    {
        /// <summary>
        /// オブジェクトの初期化。
        /// </summary>
        /// <returns></returns>
        // void Init();

        /// <summary>
        /// ハードウェアに対するアタッチ。
        /// </summary>
        /// <returns></returns>
        // void Attach();

        /// <summary>
        /// リモート側プログラムとの通信の確立
        /// </summary>
        /// <returns></returns>
        void Negotiate();

        /// <summary>
        /// ハードウェアに対するデタッチ。
        /// </summary>
        /// <returns></returns>
        // void Detach();

        string Name
        {
            get;
        }

        bool IsAttach
        {
            get;
        }

        bool IsTargetConnect
        {
            get;
        }

        bool Read();

        MessageData GetMessage();

        int GetWritableBytes(bool withUpdate);

        void Write(int channel, byte[] buf, int offset, int size);

        uint GetWriteBufferSize();
    }

    public class MessageData
    {
        public int Channel
        {
            get;
            set;
        }

        public int Offset
        {
            get;
            set;
        }

        public int Length
        {
            get;
            set;
        }

        public byte[] Data
        {
            get;
            set;
        }

        public MessageData(int channel, byte[] data)
        {
            if (data == null)
            {
                throw new ArgumentNullException("data");
            }

            this.Channel = channel;
            this.Data = data;
            this.Offset = 0;
            this.Length = data.Length;
        }

        public MessageData(int channel, byte[] data, int offset, int length)
        {
            if (data == null)
            {
                throw new ArgumentNullException("data");
            }

            if (offset > data.Length)
            {
                throw new ArgumentOutOfRangeException("offset");
            }

            if (offset + length > data.Length)
            {
                throw new ArgumentOutOfRangeException("length");
            }

            this.Channel = channel;
            this.Data = data;
            this.Offset = offset;
            this.Length = length;
        }
    }

    /// <summary>
    /// CommDeviceがターゲットプログラムの生存確認の機能を持つ場合のインターフェース。
    /// </summary>
    public interface ICheckAlive
    {
        /// <summary>
        /// ターゲットプログラムが停止したと判断する上限の時間[秒]。
        /// 0 の場合はタイムアウトによる生存判定は無効。
        /// </summary>
        uint CheckAliveTimeout
        {
            get;
            set;
        }

        /// <summary>
        /// 生存確認によってターゲットとの接続が切断された場合は true を返します。
        /// Router のポーリングで ICommDevice.IsTargetConnect が false になった場合に参照されます。
        /// </summary>
        bool IsDisconnectByCheckAlive
        {
            get;
        }
    }
}
