﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "MultiControllerManager.h"

namespace
{
    const ControllerDeviceType ControllerDefaultDeviceType =
        ControllerDeviceType_SwitchProController;

    const ControllerInterfaceType ControllerDefaultInterfaceType =
        ControllerInterfaceType::ControllerInterfaceType_Bluetooth;
    const int ControllerDefaultMainColor[3] = {130,130,130};
    const int ControllerDefaultSubColor[3] = {15,15,15};

    const ControllerDeviceInfo DefaultControllerDeviceInfo = {
        ControllerDefaultDeviceType,
        ControllerDefaultInterfaceType,
        ControllerDefaultMainColor[0], ControllerDefaultMainColor[1], ControllerDefaultMainColor[2],
        ControllerDefaultSubColor[0],ControllerDefaultSubColor[1],ControllerDefaultSubColor[2]
    };

    enum ControllerId
    {
        ControllerId_ControllerId01 = 0,
        ControllerId_ControllerId02,
        ControllerId_ControllerId03,
        ControllerId_ControllerId04,
        ControllerId_ControllerId05,
        ControllerId_ControllerId06,
        ControllerId_ControllerId07,
        ControllerId_ControllerId08,
        ControllerId_VirtualControllerId01,
        ControllerId_VirtualControllerId02,
        ControllerId_VirtualControllerId03,
        ControllerId_VirtualControllerId04,
        ControllerId_VirtualControllerId05,
        ControllerId_VirtualControllerId06,
        ControllerId_VirtualControllerId07,
        ControllerId_VirtualControllerId08,
        ControllerId_DebugControllerId01,
    };

    // コントローラー数
    const int ControllerMinId = ControllerId_ControllerId01;
    const int ControllerMaxId = ControllerId_ControllerId08;

    // デバッグコントローラー数
    const int DebugControllerMinId = ControllerId_DebugControllerId01;
    const int DebugControllerMaxId = ControllerId_DebugControllerId01;

    // 仮想コントローラー数
    const int VirtualControllerMinId = ControllerId_VirtualControllerId01;
    const int VirtualControllerMaxId = ControllerId_VirtualControllerId08;

    const int ConnectableControllerCount = 8;
}


MultiControllerManager& MultiControllerManager::GetInstance(IoPortSender *sender)
{
    static MultiControllerManager instance(sender);
    return instance;
}

MultiControllerManager::MultiControllerManager(IoPortSender *sender)
{
    m_Sender = sender;
    for (int i = 0; i < ControllerTotalCount; i++)
    {
        m_ControllerInterfaces[i] = nullptr;
    }
}

MultiControllerManager::~MultiControllerManager()
{
    ClearAllControllerState();
}

void MultiControllerManager::InitializeController()
{
    std::vector<HidShellBasicXpadState> controllerStates;
    m_Sender->GetXpadStates(controllerStates);

    int controllerCount = 0;
    for (auto it = controllerStates.begin(); it != controllerStates.end(); it++)
    {
        int controllerId = ControllerMinId + controllerCount;
        if (it->powerState != HidShellXpadPowerState::Disconnected)
        {
            AddController(controllerId);
        }

        controllerCount++;
        if (controllerCount >= HidShellBasicXpadCountMax)
        {
            break;
        }
    }
}

void MultiControllerManager::InitializeDebugController()
{
    HidShellDebugPadState debugPadState;
    m_Sender->GetDebugPadStates(debugPadState);

    if (debugPadState.attributes)
    {
        int controllerId;
        AddDebugController(&controllerId);
    }
}

void MultiControllerManager::InitializeVirtualController()
{
    std::vector<HidShellAbstractedPadState> virtualControllerStates;
    m_Sender->GetVirtualPadStates(virtualControllerStates);

    int virtualControllerCount = 0;
    int controllerId = VirtualControllerMinId;
    for (auto it = virtualControllerStates.begin(); it != virtualControllerStates.end(); it++)
    {
        if (it->attribute & AbstractedPadAttributeMask::AbstractedPadAttribute_IsConnected)
        {
            // デバイス情報
            ControllerDeviceType deviceType = ControllerDefaultDeviceType;
            ControllerInterfaceType interfaceType = ControllerDefaultInterfaceType;
            switch (it->deviceType)
            {
                case AbstractedPadDeviceType::AbstractedPadDeviceType_SwitchProController:
                    deviceType = ControllerDeviceType::ControllerDeviceType_SwitchProController;
                    break;
                case AbstractedPadDeviceType::AbstractedPadDeviceType_HandheldJoyLeft:
                case AbstractedPadDeviceType::AbstractedPadDeviceType_JoyConLeft:
                    deviceType = ControllerDeviceType::ControllerDeviceType_JoyConLeft;
                    break;
                case AbstractedPadDeviceType::AbstractedPadDeviceType_HandheldJoyRight:
                case AbstractedPadDeviceType::AbstractedPadDeviceType_JoyConRight:
                    deviceType = ControllerDeviceType::ControllerDeviceType_JoyConRight;
                    break;
                default:
                    break;
            }

            switch (it->interfaceType)
            {
                case AbstractedPadInterfaceType::AbstractedPadInterfaceType_Bluetooth:
                    interfaceType = ControllerInterfaceType::ControllerInterfaceType_Bluetooth;
                    break;
                case AbstractedPadInterfaceType::AbstractedPadInterfaceType_Rail:
                    interfaceType = ControllerInterfaceType::ControllerInterfaceType_Rail;
                    break;
                case AbstractedPadInterfaceType::AbstractedPadInterfaceType_Usb:
                    // TIPS: USB 接続のデバイスは、Bluetooth 接続のデバイスとして扱う。
                    interfaceType = ControllerInterfaceType::ControllerInterfaceType_Bluetooth;
                    break;
                default:
                    break;
            }

            // 接続
            ControllerDeviceInfo deviceInfo;
            deviceInfo.deviceType    = deviceType;
            deviceInfo.interfaceType = interfaceType;
            deviceInfo.mainColor[0]  = it->color.main[0];
            deviceInfo.mainColor[1]  = it->color.main[1];
            deviceInfo.mainColor[2]  = it->color.main[2];
            deviceInfo.subColor[0]   = it->color.sub[0];
            deviceInfo.subColor[1]   = it->color.sub[1];
            deviceInfo.subColor[2]   = it->color.sub[2];
            AddControllerWithDeviceInfo(controllerId, deviceInfo);
        }
        controllerId++;
        virtualControllerCount++;
        if (virtualControllerCount >= HidShellAbstractedPadCountMax)
        {
            break;
        }
    }
}

void MultiControllerManager::InitializeControllerConnection()
{
    InitializeController();
    InitializeDebugController();
    InitializeVirtualController();

    for (int i = 0; i < ControllerTotalCount; i++)
    {
        if (m_ControllerInterfaces[i] != nullptr)
        {
            m_ControllerInterfaces[i]->SetManagedStatus(false);
        }
    }
}


void MultiControllerManager::CheckActiveControllerIdRange(int controllerId)
{
    // コントローラー
    if (controllerId >= ControllerMinId && controllerId <= ControllerMaxId)
    {
        return;
    }

    // デバッグコントローラー
    if (controllerId >= DebugControllerMinId && controllerId <= DebugControllerMaxId)
    {
        return;
    }

    // 仮想コントローラー
    if (controllerId >= VirtualControllerMinId && controllerId <= VirtualControllerMaxId)
    {
        return;
    }

    throw InvalidControllerIdException(controllerId);
}

void MultiControllerManager::CheckActiveControllerId(int controllerId)
{
    CheckActiveControllerIdRange(controllerId);
    if (m_ControllerInterfaces[controllerId] == nullptr)
    {
        throw InvalidControllerIdException(controllerId);
    }
}

void MultiControllerManager::CheckControllerDeviceInfo(const ControllerDeviceInfo& deviceInfo)
{
    // 組み合わせチェック
    switch (deviceInfo.deviceType)
    {
        case ControllerDeviceType::ControllerDeviceType_SwitchProController:
            if (deviceInfo.interfaceType != ControllerInterfaceType::ControllerInterfaceType_Bluetooth)
            {
                throw InvalidControllerDeviceInofException(deviceInfo.deviceType, deviceInfo.interfaceType);
            }
            break;
        case ControllerDeviceType::ControllerDeviceType_JoyConLeft:
        case ControllerDeviceType::ControllerDeviceType_JoyConRight:
            if (   deviceInfo.interfaceType != ControllerInterfaceType::ControllerInterfaceType_Bluetooth
                && deviceInfo.interfaceType != ControllerInterfaceType::ControllerInterfaceType_Rail)
            {
                throw InvalidControllerDeviceInofException(deviceInfo.deviceType, deviceInfo.interfaceType);
            }
            break;
        default:
            throw InvalidControllerDeviceInofException(deviceInfo.deviceType, deviceInfo.interfaceType);
    }
}

// コントローラー接続
ControllerResult MultiControllerManager::AddController(int *pOutControllerId )
{
    try
    {
        int controllerCount = 0;
        GetConnectedControllerCount(&controllerCount);
        if (controllerCount >= ConnectableControllerCount)
        {
            return ControllerResult::ControllerResult_ControllerCountOver;
        }

        return AddControllerWithDeviceInfo(pOutControllerId, DefaultControllerDeviceInfo);
    }
    catch (...)
    {
        ERRMSG("MultiControllerManager::AddController: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return ControllerResult::ControllerResult_Unexpected;
    }

    return ControllerResult::ControllerResult_ConnectControllerFailed;
}

ControllerResult MultiControllerManager::AddController(int controllerId)
{
    try
    {
        int controllerCount = 0;
        GetConnectedControllerCount(&controllerCount);
        if (controllerCount >= ConnectableControllerCount)
        {
            return ControllerResult::ControllerResult_ControllerCountOver;
        }

        // xpad 以外の範囲化をチェック
        if (controllerId < ControllerMinId || controllerId > ControllerMaxId)
        {
            throw InvalidControllerIdException(controllerId);
        }

        if (m_ControllerInterfaces[controllerId] != nullptr)
        {
            ERRMSG("MultiControllerManager::AddController: Connection is executed to connected controller.");
            return ControllerResult::ControllerResult_ConnectControllerFailed;
        }

        m_ControllerInterfaces[controllerId] = new HacController(m_Sender, controllerId);
        m_ControllerInterfaces[controllerId]->Begin();
    }
    catch (InvalidButtonException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::AddController: ") + exception.what());
        return ControllerResult::ControllerResult_ConnectControllerFailed;
    }
    catch (InvalidControllerIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::AddController: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidControllerId;
    }
    catch (...)
    {
        ERRMSG("MultiControllerManager::AddController: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return ControllerResult::ControllerResult_Unexpected;
    }
    return ControllerResult::ControllerResult_Success;
}


ControllerResult MultiControllerManager::AddDebugController( int *pOutControllerId  )
{
    if (m_ControllerInterfaces[ControllerId_DebugControllerId01] != nullptr)
    {
        ERRMSG("MultiControllerManager::AddDebugController: Connection is executed to connected debug pad.");
        return ControllerResult::ControllerResult_ConnectControllerFailed;
    }

    try{
        m_ControllerInterfaces[ControllerId_DebugControllerId01] = new HacDebugController(m_Sender);
        m_ControllerInterfaces[ControllerId_DebugControllerId01]->Begin();

        (*pOutControllerId) = ControllerId_DebugControllerId01;
    }
    catch (...)
    {
        ERRMSG("MultiControllerManager::AddDebugController: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return ControllerResult::ControllerResult_Unexpected;
    }

    return ControllerResult::ControllerResult_Success;
}

ControllerResult MultiControllerManager::AddControllerWithDeviceInfo(int *pOutControllerId, const ControllerDeviceInfo& deviceInfo)
{
    try
    {
        int controllerCount;
        GetConnectedControllerCount(&controllerCount);
        // コントローラー接続最大数チェック
        if (controllerCount >= ConnectableControllerCount)
        {
            return ControllerResult::ControllerResult_ControllerCountOver;
        }
        // 本体レール接続チェック
        if (deviceInfo.interfaceType == ControllerInterfaceType::ControllerInterfaceType_Rail)
        {
            for (int i = VirtualControllerMinId; i <= VirtualControllerMaxId; i++)
            {
                int controllerNumber = i - VirtualControllerMinId;
                if (m_ControllerInterfaces[i] == nullptr)
                {
                    continue;
                }

                if (m_VirtualControllerDeviceInfo[controllerNumber].interfaceType != ControllerInterfaceType::ControllerInterfaceType_Rail)
                {
                    continue;
                }

                // 既に接続されている場合はエラーを返す
                if (deviceInfo.deviceType == m_VirtualControllerDeviceInfo[controllerNumber].deviceType)
                {
                    return ControllerResult::ControllerResult_ControllerCountOver;
                }
            }
        }

        for (int i = VirtualControllerMinId; i <= VirtualControllerMaxId; i++)
        {
            if (m_ControllerInterfaces[i] != nullptr)
            {
                continue;
            }
            // デバイス情報チェック
            CheckControllerDeviceInfo(deviceInfo);

            // コントローラー接続
            AddControllerWithDeviceInfo(i, deviceInfo);
            (*pOutControllerId ) = i;
            return ControllerResult::ControllerResult_Success;
        }
    }
    catch (InvalidControllerIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::AddControllerWithDeviceInfo: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidControllerId;

    }
    catch (InvalidControllerDeviceInofException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::AddControllerWithDeviceInfo: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidDeviceInfo;
    }
    catch (...)
    {
        ERRMSG("MultiControllerManager::AddControllerWithDeviceInfo: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return ControllerResult::ControllerResult_Unexpected;
    }

    return ControllerResult::ControllerResult_ConnectControllerFailed;
}

ControllerResult MultiControllerManager::AddControllerWithDeviceInfo(
    int controllerId, const ControllerDeviceInfo& deviceInfo)
{
    int controllerNumber = controllerId - VirtualControllerMinId;
    // コントローラー初期化
    ControllerColor mainColor = {deviceInfo.mainColor[0], deviceInfo.mainColor[1], deviceInfo.mainColor[2]};
    ControllerColor subColor  = {deviceInfo.subColor[0],  deviceInfo.subColor[1],  deviceInfo.subColor[2]};
    m_ControllerInterfaces[controllerId] = new HacVirtualController(m_Sender, controllerNumber,
        deviceInfo.deviceType,
        deviceInfo.interfaceType,
        mainColor,
        subColor );

    m_VirtualControllerDeviceInfo[controllerNumber] = deviceInfo;

    // コントローラー接続
    m_ControllerInterfaces[controllerId]->Begin();

    return ControllerResult::ControllerResult_Success;
}

ControllerResult MultiControllerManager::ReleaseController(int controllerId)
{
    try
    {
        CheckActiveControllerId(controllerId);

        m_ControllerInterfaces[controllerId]->Close();
        delete m_ControllerInterfaces[controllerId];
        m_ControllerInterfaces[controllerId] = nullptr;
    }
    catch (InvalidControllerIdException &exception)
    {
        (void)exception;
        ERRMSG("MultiControllerManager::ReleaseController: " + exception.what() );
        return ControllerResult::ControllerResult_InvalidControllerId;
    }
    catch (...)
    {
        ERRMSG("MultiControllerManager::ReleaseController: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return ControllerResult::ControllerResult_Unexpected;
    }

    return ControllerResult::ControllerResult_Success;
}

ControllerResult MultiControllerManager::GetConnectedControllerCount(int *pOutCount)
{
    int count = 0;
    for (int i = 0; i < ControllerTotalCount; i++)
    {
        if (i >= DebugControllerMinId && i <= DebugControllerMaxId)
        {
            continue;
        }

        if (m_ControllerInterfaces[i] != nullptr)
        {
            count++;
        }
    }
    (*pOutCount) = count;
    return ControllerResult::ControllerResult_Success;
}

ControllerResult MultiControllerManager::GetConnectedControllerIds(int *pOutCount,
    int pOutIdsArray[], int arrayLength)
{
    int count = 0;
    for (int i = 0; i < ControllerTotalCount; i++)
    {
        if (count >= arrayLength)
        {
            break;
        }
        if (i >= DebugControllerMinId && i <= DebugControllerMaxId)
        {
            continue;
        }
        if (m_ControllerInterfaces[i] != nullptr)
        {
            pOutIdsArray[count] = i;
            count++;
        }
    }
    (*pOutCount) = count;

    return ControllerResult::ControllerResult_Success;
}

ControllerResult MultiControllerManager::GetConnectedControllerDeviceInfo(
    ControllerDeviceInfo *pOutDeviceInfo,
    int controllerId)
{
    try
    {
        CheckActiveControllerId(controllerId);
        if (controllerId >= ControllerMinId && controllerId <= ControllerMaxId)
        {
            (*pOutDeviceInfo) = DefaultControllerDeviceInfo;
        }
        else if (controllerId >= DebugControllerMinId && controllerId <= DebugControllerMaxId)
        {
            return ControllerResult::ControllerResult_InvalidControllerId;
        }
        else if(controllerId >= VirtualControllerMinId && controllerId <= VirtualControllerMaxId)
        {
            int controllerNumber = controllerId - VirtualControllerMinId;
            (*pOutDeviceInfo) = m_VirtualControllerDeviceInfo[controllerNumber];
        }
        else
        {
            return ControllerResult::ControllerResult_InvalidControllerId;
        }
    }
    catch (InvalidControllerIdException &exception)
    {
        (void)exception;
        ERRMSG("MultiControllerManager::ReleaseController: " + exception.what() );
        return ControllerResult::ControllerResult_InvalidControllerId;
    }
    catch (...)
    {
        return ControllerResult::ControllerResult_Unexpected;
    }
    return ControllerResult::ControllerResult_Success;
}

ControllerResult MultiControllerManager::GetConnectedDebugControllerCount(int *pOutCount)
{
    int count = 0;
    for (int i = DebugControllerMinId; i <= DebugControllerMaxId; i++)
    {
        if (m_ControllerInterfaces[i] != nullptr)
        {
            count++;
        }
    }
    (*pOutCount) = count;
    return ControllerResult::ControllerResult_Success;
}

ControllerResult MultiControllerManager::GetConnectedDebugControllerIds(int *pOutCount,
    int pOutIdsArray[], int arrayLength)
{
    int count = 0;
    for (int i = DebugControllerMinId; i <= DebugControllerMaxId; i++)
    {
        if (count >= arrayLength)
        {
            break;
        }

        if (m_ControllerInterfaces[i] != nullptr)
        {
            pOutIdsArray[count] = i;
            count++;
        }
    }
    (*pOutCount) = count;
    return ControllerResult::ControllerResult_Success;
}

ControllerResult MultiControllerManager::ClearAllControllerState()
{
    try
    {
        for (int i = 0; i < ControllerTotalCount; i++)
        {
            if (m_ControllerInterfaces[i] == nullptr)
            {
                continue;
            }

            delete m_ControllerInterfaces[i];
            m_ControllerInterfaces[i] = nullptr;
        }
    }
    catch (...)
    {
        ERRMSG("MultiControllerManager::ClearAllControllerState: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return ControllerResult::ControllerResult_Unexpected;
    }

    return ControllerResult::ControllerResult_Success;
}

// ボタン操作
ControllerResult MultiControllerManager::PressButton(int controllerId, Button button)
{
    try
    {
        CheckActiveControllerId(controllerId);

        m_ControllerInterfaces[controllerId]->Press(button);
    }
    catch (InvalidButtonException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::PressButton: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidButton;
    }
    catch (InvalidControllerIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::PressButton: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidControllerId;
    }
    catch (...)
    {
        ERRMSG("MultiControllerManager::PressButton: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return ControllerResult::ControllerResult_Unexpected;
    }

    return ControllerResult::ControllerResult_Success;
}

ControllerResult MultiControllerManager::ReleaseButton(int controllerId, Button button)
{
    try
    {
        CheckActiveControllerId(controllerId);

        m_ControllerInterfaces[controllerId]->Release(button);
    }
    catch (InvalidButtonException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::ReleaseButton: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidButton;
    }
    catch (InvalidControllerIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::ReleaseButton: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidControllerId;
    }
    catch (...)
    {
        ERRMSG("MultiControllerManager::ReleaseButton: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return ControllerResult::ControllerResult_Unexpected;
    }

    return ControllerResult::ControllerResult_Success;
}
ControllerResult MultiControllerManager::PushButton(int controllerId, Button button, int waitPress)
{
    try
    {
        CheckActiveControllerId(controllerId);

        m_ControllerInterfaces[controllerId]->Push(button, waitPress);
    }
    catch (InvalidButtonException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::PushButton: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidButton;
    }
    catch (InvalidControllerIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::PushButton: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidControllerId;
    }
    catch (...)
    {
        ERRMSG("MultiControllerManager::PushButton: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return ControllerResult::ControllerResult_Unexpected;
    }

    return ControllerResult::ControllerResult_Success;

}


ControllerResult MultiControllerManager::PushMultiSeq(int controllerId, Button buttonList[], int buttonCount, int waitPress, int waitSpan)
{
    try
    {
        CheckActiveControllerId(controllerId);

        for (int i = 0; i < buttonCount; i++)
        {
            ControllerResult result = PushButton(controllerId, buttonList[i], waitPress);
            if ( result != ControllerResult::ControllerResult_Success)
            {
                return result;
            }

            if (i != (buttonCount - 1))
            {
                // 最後のボタン押下後は、span を挟まない
                Sleep(waitSpan);
            }
        }
    }
    catch (InvalidButtonException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::PushMultiSeq: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidButton;
    }
    catch (InvalidControllerIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::PushMultiSeq: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidControllerId;
    }
    catch (...)
    {
        ERRMSG("MultiControllerManager::PushMultiSeq: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return ControllerResult::ControllerResult_Unexpected;
    }


    return ControllerResult::ControllerResult_Success;
}


ControllerResult MultiControllerManager::ResetControllerState(int controllerId)
{
    try
    {
        CheckActiveControllerId(controllerId);

        m_ControllerInterfaces[controllerId]->SetDefault();
    }
    catch (InvalidButtonException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::ResetControllerState: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidButton;
    }
    catch (InvalidControllerIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::ResetControllerState: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidControllerId;
    }
    catch (...)
    {
        ERRMSG("MultiControllerManager::ResetControllerState: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return ControllerResult::ControllerResult_Unexpected;
    }

    return ControllerResult::ControllerResult_Success;
}

ControllerResult MultiControllerManager::SetControllerState(int controllerId, Button buttonList[], int buttonCount, Stick stickList[], float stickDegreeList[], float stickPowerList[], int stickCount)
{
    try
    {
        CheckActiveControllerId(controllerId);

        m_ControllerInterfaces[controllerId]->SetState(buttonList, buttonCount, stickList, stickDegreeList, stickPowerList, stickCount);
    }
    catch (InvalidButtonException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::SetControllerState: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidButton;
    }
    catch (InvalidControllerIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::SetControllerState: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidControllerId;
    }
    catch (InvalidStickException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::SetControllerState: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidStick;
    }
    catch (...)
    {
        ERRMSG("MultiControllerManager::SetControllerState: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return ControllerResult::ControllerResult_Unexpected;
    }

    return ControllerResult::ControllerResult_Success;

}

// 複数ボタン操作
ControllerResult MultiControllerManager::PressMultiButton(int controllerId, Button buttonList[], int buttonCount)
{
    try
    {
        CheckActiveControllerId(controllerId);

        m_ControllerInterfaces[controllerId]->PressMulti(buttonList, buttonCount);
    }
    catch (InvalidButtonException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::PressMultiButton: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidButton;
    }
    catch (InvalidControllerIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::PressMultiButton: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidControllerId;
    }
    catch (...)
    {
        ERRMSG("MultiControllerManager::PressMultiButton: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return ControllerResult::ControllerResult_Unexpected;
    }

    return ControllerResult::ControllerResult_Success;
}
ControllerResult MultiControllerManager::ReleaseMultiButton(int controllerId, Button buttonList[], int buttonCount)
{
    try
    {
        CheckActiveControllerId(controllerId);

        m_ControllerInterfaces[controllerId]->ReleaseMulti(buttonList, buttonCount);
    }
    catch (InvalidButtonException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::ReleaseMultiButton: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidButton;
    }
    catch (InvalidControllerIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::ReleaseMultiButton: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidControllerId;
    }
    catch (...)
    {
        ERRMSG("MultiControllerManager::ReleaseMultiButton: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return ControllerResult::ControllerResult_Unexpected;
    }

    return ControllerResult::ControllerResult_Success;
}
ControllerResult MultiControllerManager::PushMultiButton(int controllerId, Button buttonList[], int buttonCount, int waitPress)
{
    try
    {
        CheckActiveControllerId(controllerId);

        m_ControllerInterfaces[controllerId]->PushMulti(buttonList, buttonCount, waitPress);
    }
    catch (InvalidButtonException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::PushMultiButton: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidButton;
    }
    catch (InvalidControllerIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::PushMultiButton: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidControllerId;
    }
    catch (...)
    {
        ERRMSG("MultiControllerManager::PushMultiButton: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return ControllerResult::ControllerResult_Unexpected;
    }

    return ControllerResult::ControllerResult_Success;
}


// アナログスティック
ControllerResult MultiControllerManager::HoldAnalogStick(int controllerId, Stick stick, float degree, float power)
{
    try
    {
        CheckActiveControllerId(controllerId);

        m_ControllerInterfaces[controllerId]->HoldAnalogStick(stick, degree, power);
    }
    catch (InvalidStickException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::HoldAnalogStick: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidStick;
    }
    catch (InvalidControllerIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::HoldAnalogStick: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidControllerId;
    }
    catch (...)
    {
        ERRMSG("MultiControllerManager::HoldAnalogStick: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return ControllerResult::ControllerResult_Unexpected;
    }

    return ControllerResult::ControllerResult_Success;

}
ControllerResult MultiControllerManager::ReleaseAnalogStick(int controllerId, Stick stick)
{
    try
    {
        CheckActiveControllerId(controllerId);
        m_ControllerInterfaces[controllerId]->ReleaseAnalogStick(stick);
    }
    catch (InvalidStickException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::ReleaseAnalogStick: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidStick;
    }
    catch (InvalidControllerIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::ReleaseAnalogStick: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidControllerId;
    }
    catch (...)
    {
        ERRMSG("MultiControllerManager::ReleaseAnalogStick: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return ControllerResult::ControllerResult_Unexpected;
    }

    return ControllerResult::ControllerResult_Success;
}
ControllerResult MultiControllerManager::TiltAnalogStick(int controllerId, Stick stick, float degree, float power, int waitHold)
{
    try
    {
        CheckActiveControllerId(controllerId);
        m_ControllerInterfaces[controllerId]->TiltAnalogStick(stick, degree, power, waitHold);
    }
    catch (InvalidStickException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::TiltAnalogStick: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidStick;
    }
    catch (InvalidControllerIdException &exception)
    {
        (void)exception;
        ERRMSG(std::string("MultiControllerManager::TiltAnalogStick: ") + exception.what());
        return ControllerResult::ControllerResult_InvalidControllerId;
    }
    catch (...)
    {
        ERRMSG("MultiControllerManager::TiltAnalogStick: " + UNEXPECTED_EXCEPTION_MESSAGE);
        return ControllerResult::ControllerResult_Unexpected;
    }

    return ControllerResult::ControllerResult_Success;
}
