﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DebugPadInputer.h"

namespace
{
    const int g_DebugPadButtonMaxValue = 15;
    const int g_DebugPadButtonMinValue = 0;
}

DebugPadInputer::DebugPadInputer(IoPortSender* sender, int controllerId) : m_Sender(sender), m_ControllerId(controllerId)
{
}

DebugPadInputer::~DebugPadInputer()
{
}

void DebugPadInputer::CheckButton(Button type)
{
    if (type < g_DebugPadButtonMinValue || type > g_DebugPadButtonMaxValue)
    {
        throw InvalidButtonException(type);
    }
}

void DebugPadInputer::CheckStick(Stick type)
{
    if (type < 0 || type >= Stick::Stick_MaxCount)
    {
        throw InvalidStickException(type);
    }

}

ControllerButtonValueType DebugPadInputer::Button2Flag(Button button)
{
    switch (button)
    {
        case Button::Button_A:
            return static_cast<ControllerButtonValueType>(1 << 0);
        case Button::Button_B:
            return static_cast<ControllerButtonValueType>(1 << 1);
        case Button::Button_X:
            return static_cast<ControllerButtonValueType>(1 << 2);
        case Button::Button_Y:
            return static_cast<ControllerButtonValueType>(1 << 3);
        case Button::Button_L:
            return static_cast<ControllerButtonValueType>(1 << 4);
        case Button::Button_R:
            return static_cast<ControllerButtonValueType>(1 << 5);
        case Button::Button_ZL:
            return static_cast<ControllerButtonValueType>(1 << 6);
        case Button::Button_ZR:
            return static_cast<ControllerButtonValueType>(1 << 7);
        case Button::Button_Plus:
            return static_cast<ControllerButtonValueType>(1 << 8);
        case Button::Button_Minus:
            return static_cast<ControllerButtonValueType>(1 << 9);
        case Button::Button_Left:
            return static_cast<ControllerButtonValueType>(1 << 10);
        case Button::Button_Up:
            return static_cast<ControllerButtonValueType>(1 << 11);
        case Button::Button_Right:
            return static_cast<ControllerButtonValueType>(1 << 12);
        case Button::Button_Down:
            return static_cast<ControllerButtonValueType>(1 << 13);
        default:
            throw InvalidButtonException(button);
    }

    return 0;
}

void DebugPadInputer::BeginAutoPilot()
{
    m_Sender->SetDebugPadConnection(1);
}

void DebugPadInputer::EndAutoPilot()
{
    m_Sender->SetDebugPadConnection(0);
}

void DebugPadInputer::SetDefault()
{
    m_Sender->SetDebugPadDefault();
}

void DebugPadInputer::SetState(const Button buttonList[], int buttonCount,
    const Stick stickList[], const ControllerStickValueType stickXList[], const ControllerStickValueType stickYList[], int stickCount)
{
    ControllerButtonValueType buttons = static_cast<ControllerButtonValueType>(0);
    for (int i = 0; i < buttonCount; i++)
    {
        CheckButton(buttonList[i]);
        buttons |= Button2Flag(buttonList[i]);
    }
    for (int i = 0; i < stickCount; i++)
    {
        CheckStick(stickList[i]);
    }

    m_Sender->SetDebugPadState(buttons, stickList, stickXList, stickYList, stickCount);
}

void DebugPadInputer::Press(Button button)
{
    if (button == Button::Button_Home)
    {
        m_Sender->SetHomeButton(m_ControllerId, 1);
    }
    else
    {
        CheckButton(button);
        m_Sender->SetDebugPadPressButton(Button2Flag(button));
    }
}

void DebugPadInputer::Release(Button button)
{
    if (button == Button::Button_Home)
    {
        m_Sender->SetHomeButton(m_ControllerId, 0);
    }
    else
    {
        CheckButton(button);
        m_Sender->SetDebugPadReleaseButton(Button2Flag(button));
    }
}

void DebugPadInputer::SetStickPos(Stick stick, ControllerStickValueType x, ControllerStickValueType y)
{
    CheckStick(stick);
    switch (stick)
    {
        case Stick::Stick_L:
            SetLStickPos(x, y);
            break;
        case Stick::Stick_R:
            SetRStickPos(x, y);
            break;
        default:
            break;
    }
}

void DebugPadInputer::SetLStickPos(ControllerStickValueType x, ControllerStickValueType y)
{
    m_Sender->SetDebugPadStickL(x, y);
}

void DebugPadInputer::SetRStickPos(ControllerStickValueType x, ControllerStickValueType y)
{
    m_Sender->SetDebugPadStickR(x, y);
}
