﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <chrono>
#include "common.h"
#include "Common\autoTestAssistTool_FpsController.h"
#include "WindowManager.h"

namespace
{
    const char  WindowClassname[] = "AutoTestAssistTool";
    const DWORD WindowStyle       = WS_OVERLAPPEDWINDOW & (~WS_MAXIMIZEBOX);

    const int   MessageTimeoutTime = 10000;
    const int   MessageWaitSpan    = 50;

    const unsigned int RestoreWindowSizeEvent = 1;
    const char*        RestoreWindowSizeText  = "Restore window size";

    const PIXELFORMATDESCRIPTOR PixelFormatDescriptor = {
            sizeof(PIXELFORMATDESCRIPTOR),
            1,
            PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER,     //Flags
            PFD_TYPE_RGBA,                                                  //The kind of framebuffer. RGBA or palette.
            32,                                                             //Colordepth of the framebuffer.
            0, 0, 0, 0, 0, 0,
            0,
            0,
            0,
            0, 0, 0, 0,
            24,                                                             //Number of bits for the depthbuffer
            8,                                                              //Number of bits for the stencilbuffer
            0,                                                              //Number of Aux buffers in the framebuffer.
            PFD_MAIN_PLANE,
            0,
            0, 0, 0
    };

    LRESULT CALLBACK WndProc( HWND windowHandle, UINT message, WPARAM wparam, LPARAM lparam ){
        static WindowManager& windowManager = WindowManager::GetInstance();
        static bool isSizing = false;

        switch( message ){
            case WM_DESTROY:
            {
                PostQuitMessage( 0 );
                break;
            }
            case WM_CLOSE:
            {
                windowManager.RemoveWindowList(windowHandle);
                DestroyWindow(windowHandle);
                break;
            }
            case WM_EXITSIZEMOVE:
            {
                if (isSizing)
                {
                    isSizing = false;
                    windowManager.AdjustWindowViewPort(windowHandle);
                }
                break;
            }
            case WM_SIZE:
            case WM_SIZING:
            {
                isSizing = true;
                break;
            }
            case WM_ERASEBKGND:
            {
                // TIPS: WM_ERASEBKGND を DefWindowProc へ渡すと、画面リサイズ時に度々背景が黒塗りされてちらつくため、固定値を返して抑制
                return 1;
            }
            case WM_PAINT:
            {
                windowManager.RedrawWindow(windowHandle);
                return (DefWindowProc(windowHandle, message, wparam, lparam));
            }
            case WM_COMMAND:
            {
                switch (LOWORD(wparam))
                {
                    case RestoreWindowSizeEvent:
                        windowManager.RestoreWindowSize(windowHandle);
                        break;
                    default:
                        return (DefWindowProc( windowHandle, message, wparam, lparam ) );
                }
                break;
            }
            default:
                return (DefWindowProc( windowHandle, message, wparam, lparam ) );
        }

        return 0;
    }
}

WindowManager& WindowManager::GetInstance()
{
    static WindowManager instance;
    return instance;
}

WindowManager::WindowManager()
    : m_OpenWindowRequest()
    , m_OpenWindowResult()
    , m_CloseWindowRequest()
    , m_CloseWindowResult()
    , m_IsActiveMessageThread(false)
    , m_ProcessMessageThreadObject(nullptr)
{
    RegisterWindowClass();
    ProcessMessageThreadBegin();
}

WindowManager::~WindowManager()
{
    {
        std::lock_guard<std::mutex> lock(m_WindowListMutex);
        for (auto it = m_WindowList.begin(); it != m_WindowList.end(); it++)
        {
            DestroyWindow(it->hwnd);
            glDeleteTextures(1, &it->textureId);
        }
        m_WindowList.clear();
    }
    ProcessMessageThreadEnd();
}

bool WindowManager::OpenImageWindow(const char* windowName, const cv::Mat& image, int imageWidth, int imageHeight, bool isUseMessage )
{
    // 既に開いている場合は、画像更新のみを行う
    {
        std::lock_guard<std::mutex> lock(m_WindowListMutex);
        for (auto it = m_WindowList.begin(); it != m_WindowList.end(); it++)
        {
            if (it->name != windowName)
            {
                continue;
            }

            UpdateWindowInfoTexture((*it), image, imageWidth, imageHeight, true);
            DrawWindowInfo((*it));
            return true;
        }
    }

    int width = imageWidth;
    int height = imageHeight;
    if (isUseMessage)
    {
        std::lock_guard<std::mutex> lockOpenWindow(m_OpenWindowMutex);

        // ウィンドウ作成メッセージ
        {
            std::lock_guard<std::mutex> lock(m_OpenWindowMessageMutex);
            OpenWindowRequestMessage request{
                windowName,
                width,
                height,
                true
            };
            m_OpenWindowRequest = request;
            m_OpenWindowResult.isEnableResult = false;
        }

        // ウィンドウ作成結果同期
        std::chrono::system_clock::time_point beginTime = std::chrono::system_clock::now();
        std::chrono::system_clock::time_point nowTime;
        int durationTime = 0;
        do
        {
            Sleep(MessageWaitSpan);
            nowTime = std::chrono::system_clock::now();
            durationTime = static_cast<unsigned int>(std::chrono::duration_cast<std::chrono::milliseconds>(nowTime - beginTime).count());

        } while ((m_OpenWindowRequest.isEnableRequest || !(m_OpenWindowResult.isEnableResult)) && durationTime < MessageTimeoutTime);


        // メッセージの処理が完了していない場合(タイムアウトした場合)
        if (m_OpenWindowRequest.isEnableRequest || !(m_OpenWindowResult.isEnableResult))
        {
            throw std::exception("CreateWindow is failed");
        }
    }
    else
    {
        ProcessOpenWindowMessageHelper(windowName, width, height);
    }
    // 結果メッセージ受け取り完了
    m_OpenWindowResult.isEnableResult = false;

    // ウィンドウ作成失敗
    if (!(m_OpenWindowResult.result))
    {
        return false;
    }

    // ウィンドウ情報取得
    WindowInfo info;
    info.name = windowName;
    info.hwnd = m_OpenWindowResult.hwnd;
    info.imageHeight  = height;
    info.imageWidth   = width;

    RECT windowClientRect;
    GetClientRect(info.hwnd, &windowClientRect);
    info.windowClientWidth  = windowClientRect.right  - windowClientRect.left;
    info.windowClientHeight = windowClientRect.bottom - windowClientRect.top;

    info.hdc = GetDC(info.hwnd);
    if (info.hdc == NULL)
    {
        return false;
    }

    int selectedPixelFormat = ChoosePixelFormat(info.hdc, &PixelFormatDescriptor);
    SetPixelFormat(info.hdc, selectedPixelFormat, &PixelFormatDescriptor);

    info.hglrc = wglCreateContext(info.hdc);
    if (info.hglrc == NULL)
    {
        return false;
    }

    {
        // テクスチャ生成・描画
        CreateTexture(&(info.textureId), info.hdc, info.hglrc, image);
        DrawWindowInfo(info);

        // リストへ追加
        std::lock_guard<std::mutex> lock(m_WindowListMutex);
        m_WindowList.push_back(info);
    }

    return true;
}

void WindowManager::UpdateWindowImage(const char* windowName, const cv::Mat& image, int imageWidth, int imageHeight, bool isChangeWindowSize)
{
    {
        std::lock_guard<std::mutex> lock(m_WindowListMutex);
        for (auto it = m_WindowList.begin(); it != m_WindowList.end(); it++)
        {
            if (it->name != std::string(windowName))
            {
                continue;
            }
            if (!(IsWindow(it->hwnd)))
            {
                return;
            }

            UpdateWindowInfoTexture((*it), image, imageWidth, imageHeight, isChangeWindowSize);
            DrawWindowInfo((*it));
            return;
        }
    }

    // ウィンドウが無い場合は、再オープン
    OpenImageWindow(windowName, image, imageWidth, imageHeight, true);
}

void WindowManager::UpdateWindowInfoTexture(WindowInfo& windowInfo, const cv::Mat& image, int imageWidth, int imageHeight, bool isChangeWindowSize)
{
    wglMakeCurrent(windowInfo.hdc, windowInfo.hglrc);
    glBindTexture(GL_TEXTURE_RECTANGLE_EXT, windowInfo.textureId);
    glTexImage2D(GL_TEXTURE_RECTANGLE_EXT, 0, GL_RGBA, image.cols, image.rows, 0, GL_BGR_EXT, GL_UNSIGNED_BYTE, image.data);
    wglMakeCurrent(NULL, NULL);

    windowInfo.imageWidth = imageWidth;
    windowInfo.imageHeight = imageHeight;

    if (isChangeWindowSize
        && (windowInfo.windowClientWidth != imageWidth || windowInfo.windowClientHeight != imageHeight))
    {
        PostMessage(windowInfo.hwnd, WM_COMMAND, RestoreWindowSizeEvent, 0);
    }
}

void WindowManager::AdjustWindowInfoAspectRatio(WindowInfo& windowInfo)
{
    RECT rect;
    if (!GetClientRect(windowInfo.hwnd, &rect))
    {
        return;
    }
    int width = rect.right - rect.left;
    int height = rect.bottom - rect.top;

    if (windowInfo.imageWidth > 0 && windowInfo.imageHeight > 0)
    {
        int windowClientWidthDiff  = std::abs(width  - windowInfo.windowClientWidth);
        int windowClientHeightDiff = std::abs(height - windowInfo.windowClientHeight);

        int adjustedWindowClientWidth = width;
        int adjustedWindowClientHeight = height;

        if (!windowClientWidthDiff && !windowClientHeightDiff)
        {
            return;
        }

        if (windowClientWidthDiff > windowClientHeightDiff)
        {
            if (width < windowInfo.imageWidth)
            {
                adjustedWindowClientHeight = static_cast<int>((static_cast<float>(adjustedWindowClientWidth) / windowInfo.imageWidth) * windowInfo.imageHeight);
            }
            else
            {
                adjustedWindowClientWidth  = windowInfo.imageWidth;
                adjustedWindowClientHeight = windowInfo.imageHeight;
            }
        }
        else
        {
            if (height < windowInfo.imageHeight)
            {
                adjustedWindowClientWidth = static_cast<int>((static_cast<float>(adjustedWindowClientHeight) / windowInfo.imageHeight) * windowInfo.imageWidth);
            }
            else
            {
                adjustedWindowClientWidth  = windowInfo.imageWidth;
                adjustedWindowClientHeight = windowInfo.imageHeight;
            }
        }

        if (adjustedWindowClientHeight != height || adjustedWindowClientWidth != width)
        {
            RECT rect = {0, 0, adjustedWindowClientWidth, adjustedWindowClientHeight};
            AdjustWindowRect(&rect, WindowStyle, FALSE);

            SetWindowPos(windowInfo.hwnd,
                NULL,
                0, 0,
                rect.right - rect.left,
                rect.bottom - rect.top,
                SWP_NOMOVE | SWP_NOZORDER);

        }
        RECT windowClientRect;
        GetClientRect(windowInfo.hwnd, &windowClientRect);
        windowInfo.windowClientWidth  = windowClientRect.right  - windowClientRect.left;
        windowInfo.windowClientHeight = windowClientRect.bottom - windowClientRect.top;
    }
}

void WindowManager::DrawWindowInfo(WindowInfo& windowInfo)
{
    RECT rect;
    GetClientRect(windowInfo.hwnd, &rect);

    // ウィンドウクライアント領域
    int clientRectWidth = rect.right - rect.left;
    int clientRectHeight = rect.bottom - rect.top;
    // 描画領域
    int renderRectWidth = static_cast<int>(clientRectHeight * (static_cast<float>(windowInfo.imageWidth) / windowInfo.imageHeight));
    int renderRectHeight = static_cast<int>(clientRectWidth * (static_cast<float>(windowInfo.imageHeight) / windowInfo.imageWidth));

    // 描画領域調整
    int renderRectWidthDiff = renderRectWidth - clientRectWidth;
    int renderRectHeightDiff = renderRectHeight - clientRectHeight;
    if (renderRectWidthDiff || renderRectHeightDiff)
    {
        if( renderRectHeightDiff < renderRectWidthDiff)
        {
            renderRectWidth = static_cast<int>(renderRectHeight * (static_cast<float>(windowInfo.imageWidth) / windowInfo.imageHeight));
        }
        else
        {
            renderRectHeight = static_cast<int>(renderRectWidth * (static_cast<float>(windowInfo.imageHeight) / windowInfo.imageWidth));
        }
    }
    if (renderRectWidth > windowInfo.imageWidth)
    {
        renderRectWidth = windowInfo.imageWidth;
    }
    if (renderRectHeight > windowInfo.imageHeight)
    {
        renderRectHeight = windowInfo.imageHeight;
    }

    // 描画
    wglMakeCurrent(windowInfo.hdc, windowInfo.hglrc);

    glClear(GL_COLOR_BUFFER_BIT);

    glEnable(GL_TEXTURE_2D);
    glEnable(GL_TEXTURE_RECTANGLE_EXT);
    glBindTexture(GL_TEXTURE_RECTANGLE_EXT, windowInfo.textureId);

    // ビューポート指定
    glViewport(0, 0, renderRectWidth, renderRectHeight);

    glBegin(GL_TRIANGLE_STRIP);

    glColor4ub(255,255,255,255);
    glTexCoord2d(0, windowInfo.imageHeight);
    glVertex3f(-1.0f, -1.0f, 0.0f);

    glColor4ub(255,255,255,255);
    glTexCoord2d(0, 0);
    glVertex3f(-1.0f, 1.0f, 0.0f);

    glColor4ub(255,255,255,255);
    glTexCoord2d(windowInfo.imageWidth, windowInfo.imageHeight);
    glVertex3f( 1.0f, -1.0f, 0.0f);

    glColor4ub(255,255,255,255);
    glTexCoord2d(windowInfo.imageWidth, 0);
    glVertex3f( 1.0f, 1.0f, 0.0f);

    glEnd();
    glDisable(GL_TEXTURE_RECTANGLE_EXT);
    glDisable( GL_TEXTURE_2D );

    wglMakeCurrent(NULL, NULL);
    SwapBuffers(windowInfo.hdc);
}

bool WindowManager::CreateTexture(GLuint* pOutTextureId, HDC hdc, HGLRC hglrc, const cv::Mat& image)
{
    wglMakeCurrent(hdc, hglrc);
    GLuint textureId;
    glEnable(GL_TEXTURE_RECTANGLE_EXT);
    glGenTextures(1, &textureId);
    glBindTexture(GL_TEXTURE_RECTANGLE_EXT, textureId);
    glTexImage2D(GL_TEXTURE_RECTANGLE_EXT, 0, GL_RGBA, image.cols, image.rows, 0, GL_BGR_EXT, GL_UNSIGNED_BYTE, image.data);
    glDisable( GL_TEXTURE_RECTANGLE_EXT );
    wglMakeCurrent(NULL, NULL);

    (*pOutTextureId) = textureId;
    return true;
}

bool WindowManager::CloseImageWindow(const char* windowName, bool isUseMessage)
{
    if (isUseMessage)
    {
        std::lock_guard<std::mutex> lockCloseWindow(m_CloseWindowMutex);
        {
            std::lock_guard<std::mutex> lock(m_CloseWindowMessageMutex);
            CloseWindowRequestMessage request{
                windowName,
                true
            };
            m_CloseWindowRequest = request;
        }

        // ウィンドウクローズ結果同期
        std::chrono::system_clock::time_point beginTime = std::chrono::system_clock::now();
        std::chrono::system_clock::time_point nowTime;
        int durationTime = 0;
        do
        {
            Sleep(MessageWaitSpan);
            nowTime = std::chrono::system_clock::now();
            durationTime = static_cast<unsigned int>(std::chrono::duration_cast<std::chrono::milliseconds>(nowTime - beginTime).count());

        } while ((m_CloseWindowRequest.isEnableRequest || !(m_CloseWindowResult.isEnableResult)) && durationTime < MessageTimeoutTime);


        if (m_CloseWindowRequest.isEnableRequest || !(m_CloseWindowResult.isEnableResult))
        {
            throw std::exception("CloseWindow is failed");
        }
    }
    else
    {
        ProcessCloseWindowMessageHelper(windowName);
    }

    // 結果メッセージ受け取り完了
    m_CloseWindowResult.isEnableResult = false;

    if (!(m_CloseWindowResult.result))
    {
        return false;
    }

    return true;
}

void WindowManager::ProcessMessageThreadBegin()
{
    m_IsActiveMessageThread = true;
    if (m_ProcessMessageThreadObject != nullptr)
    {
        return;
    }
    m_ProcessMessageThreadObject.reset(new std::thread(&WindowManager::ProcessMessage, this));
}

void WindowManager::ProcessMessageThreadEnd()
{
    m_IsActiveMessageThread = false;
    if (m_ProcessMessageThreadObject == nullptr)
    {
        return;
    }
    m_ProcessMessageThreadObject->join();
    m_ProcessMessageThreadObject = nullptr;
}

void WindowManager::ProcessMessage()
{
    MSG message = { 0 };
    autoTestAssistTool::util::FpsController fpsController(30);
    fpsController.BeginLoop();
    while (m_IsActiveMessageThread)
    {
        while (PeekMessage(&message, NULL, 0U, 0U, PM_REMOVE))
        {
            TranslateMessage(&message);
            DispatchMessage(&message);
        }
        // ウィンドウオープン
        ProcessOpenWindowMessage();

        // ウィンドウクローズ
        ProcessCloseWindowMessage();

        fpsController.SleepFrameInterval();
    }
    fpsController.EndLoop();
}

void WindowManager::ProcessOpenWindowMessage()
{
    if (m_OpenWindowRequest.isEnableRequest)
    {
        std::lock_guard<std::mutex> lock(m_OpenWindowMessageMutex);

        ProcessOpenWindowMessageHelper(m_OpenWindowRequest.windowName.c_str(), m_OpenWindowRequest.width, m_OpenWindowRequest.height);

        // メッセージを既読にする。
        m_OpenWindowRequest.isEnableRequest = false;
    }
}

void WindowManager::ProcessOpenWindowMessageHelper(const char* title, int width, int height)
{
    RECT rect = {0,0,width,height};
    HWND windowHandle = NULL;

    AdjustWindowRect(&rect,WindowStyle,FALSE);
    windowHandle = CreateWindow( WindowClassname,
                        title,
                        WindowStyle,
                        CW_USEDEFAULT,
                        CW_USEDEFAULT,
                        rect.right - rect.left,
                        rect.bottom - rect.top,
                        NULL,
                        NULL,
                        0,
                        NULL );

    if (!windowHandle)
    {
        ERRMSG("CreateWindow is failed");
        m_OpenWindowResult.result = false;
        return;
    }

    ShowWindow( windowHandle, SW_SHOWDEFAULT );

    if (!UpdateWindow(windowHandle))
    {
        ERRMSG("UpdateWindow is failed");
        m_OpenWindowResult.result = false;
        return;
    }

    // ウィンドウ作成結果
    m_OpenWindowResult.isEnableResult = true;
    m_OpenWindowResult.result = true;
    m_OpenWindowResult.hwnd = windowHandle;
}

void WindowManager::ProcessCloseWindowMessage()
{
    if (m_CloseWindowRequest.isEnableRequest)
    {
        std::lock_guard<std::mutex> lock(m_CloseWindowMessageMutex);
        ProcessCloseWindowMessageHelper(m_CloseWindowRequest.windowName.c_str());

        m_CloseWindowRequest.isEnableRequest = false;
    }
}

void WindowManager::ProcessCloseWindowMessageHelper(const char* title)
{
    std::lock_guard<std::mutex> lock(m_WindowListMutex);
    std::string name = title;
    for (auto it = m_WindowList.begin(); it != m_WindowList.end(); it++)
    {
        if (name == it->name)
        {
            DestroyWindow(it->hwnd);
            glDeleteTextures(1, &(it->textureId));
            m_WindowList.erase(it);

            m_CloseWindowResult.result = true;
            m_CloseWindowResult.isEnableResult = true;
            return;
        }
    }

    m_CloseWindowResult.result = false;
    m_CloseWindowResult.isEnableResult = true;
}

void WindowManager::AdjustWindowViewPort(HWND windowHandle)
{
    std::lock_guard<std::mutex> lock(m_WindowListMutex);
    for (auto it = m_WindowList.begin(); it != m_WindowList.end(); it++)
    {
        if (it->hwnd != windowHandle)
        {
            continue;
        }

        AdjustWindowInfoAspectRatio((*it));
        DrawWindowInfo((*it));
        break;
    }
}

void WindowManager::RedrawWindow(HWND windowHandle)
{
    std::lock_guard<std::mutex> lock(m_WindowListMutex);
    for (auto it = m_WindowList.begin(); it != m_WindowList.end(); it++)
    {
        if (it->hwnd == windowHandle)
        {
            DrawWindowInfo((*it));
            break;
        }
    }
}

void WindowManager::RemoveWindowList(HWND windowHandle)
{
    std::lock_guard<std::mutex> lock(m_WindowListMutex);
    for (auto it = m_WindowList.begin(); it != m_WindowList.end();)
    {
        if (it->hwnd == windowHandle)
        {
            glDeleteTextures(1, &(it->textureId));
            it = m_WindowList.erase(it);
            return;
        }
        it++;
    }
}

void WindowManager::RegisterWindowClass()
{
    m_WindowClass.cbSize        = sizeof(WNDCLASSEX);
    m_WindowClass.style         = CS_OWNDC | CS_HREDRAW | CS_VREDRAW;
    m_WindowClass.lpfnWndProc   = WndProc;
    m_WindowClass.cbClsExtra    = 0;
    m_WindowClass.cbWndExtra    = 0;
    m_WindowClass.hIcon = (HICON)LoadImage( NULL,
                                IDI_APPLICATION,
                                IMAGE_ICON,
                                0,
                                0,
                                LR_DEFAULTSIZE | LR_SHARED );

    m_WindowClass.hCursor = (HCURSOR)LoadImage( NULL,
                                IDC_ARROW,
                                IMAGE_CURSOR,
                                0,
                                0,
                                LR_DEFAULTSIZE | LR_SHARED );
    m_WindowClass.hInstance     = 0;
    m_WindowClass.hbrBackground = (HBRUSH)GetStockObject( BLACK_BRUSH );
    m_WindowClass.lpszMenuName  = NULL;
    m_WindowClass.lpszClassName = WindowClassname;
    m_WindowClass.hIconSm = (HICON)LoadImage( NULL,
                                IDI_APPLICATION,
                                IMAGE_ICON,
                                0,
                                0,
                                LR_DEFAULTSIZE | LR_SHARED );

    m_WindowClassAtom = RegisterClassEx(&m_WindowClass);
    if( !m_WindowClassAtom ){
        throw std::exception("Register window class was failed.");
    }
}

/*----------------------------------------------------
/ ポップアップメニュー
/---------------------------------------------------*/
void WindowManager::InitializePopupMenu()
{
    m_PopupMenuHandle = CreatePopupMenu();

    /* メニュー項目追加 */
    AppendMenu(m_PopupMenuHandle, MF_STRING, RestoreWindowSizeEvent, RestoreWindowSizeText );
}

void WindowManager::UninitializePoupMenu()
{
    DestroyMenu(m_PopupMenuHandle);
}

void WindowManager::RestoreWindowSize(HWND windowHandle)
{
    std::lock_guard<std::mutex> lock(m_WindowListMutex);
    for (auto it = m_WindowList.begin(); it != m_WindowList.end(); it++)
    {
        if (it->hwnd != windowHandle)
        {
            continue;
        }

        RECT rect = { 0, 0, it->imageWidth, it->imageHeight };
        AdjustWindowRect(&rect, WindowStyle, false);

        SetWindowPos(windowHandle,
            NULL,
            0, 0,
            rect.right - rect.left,
            rect.bottom - rect.top,
            SWP_NOMOVE | SWP_NOZORDER);

        RECT windowClientRect;
        GetClientRect(it->hwnd, &windowClientRect);
        it->windowClientWidth  = windowClientRect.right  - windowClientRect.left;
        it->windowClientHeight = windowClientRect.bottom - windowClientRect.top;

        DrawWindowInfo((*it));
    }
}

void WindowManager::OpenPopupMenu(HWND windowHandle, int x, int y )
{
    // ポップアップメニュー表示
    TrackPopupMenu(m_PopupMenuHandle,
        TPM_LEFTALIGN | TPM_TOPALIGN,
        x, y,
        0,
        windowHandle,
        NULL);
}
