﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <windows.h>


#include "FilterManager.h"

bool operator <(const FilterIdType &left, const FilterIdType &right)
{
    return left.id < right.id;
}

FilterManager::FilterManager()
{
    std::lock_guard<std::mutex> lock(m_FilterListMutex);
    m_FilterList.clear();
}

FilterManager::~FilterManager()
{
    std::lock_guard<std::mutex> lock(m_FilterListMutex);
    m_FilterList.clear();
}

void FilterManager::CreateFilter(FilterIdType *pOutFilterId)
{
    FilterIdType id;
    id.id = GetTickCount();

    std::lock_guard<std::mutex> lock(m_FilterListMutex);
    while (m_FilterList.count(id))
    {
        id.id = GetTickCount();
    }
    m_FilterList[id] = std::vector<FilterAction>();

    (*pOutFilterId) = id;
}

void FilterManager::DeleteFilter(FilterIdType filterId)
{
    CheckFilterId(filterId);

    std::lock_guard<std::mutex> lock(m_FilterListMutex);
    m_FilterList.erase(filterId);
}

void FilterManager::SetBinarizationFilter(FilterIdType filterId, BinarizationFilterParam param)
{
    CheckFilterId(filterId);
    if (param.threshold < 0 || param.threshold > 255)
    {
        throw InvalidFilterParameterException(FilterActionType::FilterActionType_Binarization);
    }
    FilterAction action;
    action.type = FilterActionType::FilterActionType_Binarization;
    action.param.binarizationFilterParam = param;

    std::lock_guard<std::mutex> lock(m_FilterListMutex);
    m_FilterList[filterId].push_back(action);
}

void FilterManager::SetColorRangeFilter(FilterIdType filterId, ColorRangeFilterParam param)
{
    CheckFilterId(filterId);

    if ( param.lowerColor.red < 0 || param.lowerColor.red > 255
        || param.lowerColor.green < 0 || param.lowerColor.green > 255
        || param.lowerColor.blue < 0  || param.lowerColor.blue > 255
        || param.upperColor.red < 0   || param.upperColor.red > 255
        || param.upperColor.green < 0 || param.upperColor.green > 255
        || param.upperColor.blue < 0  || param.upperColor.blue > 255)
    {
        throw InvalidFilterParameterException(FilterActionType::FilterActionType_ColorRange);
    }

    FilterAction action;
    action.type = FilterActionType::FilterActionType_ColorRange;
    action.param.colorRangeFilterParam = param;

    std::lock_guard<std::mutex> lock(m_FilterListMutex);
    m_FilterList[filterId].push_back(action);
}

void FilterManager::SetMorphologyFilter(FilterIdType filterId, MorphologyFilterParam param )
{
    CheckFilterId(filterId);

    FilterAction action;
    action.type = FilterActionType::FilterActionType_Morphology;
    action.param.morphologyFilterParam = param;
    if ( param.iterations <= 0
        || (param.type != MorphologyConversionType::MorphologyConversionType_Erosion
            && param.type != MorphologyConversionType::MorphologyConversionType_Closing
            && param.type != MorphologyConversionType::MorphologyConversionType_Dilation
            && param.type != MorphologyConversionType::MorphologyConversionType_Opening))
    {
        throw InvalidFilterParameterException(FilterActionType::FilterActionType_Morphology);
    }

    std::lock_guard<std::mutex> lock(m_FilterListMutex);
    m_FilterList[filterId].push_back(action);
}

void FilterManager::ApplyFilter(cv::Mat *pOutImage, cv::Mat srcImage, FilterIdType filterId)
{
    CheckFilterId(filterId);

    std::lock_guard<std::mutex> lock(m_FilterListMutex);
    const std::vector<FilterAction> &list = m_FilterList[filterId];


    cv::Mat outputImage = srcImage.clone();
    cv::Mat filterdImage;

    for (std::vector<FilterAction>::const_iterator it = list.begin(); it != list.end(); it++)
    {
        switch (it->type)
        {
            case FilterActionType::FilterActionType_Binarization:
            {
                ApplyBinarizationFilter(filterdImage, outputImage, it->param.binarizationFilterParam);
                break;
            }
            case FilterActionType::FilterActionType_ColorRange:
            {
                ApplyColorRangeFilter(filterdImage, outputImage, it->param.colorRangeFilterParam);
                break;
            }
            case FilterActionType::FilterActionType_Morphology:
            {
                ApplyMorphologyFilter(filterdImage, outputImage, it->param.morphologyFilterParam);
                break;
            }
            default:
                throw InvalidFilterActionTypeException(it->type);
        }
        // TIPS: filterdImage をコピーして参照カウンターを増やした後は、release して指すポインタ位置を同一にさせないようにする。
        outputImage = filterdImage;
        filterdImage.release();
    }

    (*pOutImage) = outputImage;
}

void FilterManager::CheckFilterId(FilterIdType filterId)
{
    std::lock_guard<std::mutex> lock(m_FilterListMutex);
    if (m_FilterList.count(filterId) == 0)
    {
        throw InvalidFilterIdException(filterId);
    }

}

void FilterManager::ApplyBinarizationFilter(cv::Mat& dest, const cv::Mat src, BinarizationFilterParam param)
{
    // グレースケール
    cv::Mat grayImage;
    cv::cvtColor(src, grayImage, cv::COLOR_BGR2GRAY);
    // 二値化
    cv::Mat binarizeImage;
    cv::threshold(grayImage, binarizeImage, param.threshold, 255, cv::THRESH_BINARY);
    // チャンネル数調整
    cv::cvtColor(binarizeImage, dest, CV_GRAY2BGR);
}

void FilterManager::ApplyColorRangeFilter(cv::Mat& dest, const cv::Mat src, ColorRangeFilterParam param)
{
    cv::Mat maskImage;
    cv::inRange(src,
        cv::Scalar(param.lowerColor.blue, param.lowerColor.green, param.lowerColor.red),
        cv::Scalar(param.upperColor.blue, param.upperColor.green, param.upperColor.red),
        maskImage);

    cv::bitwise_and(src, src, dest, maskImage);
}

void FilterManager::ApplyMorphologyFilter(cv::Mat& dest, const cv::Mat src, MorphologyFilterParam param)
{
    switch (param.type)
    {
        case MorphologyConversionType::MorphologyConversionType_Erosion:
        {
            cv::morphologyEx(src, dest, cv::MORPH_ERODE, cv::Mat(), cv::Point(-1, -1), param.iterations);
            break;
        }
        case MorphologyConversionType::MorphologyConversionType_Dilation:
        {
            cv::morphologyEx(src, dest, cv::MORPH_DILATE, cv::Mat(), cv::Point(-1, -1), param.iterations);
            break;
        }
        case MorphologyConversionType::MorphologyConversionType_Opening:
        {
            cv::morphologyEx(src, dest, cv::MORPH_OPEN, cv::Mat(), cv::Point(-1, -1), param.iterations);
            break;
        }
        case MorphologyConversionType::MorphologyConversionType_Closing:
        {
            cv::morphologyEx(src, dest, cv::MORPH_CLOSE, cv::Mat(), cv::Point(-1, -1), param.iterations);
            break;
        }
        default:
            throw InvalidFilterParameterException(FilterActionType::FilterActionType_Morphology);
            break;
    }
}
