﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "DevkitCapture.h"
#include <string>
#include <sstream>
#include <tchar.h>
#include <WinInet.h>

#pragma comment( lib, "Wininet.lib" )

typedef std::basic_string<TCHAR>  tstring;

CaptureResult DevkitCapture::GetImageDataFromHostBridge(uint8_t* pOutColorBuffer, size_t colorBufferSize, std::string strIp)
{
    tstring strServer = _T(strIp);
    tstring strUpdateImagePath = _T("/cgi-bin/lcd/landscape/bmp");
    tstring strGetImageUrl = _T("/capture/image.bmp");

    HINTERNET hInternetOpen = NULL;
    HINTERNET hInternetConnect = NULL;
    HINTERNET hInternetRequest = NULL;
    DWORD dwFlags = INTERNET_FLAG_RELOAD    // 要求されたファイル、オブジェクト、またはフォルダ一覧を、キャッシュからではなく、元のサーバーから強制的にダウンロードします。
        | INTERNET_FLAG_DONT_CACHE          // 返されたエンティティをキャシュへ追加しません。
        | INTERNET_FLAG_NO_AUTO_REDIRECT;   // HTTP だけで使用され、リダイレクトが HttpSendRequest で処理されないことを指定します。

    bool result = false;

    InternetCloseHandle(NULL);
    do
    {
        hInternetOpen = InternetOpen(NULL, INTERNET_OPEN_TYPE_PRECONFIG, NULL, NULL, 0);
        if (NULL == hInternetOpen)
        {
            ERRMSG("Initialize WinInet failed\n");
            break;
        }

        hInternetConnect = InternetConnect(hInternetOpen, strServer.c_str(),
            INTERNET_DEFAULT_HTTP_PORT, NULL, NULL, INTERNET_SERVICE_HTTP, 0, 0);
        if (NULL == hInternetConnect)
        {
            ERRMSG("InternetConnect failed\n");
            break;
        }

        // 画像の更新要求
        {
            hInternetRequest = HttpOpenRequest(hInternetConnect, "GET",
                strUpdateImagePath.c_str(), NULL, NULL, NULL, dwFlags, NULL);
            if (NULL == hInternetRequest)
            {
                ERRMSG("HttpOpenRequest failed\n");
                break;
            }

            if (!HttpSendRequest(hInternetRequest, NULL, 0, NULL, 0))
            {
                ERRMSG("HttpSendRequest failed\n");
                break;
            }

            // HTTP要求に対応するステータスコードの取得
            DWORD dwStatusCode;
            DWORD dwLength = sizeof(DWORD);
            if (!HttpQueryInfo(hInternetRequest,
                HTTP_QUERY_STATUS_CODE | HTTP_QUERY_FLAG_NUMBER, &dwStatusCode, &dwLength, 0))
            {
                ERRMSG("HttpQueryInfo failed\n");
                break;
            }
            if (HTTP_STATUS_OK != dwStatusCode)
            {
                ERRMSG("UpdateImage failed\n");
                break;
            }

            InternetCloseHandle(hInternetRequest);
        }

        // 画像の取得
        {
            hInternetRequest = HttpOpenRequest(hInternetConnect, "GET",
                strGetImageUrl.c_str(), NULL, NULL, NULL, dwFlags, NULL);
            if (NULL == hInternetRequest)
            {
                ERRMSG("HttpOpenRequest failed\n");
                break;
            }

            if (!HttpSendRequest(hInternetRequest, NULL, 0, NULL, 0))
            {
                ERRMSG("HttpSendRequest failed\n");
                break;
            }

            // HTTP要求に対応するステータスコードの取得
            DWORD dwStatusCode;
            DWORD dwLength = sizeof(DWORD);
            if (!HttpQueryInfo(hInternetRequest,
                HTTP_QUERY_STATUS_CODE | HTTP_QUERY_FLAG_NUMBER, &dwStatusCode, &dwLength, 0))
            {
                ERRMSG("HttpQueryInfo failed\n");
                break;
            }
            if (HTTP_STATUS_OK != dwStatusCode)
            {
                ERRMSG("GetImage failed\n");
                break;
            }

            // bmp 読み込み
            const size_t headerSize = 54;
            char headerBuffer[headerSize];
            DWORD dwRead = 0;
            if (!InternetReadFile(hInternetRequest, headerBuffer, headerSize, &dwRead))
            {
                ERRMSG("InternetReadFile failed\n");
                break;
            }
            if (headerSize != dwRead)
            {
                ERRMSG("Reading bmp header failed\n");
                break;
            }
            // bmp のヘッダから情報取得
            int32_t width = *(int32_t*)&headerBuffer[18];
            int32_t height = *(int32_t*)&headerBuffer[22];
            int16_t bit = *(int16_t*)&headerBuffer[28];
            height = std::abs(height);

            if (bit != 24 || width * height * 3 != colorBufferSize)
            {
                ERRMSG("bmp size is not expected\n");
                break;
            }

            dwRead = 0;
            // colorBufferSize は 32bit を超えない想定のため、キャスト
            if (!InternetReadFile(hInternetRequest, pOutColorBuffer, static_cast<DWORD>(colorBufferSize), &dwRead))
            {
                ERRMSG("InternetReadFile failed\n");
                break;
            }
            if (colorBufferSize != dwRead)
            {
                ERRMSG("Reading bmp data failed\n");
                break;
            }
        }

        result = true;
    } while (false);

    InternetCloseHandle(hInternetRequest);
    InternetCloseHandle(hInternetConnect);
    InternetCloseHandle(hInternetOpen);
    return result ? CaptureResult::CaptureResult_Success : CaptureResult::CaptureResult_Unexpected;
} // NOLINT(impl/function_size)

DevkitCapture::DevkitCapture(std::string strIp, std::string serialNumber, bool isPreview)
{
    m_CaptureThread = NULL;

    m_Ip = strIp;
    m_SerialNumber = serialNumber;
    m_IsPreview = isPreview;

    StartCapture();
}

DevkitCapture::~DevkitCapture()
{
    EndCapture();
}

std::string DevkitCapture::GetPreviewName()
{
    std::ostringstream previewName;
    previewName << "[" << m_SerialNumber << "] " << m_Ip << " - Auto Test Assist Tool";
    return previewName.str();
}

int DevkitCapture::GetOriginalPreviewWidth()
{
    return DefaultWindowWidth;
}

int DevkitCapture::GetOriginalPreviewHeight()
{
    return DefaultWindowHeight;
}

/*----------------------------------------------------
/ キャプチャ処理
/---------------------------------------------------*/
void DevkitCapture::StartCapture()
{
    m_IsCapture = true;
    if (m_CaptureThread != NULL)
    {
        return;
    }

    // m_CaptureImage 初期化
    {
        std::lock_guard<std::mutex> lock(m_CaptureMutex);
        m_CaptureImage.create(cv::Size(DefaultWindowWidth, DefaultWindowHeight), CV_8UC3);
    }

    // 開始前に一度キャプチャを取得しておく
    Capture();

    m_CaptureThread.reset(new std::thread(&DevkitCapture::RunCapture, this));

    // プレビュー開始
    if (m_IsPreview)
    {
        StartPreview();
    }
}

void DevkitCapture::EndCapture()
{
    // プレビュー終了
    EndPreview();

    m_IsCapture = false;
    if (m_CaptureThread != NULL)
    {
        m_CaptureThread->join();
        m_CaptureThread = NULL;
    }
}

void DevkitCapture::RunCapture()
{
    while (m_IsCapture)
    {
        Capture();
        Sleep((DWORD)PreviewFrameSpan);
    }
}

void DevkitCapture::Capture()
{
    std::lock_guard<std::mutex> lock(m_CaptureMutex);
    GetImageDataFromHostBridge(m_CaptureImage.ptr(), DefaultWindowWidth * DefaultWindowHeight * 3, m_Ip);
}
