﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <chrono>
#include "CaptureGrabber.h"

// SampleGrabber、NullRenderer の CLSID, IID を定義
DEFINE_GUID(IID_ISampleGrabber,0x6b652fff,0x11fe,0x4fce,0x92,0xad,0x02,0x66,0xb5,0xd7,0xc7,0x8f);
DEFINE_GUID(CLSID_SampleGrabber,0xc1f400a0,0x3f08,0x11d3,0x9f,0x0b,0x00,0x60,0x08,0x03,0x9e,0x37);

namespace
{
    const int   OPEN_TRY_COUNT = 10;
    const int   OPEN_TRY_CONNECT_TIME = 1000;
    const int   OPEN_TRY_CONNECT_INTERVAL = 50;
    const char* UNKNOWN_DEVICE_NAME = "Unknown Device";
};

CaptureGrabber::CaptureGrabber()
{
    // COMを初期化
    CoInitialize(NULL);

    // フィルタグラフビルダ
    m_GraphBuilder           = NULL;
    m_CaptureGraphBuilder2   = NULL;

    // フィルタ
    m_DeviceFilter           = NULL;
    m_SampleGrabberFilter    = NULL;
    m_SampleGrabber          = NULL;

    // フィルタグラフマネージャー
    m_MediaControl           = NULL;

    // デバイス名
    m_DeviceName = "";

    m_Width  = 0;
    m_Height = 0;

    m_DeviceId           = 0;
    m_ImageBufferSize    = 0;
    m_ImageBuffer        = NULL;
    m_IsOpened  = false;
}


CaptureGrabber::~CaptureGrabber()
{
    Release();

    // COM終了
    CoUninitialize();
}

bool CaptureGrabber::Open(int deviceId, int width, int height)
{
    // 開始済みの場合は接続失敗とする
    if (m_IsOpened)
    {
        return false;
    }

    m_DeviceId   = deviceId;
    m_Width      = width;
    m_Height     = height;


    m_ImageBufferSize = width * height * 3;
    if (m_ImageBuffer != NULL)
    {
        delete[] m_ImageBuffer;
    }
    m_ImageBuffer = new unsigned char[m_ImageBufferSize];


    // 入力フィルタ作成（デバイスフィルタ）
    if (FAILED(CreateSourceFilter()))
    {
        Release();
        return false;
    }

    // 中間フィルタ作成
    if (FAILED(CreateTransformFilter()))
    {
        Release();
        return false;
    }


    // フィルタグラフ作成
    if (FAILED(CreateFilterGraph()))
    {
        Release();
        return false;
    }


    // ストリーム、メディア設定
    if (FAILED(SetupStreamConfig()))
    {
        Release();
        return false;
    }
    if (FAILED(SetupMediaConfig()))
    {
        Release();
        return false;
    }


    // フィルタグラフマネージャー作成
    if (FAILED(CreateMediaControl())){
        Release();
        return false;
    }


    // レンダリング開始
    if (FAILED(RenderStart()))
    {
        Release();
        return false;
    }

    m_IsOpened = true;

    return true;
}

HRESULT CaptureGrabber::CreateSourceFilter()
{
    ICreateDevEnum *pCreateDevEnum = NULL;
    IEnumMoniker *pEnumMoniker = NULL;
    IMoniker *pMoniker = NULL;
    ULONG nFetched = 0;
    HRESULT result;

    // 接続デバイス列挙用インターフェース作成
    result = CoCreateInstance(CLSID_SystemDeviceEnum, NULL, CLSCTX_INPROC_SERVER, IID_ICreateDevEnum, (PVOID *)&pCreateDevEnum);
    if (FAILED(result))
    {
        return result;
    }

    // ビデオ入力デバイス列挙用インターフェース作成
    result = pCreateDevEnum->CreateClassEnumerator(CLSID_VideoInputDeviceCategory, &pEnumMoniker, 0);
    if (FAILED(result))
    {
        return result;
    }
    // イテレーターを先頭へ配置
    pEnumMoniker->Reset();


    // ビデオ入力デバイス探索
    // アクセスの優先順位が高いデバイスから順番に取得し、device Id と一致するデバイスを取得
    for (int i = 0; pEnumMoniker->Next(1, &pMoniker, &nFetched) == S_OK; i++)
    {
        if (i == m_DeviceId)
        {
            // デバイス番号と一致するデバイスのソースフィルタを作成
            pMoniker->BindToObject( NULL, NULL, IID_IBaseFilter, (void**)&m_DeviceFilter);

            // デバイス名取得
            IPropertyBag*   propertyBag;
            result = pMoniker->BindToStorage(0, 0, IID_IPropertyBag, (void**)(&propertyBag));
            if (SUCCEEDED(result))
            {
                VARIANT         varName;
                VariantInit(&varName);
                result = propertyBag->Read( L"FriendlyName", &varName, 0 );

                if (SUCCEEDED(result))
                {
                    const int deviceNameBufferSize = 256;
                    char deviceNameTmp[deviceNameBufferSize];

                    // BSTR 形式を C 文字列へ変換
                    WideCharToMultiByte(CP_UTF8, 0, varName.bstrVal, -1, deviceNameTmp, deviceNameBufferSize, 0, 0);

                    m_DeviceName = deviceNameTmp;
                }
                propertyBag->Release();
            }

            if (FAILED(result))
            {
                m_DeviceName = UNKNOWN_DEVICE_NAME;
            }

            break;
        }
    }

    // デバイス列挙用インターフェース解放
    pMoniker->Release();
    pEnumMoniker->Release();
    pCreateDevEnum->Release();


    if (m_DeviceFilter == NULL)
    {
        return E_FAIL;
    }
    return S_OK;
}
HRESULT CaptureGrabber::CreateTransformFilter()
{

    HRESULT result;

    // SampleGrabber フィルターを生成
    result = CoCreateInstance(CLSID_SampleGrabber,
            NULL, CLSCTX_INPROC,
            IID_IBaseFilter,
            (LPVOID *)&m_SampleGrabberFilter);


    if (FAILED(result))
    {
        return result;
    }

    result = m_SampleGrabberFilter->QueryInterface(IID_ISampleGrabber,
            (LPVOID *)&m_SampleGrabber);

    if (FAILED(result))
    {
        return result;
    }


    // 入力値をバッファ保管させる
    result = m_SampleGrabber->SetBufferSamples(TRUE);
    if (FAILED(result))
    {
        return result;
    }

    return S_OK;
}



// フィルタグラフ作成
HRESULT CaptureGrabber::CreateFilterGraph()
{
    HRESULT result;

    // フィルタグラフ作成
    result = CoCreateInstance(CLSID_FilterGraph,
        NULL, CLSCTX_INPROC,
        IID_IGraphBuilder,
        (LPVOID *)&m_GraphBuilder);

    if (FAILED(result))
    {
        return result;
    }

    // 入力フィルタ追加
    result = m_GraphBuilder->AddFilter(m_DeviceFilter, L"Device Filter");
    if (FAILED(result))
    {
        return result;
    }

    // 中間フィルタ追加
    result = m_GraphBuilder->AddFilter(m_SampleGrabberFilter, L"Sample Grabber");
    if (FAILED(result))
    {
        return result;
    }

    // キャプチャグラフビルダを作成
    result = CoCreateInstance(CLSID_CaptureGraphBuilder2, NULL, CLSCTX_INPROC,
        IID_ICaptureGraphBuilder2,
        (LPVOID *)&m_CaptureGraphBuilder2);
    if (FAILED(result))
    {
        return result;
    }

    // 作成済みのフィルタグラフをセット
    result = m_CaptureGraphBuilder2->SetFiltergraph(m_GraphBuilder);
    if (FAILED(result))
    {
        return result;
    }

    return S_OK;
}

HRESULT CaptureGrabber::RenderStart()
{
    HRESULT result;

    // フィルタ接続設定、
    result = m_CaptureGraphBuilder2->RenderStream(&PIN_CATEGORY_PREVIEW,
        &MEDIATYPE_Video,
        m_DeviceFilter,
        NULL,
        m_SampleGrabberFilter);
    if (FAILED(result))
    {
        return result;
    }

    // データ取得が成功するまで待機
    long bufferSize;
    int64_t remainingTime;
    std::chrono::system_clock::time_point  beginTime;
    for (int i = 0; i < OPEN_TRY_COUNT; i++)
    {
        remainingTime = OPEN_TRY_CONNECT_TIME;

        // キャプチャ開始
        result = m_MediaControl->Pause();
        if (FAILED(result))
        {
            continue;
        }

        result = m_MediaControl->Stop();
        if (FAILED(result))
        {
            continue;
        }

        result = m_MediaControl->Run();
        if (FAILED(result))
        {
            continue;
        }

        // データ取得
        do{
            beginTime = std::chrono::system_clock::now();

            // データ取得
            result = m_SampleGrabber->GetCurrentBuffer(reinterpret_cast<long*>(&bufferSize),  NULL);
            if (SUCCEEDED(result))
            {
                break;
            }

            // スリープ
            Sleep(OPEN_TRY_CONNECT_INTERVAL);

            remainingTime = remainingTime - std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::system_clock::now() - beginTime).count();
        } while ( remainingTime > 0);


        if (SUCCEEDED(result))
        {
            break;
        }
    }

    return result;
}



void CaptureGrabber::Release()
{
    // メディア読み込み終了
    if (m_MediaControl != NULL)
    {
        m_MediaControl->Pause();
        m_MediaControl->Stop();
    }

    // 各フィルタ解放
    if (m_DeviceFilter != NULL)
    {
        m_DeviceFilter->Release();
        m_DeviceFilter = NULL;
    }
    if (m_SampleGrabberFilter != NULL)
    {
        m_SampleGrabberFilter->Release();
        m_SampleGrabberFilter = NULL;
    }

    // SampleGrabberInterface 解放
    if (m_SampleGrabber != NULL)
    {
        m_SampleGrabber->Release();
        m_SampleGrabber = NULL;
    }


    // メディアコントローラ解放
    if (m_MediaControl != NULL)
    {
        m_MediaControl->Release();
        m_MediaControl = NULL;
    }

    // 画像取得バッファを解放
    if (m_ImageBuffer != NULL)
    {
        delete[] m_ImageBuffer;
        m_ImageBuffer = NULL;
    }


    // フィルターグラフ解放
    if (m_CaptureGraphBuilder2 != NULL)
    {
        m_CaptureGraphBuilder2->Release();
        m_CaptureGraphBuilder2 = NULL;
    }
    if (m_GraphBuilder != NULL)
    {
        m_GraphBuilder->Release();
        m_GraphBuilder = NULL;
    }

    m_IsOpened = false;
}
bool CaptureGrabber::Read(unsigned char* pOutImageBuffer, long bufferSize)
{
    // SampleGrabber の未初期化
    if (m_SampleGrabber == NULL || m_IsOpened == false)
    {
        return false;
    }

    HRESULT result;
    result = m_SampleGrabber->GetCurrentBuffer((long*)&bufferSize,  (long*)m_ImageBuffer);
    if (FAILED(result))
    {
        return false;
    }

    int widthInBytes = m_Width * 3;
    for(int y = 0; y < m_Height; y++){
        // メモリアクセス範囲確認
        if (((y * widthInBytes) + widthInBytes) >= bufferSize)
        {
            continue;
        }
        if ((((m_Height - 1) - y) * widthInBytes) >= m_ImageBufferSize)
        {
            continue;
        }

        // 上下反転コピー
        memcpy(pOutImageBuffer + (y * widthInBytes), m_ImageBuffer + ( ((m_Height - 1) - y) * widthInBytes), widthInBytes);
    }

    return true;
}


HRESULT CaptureGrabber::CreateMediaControl()
{
    // MediaControlインターフェース取得
    return m_GraphBuilder->QueryInterface(IID_IMediaControl, (LPVOID *)&m_MediaControl);
}


HRESULT CaptureGrabber::SetupStreamConfig()
{
    IAMStreamConfig*    pStreamConfig;
    int formatNum       = 0;
    int formatSize      = 0;
    HRESULT result      = E_FAIL;

    // stream config インターフェース作成
    result = m_CaptureGraphBuilder2->FindInterface(&PIN_CATEGORY_CAPTURE, &MEDIATYPE_Video, m_DeviceFilter, IID_IAMStreamConfig, (LPVOID*)&pStreamConfig);
    if (FAILED(result))
    {
        return result;
    }

    // 接続可能なストリームフォーマット数取得
    result = pStreamConfig->GetNumberOfCapabilities(&formatNum, &formatSize);
    if (FAILED(result))
    {
        pStreamConfig->Release();
        return result;
    }

    bool isMatchSize = false;
    if (formatSize == sizeof(VIDEO_STREAM_CONFIG_CAPS))
    {
        VIDEO_STREAM_CONFIG_CAPS    format;
        AM_MEDIA_TYPE*              mediaType;

        bool isMatchSizeWidth = false;
        bool isMatchSizeHeight = false;
        for (int i = 0; i < formatNum; i++)
        {
            result = pStreamConfig->GetStreamCaps(i, &mediaType, (BYTE*)&format);
            if (FAILED(result))
            {
                continue;
            }
            // メジャータイプチェック
            if (mediaType->majortype != MEDIATYPE_Video)
            {
                continue;
            }


            if (format.OutputGranularityX < 1 || format.OutputGranularityY < 1)
            {
                continue;
            }

            // フラグリセット
            isMatchSizeWidth = false;
            isMatchSizeHeight = false;

            // 一致する幅を探索
            for (int x = format.MinOutputSize.cx; x <= format.MaxOutputSize.cx; x += format.OutputGranularityX)
            {
                if (x == m_Width)
                {
                    isMatchSizeWidth = true;
                }
            }

            // 一致する高さを探索
            for (int y = format.MinOutputSize.cy; y <= format.MaxOutputSize.cy; y += format.OutputGranularityY)
            {
                if (y == m_Height)
                {
                    isMatchSizeHeight = true;
                }
            }

            if (!isMatchSizeWidth || !isMatchSizeHeight)
            {
                continue;
            }

            // ストリームフォーマット設定
            result = pStreamConfig->SetFormat(mediaType);
            if (FAILED(result))
            {
                continue;
            }

            isMatchSize = true;

            // TODO: デバッグ表示で、フォーマット名を表示させる
            break;
        }
    }
    pStreamConfig->Release();

    // 適切なサイズが見つからなかった場合
    if (!isMatchSize)
    {
        return E_FAIL;
    }

    return S_OK;
}


HRESULT CaptureGrabber::SetupMediaConfig()
{
    // メディアタイプ指定
    HRESULT result;
    AM_MEDIA_TYPE mt;
    ZeroMemory(&mt,sizeof(AM_MEDIA_TYPE));
    mt.majortype    = MEDIATYPE_Video;
    mt.subtype      = MEDIASUBTYPE_RGB24;
    mt.formattype   = FORMAT_VideoInfo;

    // API 呼び出し側への画像受け渡しは、出力フィルターのバッファデータを渡す。
    // 出力層フィルターが RGB24 形式でデータを受け取れるように、入力ピンの形式を RGB24 形式としておく。
    result = m_SampleGrabber->SetMediaType(&mt);
    if (FAILED(result))
    {
        return result;
    }
    return S_OK;
}


bool CaptureGrabber::IsOpened()
{
    return m_IsOpened;
}

std::string CaptureGrabber::GetDeviceName()
{
    return m_DeviceName;
}
