﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdio>
#include <cstring>
#include <cstdlib>
#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/dd.h>

#include "GraphicMain.h"
#include "FileManager.h"
#include "DisplayData.h"
#include "Option.h"
#include "XcieDecryptor.h"

#include <nn/sdmmc/sdmmc_Common.h>
#include <nn/gc/detail/gc_Util.h>
#include <nn/gc/gc.h>
#include <nn/gc/detail/gc_Define.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/fs/fs_GameCardForInspection.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_ResultHandler.h>
#include <nn/fs/fs_TimeStampForDebug.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/ns/ns_ApiForDfc.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/ns_GameCardStopper.h>

#include <nn/fs/fs_SystemSaveData.h>
#include <nn/fs/fs_SaveData.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_File.h>
#include <nn/fs/fs_Directory.h>
#include <nn/fs/fs_FileSystem.h>

#include <nn/hid.h>
#include <nn/hid/hid_TouchScreen.h>
#include <nn/hid/hid_DebugPad.h>
#include <nn/util/util_BitUtil.h>
#include <nn/fs/fs_IEventNotifier.h>
#include <nn/os/os_SystemEventApi.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/crypto/crypto_RsaOaepEncryptor.h>
#include <nn/crypto/crypto_RsaOaepDecryptor.h>
#include <nn/crypto/crypto_Aes128CtrDecryptor.h>
#include <nn/crypto/crypto_Aes128CtrEncryptor.h>
#include <nn/crypto/crypto_Aes128CbcDecryptor.h>
#include <nn/crypto/crypto_Aes128CbcEncryptor.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_Result.h>
#include <nn/time/time_Api.h>
#include <nn/time/time_StandardNetworkSystemClock.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/time/time_TimeZoneApi.h>

char g_CardHeader[nn::gc::GcPageSize];
char g_CardCert[nn::gc::GcPageSize];
char g_DevCardDefaultParam[75];

const size_t EncryptedAreaStartAddress = 0x200;
const size_t EncryptedAreaLength = 0xD00;
const size_t EncryptedKeyAreaLength = 0x100;
const size_t KeyAreaLength = 0x1000;
char g_EncryptedArea[EncryptedAreaLength];

static const size_t MaxFilePathLength = 512;
const size_t ApplicationHeapSize = 128 * 1024 * 1024;
const size_t WriteUnitSize = 8 * 1024 * 1024;
const size_t DataBufferSize = WriteUnitSize;

char* g_pDataBuffer;
char* g_pReadBufferFromFs;

// スレッド用
nn::os::ThreadType g_Thread;
nn::os::ThreadType g_WriteThread;
static const int ThreadPriority = 9;
static const size_t ThreadStackSize = 128 * 1024;
NN_OS_ALIGNAS_THREAD_STACK char g_ThreadStack[ThreadStackSize];
NN_OS_ALIGNAS_THREAD_STACK char g_WriteThreadStack[ThreadStackSize];
char g_ThreadName[] = "Test";
char g_ThreadName2[] = "WriteTest";
nn::os::EventType g_ThreadEndEvent;
nn::os::EventType g_ExecuteEvent;
nn::os::EventType g_WriteStartEvent;
nn::os::EventType g_WriteEndEvent;
nn::os::EventType g_WriteFinishEvent;
nn::os::EventType g_MoveToMenuEvent;

// 画面描画用
FontSystem* g_pFontSystem;
GraphicsSystem* g_pGraphicsSystem;

DisplayData* g_pDisplayData = DisplayData::GetInstance();
GraphicMain g_GraphicMain;

// GC ドライバ用
nn::os::SystemEventType g_CardDetectionEvent;

bool g_IsWriteMode = false;

struct WritePageInfo
{
    int64_t byteOffset;
    size_t byteSize;
};

WritePageInfo g_WritePageInfo;
std::unique_ptr<nn::fs::IEventNotifier> g_Notifier;

void PrintDebugArray(const void* buf, const size_t bufferLength)
{
    const char* buffer = reinterpret_cast<const char*>(buf);
    for(size_t i = 0; i < bufferLength; i++)
    {
        if(i != 0 && i % 16 == 0)
        {
            NN_LOG("\n");
        }

        NN_LOG("%02X ", buffer[i]);

        if(i % 8 == 7)
        {
            NN_LOG(" ");
        }
    }
    NN_LOG("\n");
}

static char g_CardHeaderForDev[] =
{
    0x57, 0x15, 0xCD, 0xC1, 0x11, 0x96, 0x9A, 0xBE, 0xAB, 0x33, 0x8C, 0x30, 0xDB, 0x8D, 0x58, 0x9B,
    0x43, 0x7C, 0x94, 0x6B, 0x1A, 0xE6, 0xE1, 0xC5, 0x6C, 0xF7, 0xE8, 0x46, 0x32, 0xD1, 0x13, 0x13,
    0x67, 0x2D, 0x38, 0x15, 0x65, 0x63, 0x82, 0xDB, 0x21, 0x41, 0x77, 0x6E, 0x3D, 0xDC, 0x71, 0xAC,
    0x04, 0xEE, 0x81, 0xE9, 0xEE, 0x65, 0x04, 0x4B, 0x5A, 0x81, 0x53, 0x2A, 0xDD, 0xB6, 0x72, 0x32,
    0xF5, 0x58, 0xF0, 0xF8, 0x8F, 0x9E, 0x79, 0x7E, 0xF3, 0xC7, 0x01, 0xFD, 0x17, 0xE9, 0x56, 0xBC,
    0xC8, 0x3F, 0xAE, 0xDD, 0xE9, 0xDB, 0xEB, 0x42, 0xA2, 0x2C, 0xF3, 0x1A, 0x63, 0xBE, 0xB9, 0xE2,
    0x21, 0xBE, 0x6C, 0x00, 0x1B, 0x71, 0xDD, 0x18, 0xBC, 0x81, 0xAD, 0xA2, 0xC5, 0x5A, 0xE9, 0xFB,
    0x3B, 0x35, 0x7F, 0xB0, 0x95, 0xD5, 0xD4, 0x3A, 0x3F, 0xE5, 0x01, 0xE5, 0x7A, 0xB6, 0x55, 0x08,
    0x89, 0x79, 0x53, 0x8C, 0xDE, 0x1D, 0xBE, 0x40, 0x6A, 0x76, 0xA4, 0x38, 0xB9, 0xB1, 0xD6, 0x2F,
    0xF3, 0x7B, 0x89, 0x41, 0x5C, 0x41, 0x27, 0xD7, 0xD6, 0x32, 0xB4, 0xB1, 0xC3, 0x4A, 0x5B, 0xC5,
    0x99, 0x2F, 0xD0, 0x73, 0x7F, 0x80, 0x65, 0x3E, 0x7F, 0x4D, 0x04, 0x1F, 0x75, 0xF3, 0x29, 0x8A,
    0x75, 0x1E, 0xFB, 0x1B, 0x93, 0x62, 0x47, 0x4C, 0xE8, 0x8B, 0xF2, 0x31, 0xEC, 0xDA, 0xF7, 0x82,
    0xFF, 0x50, 0x3C, 0xA8, 0xAF, 0xC5, 0x27, 0x33, 0x8E, 0x19, 0xBC, 0x86, 0xB5, 0xFF, 0x50, 0xA6,
    0x2D, 0xA8, 0xBD, 0x61, 0x03, 0xCA, 0xE7, 0xCA, 0xC9, 0x34, 0x9C, 0xFB, 0x29, 0x31, 0x8E, 0xB0,
    0xCF, 0x5E, 0xFC, 0x94, 0x06, 0x17, 0x3A, 0x5D, 0x2A, 0xAC, 0x91, 0xA4, 0xB3, 0x67, 0x22, 0xCD,
    0xC1, 0x5D, 0x3A, 0xD7, 0xEA, 0x7A, 0x77, 0x06, 0xFA, 0x80, 0x07, 0xEE, 0x4A, 0xA1, 0xDF, 0xB7,
    0x48, 0x45, 0x41, 0x44, 0xA6, 0x00, 0x00, 0x00, 0xFF, 0xFF, 0xFF, 0xFF, 0x01, 0xFA, 0x00, 0x02,
    0x64, 0x55, 0xA8, 0x97, 0xFF, 0x5F, 0xB8, 0xB8, 0x6C, 0x51, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x2B, 0x1F, 0xF3, 0x9F, 0xA9, 0x86, 0xD6, 0x29, 0x5D, 0x9D, 0xA9, 0x7A, 0x28, 0xB1, 0x56, 0x44,
    0x00, 0xF0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x72, 0xB6, 0x0B, 0xDC, 0x82, 0x29, 0x83, 0x31, 0x10, 0xAC, 0x82, 0x35, 0x69, 0xC8, 0x5F, 0x81,
    0x5B, 0x84, 0x06, 0x03, 0x5A, 0x99, 0x80, 0x06, 0x72, 0xE0, 0x35, 0xF3, 0x13, 0xB0, 0xBB, 0xD1,
    0xA0, 0x55, 0xAA, 0x78, 0xB5, 0x97, 0x38, 0x60, 0x0A, 0xEB, 0x1E, 0x50, 0x4E, 0xAB, 0xE4, 0x09,
    0x3B, 0xE6, 0x9F, 0x21, 0x7D, 0x5C, 0x69, 0xED, 0xC6, 0x08, 0xC9, 0x01, 0xDF, 0x96, 0x09, 0xC3,
    0x01, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xA6, 0x00, 0x00, 0x00,
    0x8A, 0x73, 0x4E, 0x7A, 0x0E, 0x70, 0x51, 0x7C, 0x16, 0x11, 0x90, 0xC6, 0xC1, 0x64, 0x6D, 0xE5,
    0xFA, 0x58, 0x38, 0x48, 0xED, 0x61, 0xA8, 0x76, 0x08, 0x95, 0x26, 0x73, 0x4F, 0x82, 0xC6, 0x60,
    0xEB, 0x92, 0xBE, 0xF2, 0x32, 0xB0, 0x3C, 0x81, 0xED, 0x9D, 0x34, 0xD1, 0x9B, 0xCC, 0xB3, 0x50,
    0xA9, 0x5A, 0x3B, 0xB0, 0x4A, 0x30, 0xB6, 0x0B, 0x6D, 0x80, 0x51, 0x4F, 0x4A, 0xEA, 0xB4, 0x39,
    0x11, 0x6A, 0x03, 0xB2, 0xC5, 0x52, 0x9A, 0x2E, 0xA0, 0xB2, 0x70, 0x30, 0xB2, 0xF5, 0x38, 0x37,
    0x1B, 0xCE, 0x5B, 0xAF, 0x1B, 0xEF, 0x53, 0xBC, 0x2E, 0x68, 0x3B, 0x5F, 0x87, 0xB9, 0x6C, 0x6B,
    0x77, 0x1A, 0xD7, 0x8F, 0xA0, 0xA1, 0x4D, 0x93, 0x5C, 0x91, 0x10, 0x71, 0x03, 0xDD, 0x57, 0xFC
};

//------------------------------------------------------------------------------
// 時間 取得
//------------------------------------------------------------------------------
nn::Result GetCurrentTime(nn::time::CalendarTime* calenderTime)
{
    std::string filePath(FileManager::GetInstance()->GetRootPath());
    filePath += "temp";
    nn::fs::DeleteFile(filePath.c_str());
    nn::fs::CreateFile(filePath.c_str(), 16);
    nn::fs::FileHandle file;
    auto result = nn::fs::OpenFile(&file, filePath.c_str(), (nn::fs::OpenMode::OpenMode_Write | nn::fs::OpenMode::OpenMode_AllowAppend));
    if(result.IsFailure())
    {
        NN_LOG("OpenFile() @ Create is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return result;
    }
    auto option = nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag::WriteOptionFlag_Flush);
    const char* temp = "TESTTEST";
    result = (nn::fs::WriteFile(file, 0, temp, 8, option));
    if(result.IsFailure())
    {
        NN_LOG("WriteFile() @ GetCurrentTime is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return result;
    }
    nn::fs::CloseFile(file);
    nn::fs::FileTimeStamp fileTimeStamp;
    result = nn::fs::GetFileTimeStampForDebug(&fileTimeStamp, filePath.c_str());
    if(result.IsFailure())
    {
        NN_LOG("GetFileTimeStampForDebug() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return result;
    }
    nn::fs::DeleteFile(filePath.c_str());

    result = nn::time::Initialize();
    if(result.IsFailure())
    {
        NN_LOG("Initialize() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return result;
    }
    nn::time::CalendarTime ct;
    nn::time::CalendarAdditionalInfo cai;
    result = nn::time::ToCalendarTime(&ct, &cai, fileTimeStamp.modify);
    if(result.IsFailure())
    {
        nn::time::Finalize();
        NN_LOG("ToCalendarTime() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return result;
    }
    *calenderTime = ct;
    nn::time::Finalize();
    NN_RESULT_SUCCESS;
}

//------------------------------------------------------------------------------
// 履歴ファイル出力
//------------------------------------------------------------------------------
nn::Result CreateFile(const char* filePath)
{
    nn::fs::FileHandle file;
    char outBuffer[128];
    memset(outBuffer, 0x00, sizeof(outBuffer));
    NN_RESULT_DO(nn::fs::CreateFile(filePath, 0));
    auto result = nn::fs::OpenFile(&file, filePath, (nn::fs::OpenMode::OpenMode_Write | nn::fs::OpenMode::OpenMode_AllowAppend));
    if(result.IsFailure())
    {
        NN_LOG("OpenFile() @ Create is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return result;
    }
    nn::util::Strlcpy(outBuffer, "Time,Result,Card Device ID,Param,File", sizeof(outBuffer) - 2);
    size_t strlen = nn::util::Strnlen(outBuffer, sizeof(outBuffer) - 2);
    // 改行(CR LF)
    outBuffer[strlen] = 0x0d;
    outBuffer[strlen + 1] = 0x0a;
    auto option = nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag::WriteOptionFlag_Flush);
    result = (nn::fs::WriteFile(file, 0, outBuffer, strlen + 2, option));
    if(result.IsFailure())
    {
        NN_LOG("WriteFile() @ Create is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return result;
    }
    nn::fs::CloseFile(file);
    NN_RESULT_SUCCESS;
}

void CovertMonthChar(char* pOutChar, uint8_t month)
{
    switch(month)
    {
        case 1:
            nn::util::Strlcpy(pOutChar, "Jan", 6);
            break;
        case 2:
            nn::util::Strlcpy(pOutChar, "Feb", 6);
            break;
        case 3:
            nn::util::Strlcpy(pOutChar, "Mar", 6);
            break;
        case 4:
            nn::util::Strlcpy(pOutChar, "Apr", 6);
            break;
        case 5:
            nn::util::Strlcpy(pOutChar, "May", 6);
            break;
        case 6:
            nn::util::Strlcpy(pOutChar, "Jun", 6);
            break;
        case 7:
            nn::util::Strlcpy(pOutChar, "Jul", 6);
            break;
        case 8:
            nn::util::Strlcpy(pOutChar, "Aug", 6);
            break;
        case 9:
            nn::util::Strlcpy(pOutChar, "Sep", 6);
            break;
        case 10:
            nn::util::Strlcpy(pOutChar, "Oct", 6);
            break;
        case 11:
            nn::util::Strlcpy(pOutChar, "Nov", 6);
            break;
        case 12:
            nn::util::Strlcpy(pOutChar, "Dec", 6);
            break;
        default:
            nn::util::Strlcpy(pOutChar, "Err", 6);
            break;
    }
}

size_t SetTimeBuffer(char* outBuffer, size_t offset)
{
    nn::time::CalendarTime ct;
    memset(&ct, 0x00, sizeof(nn::time::CalendarTime));
    size_t currentPos = offset;
    auto result = GetCurrentTime(&ct);
    if(result.IsFailure())
    {
        NN_LOG("GetCurrentTime() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        // ハイフンを入れる(1Byte)
        outBuffer[currentPos++] = 0x3d;
        // コンマを打つ(1Byte)
        outBuffer[currentPos++] = 0x2c;
        return currentPos;
    }

    // 日
    snprintf(outBuffer + currentPos, 4, "%02d", ct.day);
    currentPos += 2;

    // スペース
    outBuffer[currentPos++] = 0x20;

    // 月
    CovertMonthChar(outBuffer + currentPos, ct.month);
    currentPos += 3;

    // スペース
    outBuffer[currentPos++] = 0x20;

    // 年
    snprintf(outBuffer + currentPos, 8, "%04d", ct.year);
    currentPos += 4;

    // スペース
    outBuffer[currentPos++] = 0x20;

    // 時
    snprintf(outBuffer + currentPos, 4, "%02d", ct.hour);
    currentPos += 2;
    // :
    outBuffer[currentPos++] = 0x3a;
    // 分
    snprintf(outBuffer + currentPos, 4, "%02d", ct.minute);
    currentPos += 2;
    // :
    outBuffer[currentPos++] = 0x3a;
    // 秒
    snprintf(outBuffer + currentPos, 4, "%02d", ct.second);
    currentPos += 2;
    // UTC
    snprintf(outBuffer + currentPos, 10, "+0000");
    currentPos += 5;

    // コンマ
    outBuffer[currentPos++] = 0x2c;

    return currentPos;
}

size_t PrepareBuffer(char* outBuffer, size_t outBufferLength, bool isWrite, bool isSuccess, bool isCanceled)
{
    size_t currentPos = 0;
    currentPos = SetTimeBuffer(outBuffer, currentPos);

    if(isSuccess)
    {
        nn::util::Strlcpy(outBuffer + currentPos, "Success", outBufferLength - currentPos - 4);
        currentPos += 7;
    }
    else if(isCanceled)
    {
        nn::util::Strlcpy(outBuffer + currentPos, "Cancel", outBufferLength - currentPos - 4);
        currentPos += 6;
    }
    else
    {
        nn::util::Strlcpy(outBuffer + currentPos, "Failure", outBufferLength - currentPos - 4);
        currentPos += 7;
    }

    // コンマを打つ(1Byte)
    outBuffer[currentPos++] = 0x2c;

    // Device ID を入力(16Byte -> 32文字)
    for(int i = 0; i < nn::gc::GcCardDeviceIdSize; i++)
    {
        snprintf(outBuffer + currentPos, 4, "%02x", *(g_pDisplayData->GetDeviceId() + i));
        currentPos += 2;
    }
    // コンマを打つ(1Byte)
    outBuffer[currentPos++] = 0x2c;

    // 書き込み後なら書き込んだ情報を入れる
    if(isWrite)
    {
        // パラメータファイル名
        if(*(g_pDisplayData->GetParamPath()) != '\0' && g_pDisplayData->isParamExist)
        {
            size_t fileNameLength = nn::util::Strnlen(g_pDisplayData->GetParamPath(), outBufferLength - currentPos - 10);
            nn::util::Strlcpy(outBuffer + currentPos, g_pDisplayData->GetParamPath(), fileNameLength + 1);
            currentPos += fileNameLength;
        }
        else
        {
            nn::util::Strlcpy(outBuffer + currentPos, "None", outBufferLength - currentPos - 10);
            currentPos += 4;
        }
        // コンマを打つ(1Byte)
        outBuffer[currentPos++] = 0x2c;

        // 書き込みファイル名
        if(*(g_pDisplayData->GetFileName()) != '\0' && g_pDisplayData->isFileExist)
        {
            size_t fileNameLength = nn::util::Strnlen(g_pDisplayData->GetFileName(), outBufferLength - currentPos - 10);
            nn::util::Strlcpy(outBuffer + currentPos, g_pDisplayData->GetFileName(), fileNameLength + 1);
            currentPos += fileNameLength;
        }
    }
    else
    {
        nn::util::Strlcpy(outBuffer + currentPos, "Erase", outBufferLength - currentPos - 10);
        currentPos += 5;
        // コンマを打つ(1Byte)
        outBuffer[currentPos++] = 0x2c;
        // コンマを打つ(1Byte)
        outBuffer[currentPos++] = 0x2c;
    }

    // 改行(CR LF)
    outBuffer[currentPos++] = 0x0d;
    outBuffer[currentPos++] = 0x0a;

    return currentPos;
}

// ログを csv 出力する
nn::Result AddToLogFile(bool isWrite, bool isSuccess, bool isCanceled)
{
    std::string filePath(FileManager::GetInstance()->GetRootPath());
    filePath += "Log/";
    nn::fs::CreateDirectory(filePath.c_str());
#if defined(NN_XCIE_WRITER_XCIR_MODE) // XCIR は要望によりカードの Device ID ごとに csv を出力する
    const char DefaultExt[] = ".csv";
    // Device ID を入力(16Byte -> 32文字)
    for(int i = 0; i < nn::gc::GcCardDeviceIdSize; i++)
    {
        char temp[4];
        snprintf(temp, 4, "%02x", *(g_pDisplayData->GetDeviceId() + i));
        filePath += temp;
    }
    filePath += DefaultExt;
#else
    filePath += "XcieWriterLog.csv";
#endif
    NN_LOG("Log : %s\n", filePath.c_str());
    nn::Result result;
    // csv ファイルがなかったらファイル作成
    result = CreateFile(filePath.c_str());
    if(result.IsFailure() && !nn::fs::ResultPathAlreadyExists::Includes(result))
    {
        NN_LOG("CreateFile() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return result;
    }
    nn::fs::FileHandle file;
    result = nn::fs::OpenFile(&file, filePath.c_str(), (nn::fs::OpenMode::OpenMode_Write | nn::fs::OpenMode::OpenMode_AllowAppend));
    if(result.IsFailure())
    {
        NN_LOG("OpenFile() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return result;
    }
    int64_t fileSize;
    result = nn::fs::GetFileSize(&fileSize, file);
    if(result.IsFailure())
    {
        NN_LOG("GetFileSize() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return result;
    }
    char outBuffer[512];
    memset(outBuffer, 0x00, sizeof(outBuffer));

    size_t bufferSize = PrepareBuffer(outBuffer, sizeof(outBuffer), isWrite, isSuccess, isCanceled);
    auto option = nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag::WriteOptionFlag_Flush);
    result = nn::fs::WriteFile(file, fileSize, outBuffer, bufferSize, option);
    if(result.IsFailure())
    {
        NN_LOG("WriteFile() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return result;
    }
    nn::fs::CloseFile(file);
    NN_LOG("Write To Host Success\n");
    NN_RESULT_SUCCESS;
}

nn::Result AddToLogFile(bool isWrite, bool isSuccess)
{
    return AddToLogFile(isWrite, isSuccess, false);
}

//------------------------------------------------------------------------------
// Device ID 取得
//------------------------------------------------------------------------------
bool ConfirmCardRemoved()
{
    if(!nn::fs::IsGameCardInserted())
    {
        g_pDisplayData->cardState = CardState_NotInserted;
        return false;
    }
    return true;
}

void GetDeviceId()
{
    NN_LOG("Get Device ID\n");
    g_GraphicMain.SetGameStateWithTimeReset(CardState_Accessing);
    g_pDisplayData->ResetDeviceId();
    auto result = nn::fs::GetGameCardDeviceIdForProdCard(g_CardCert, sizeof(g_CardCert), g_CardHeaderForDev, sizeof(g_CardHeaderForDev));
    if(result.IsFailure())
    {
        NN_LOG("GetDeviceID is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        // Device ID の取得に失敗した場合はカードが壊れている
        g_pDisplayData->cardState = CardState_BrokenCard;
        ConfirmCardRemoved();
        return;
    }
    g_pDisplayData->cardState = CardState_ValidCard;
    g_pDisplayData->SetDeviceId(g_CardCert + 0x110, nn::gc::GcCardDeviceIdSize);
    NN_LOG("Device ID : ");
    PrintDebugArray(g_CardCert + 0x110, nn::gc::GcCardDeviceIdSize);
}

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
void* NvAllocate(size_t size, size_t alignment, void* userPtr)
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}

void NvFree(void* addr, void* userPtr)
{
    NN_UNUSED(userPtr);
    free(addr);
}

void* NvReallocate(void* addr, size_t newSize, void* userPtr)
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

//------------------------------------------------------------------------------
// 画面描画用スレッド
//------------------------------------------------------------------------------
nn::hid::TouchScreenState<1> g_PrevTouchScreenState;

bool IsTouchTrigger()
{
    if(g_PrevTouchScreenState.touches[0].x == 0 && g_PrevTouchScreenState.touches[0].y == 0)
    {
        return true;
    }
    return false;
}

/* ファイル名取得 */
// 画面表示ループから高い頻度で呼び出される、書込み可能なファイルのパス取得 API
void SetWritableFilePath()
{
    // nsp のパスが書かれたテキストファイルを読み込み
    FileManager* pFileManager = FileManager::GetInstance();
    size_t index = g_GraphicMain.GetCursorPos();
    char readBuffer[FilePathSize];
    // readBuffer には nsp のパスが入ってくる
    size_t readDataSize = pFileManager->ReadFile(readBuffer, sizeof(readBuffer), index);

    // パス名が長すぎるか、ファイルに何も書かれていない、ファイルが存在しない場合はここで終了
    if(readDataSize < 1 || readDataSize >= sizeof(readBuffer))
    {
        return;
    }
    readBuffer[readDataSize] = '\0';
    size_t actualLength = readDataSize;
    // 複数行に渡っている場合、 1 行目だけが有効
    for(size_t i = 0; i < readDataSize; i++)
    {
        if(readBuffer[i] == '\n' || readBuffer[i] == '\r')
        {
            actualLength = i;
            break;
        }
    }
    memset(readBuffer + actualLength, 0x00, sizeof(readBuffer) - 1 - actualLength);
    g_pDisplayData->SetFileName(readBuffer, actualLength);

    // readBuffer のパスにアクセスして xcie のカードヘッダハッシュだけ表示する
    nn::fs::FileHandle file;
    if(nn::fs::OpenFile(&file, readBuffer, nn::fs::OpenMode::OpenMode_Read).IsSuccess())
    {
        g_pDisplayData->isFileExist = true;
        int64_t fileSize;
        nn::fs::GetFileSize(&fileSize, file);
        g_pDisplayData->fileSize = fileSize;

        nn::fs::ReadFile(file, 0x1000, g_CardHeader, 0x200);
        nn::fs::CloseFile(file);

        char hash[32];
        nn::crypto::GenerateSha256Hash(hash, 32, g_CardHeader + 0x100, 0x100);
        g_pDisplayData->SetCardHeaderHash(hash, 32);

        // カードヘッダコピー
        nn::gc::detail::CardHeader cardHeader;
        memcpy(&cardHeader, g_CardHeader + 0x100, 0x100);

        // 書き込みサイズ設定
        g_pDisplayData->validDataSize = (cardHeader.validDataEndAddress + 1) * nn::gc::GcPageSize + KeyAreaLength;
    }
    else // テキストに書かれた xcie が存在しなかった
    {
        g_pDisplayData->isFileExist = false;
    }
    // sd カード内にパラメータ設定用のファイルが存在している場合
    if(*(g_pDisplayData->GetParamPath()) != '\0')
    {
        // パラメータファイルにアクセスしてパラメータを取得、グローバルに設定
        if(nn::fs::OpenFile(&file, g_pDisplayData->GetParamPath(), nn::fs::OpenMode::OpenMode_Read).IsSuccess())
        {
            int64_t fileSize;
            g_pDisplayData->isParamExist = true;
            nn::fs::GetFileSize(&fileSize, file);
            size_t readSize = (fileSize < 75) ? fileSize : 75;
            nn::fs::ReadFile(file, 0, g_DevCardDefaultParam, readSize);
            nn::fs::CloseFile(file);
        }
        else // テキストに書かれた パラメータファイルが存在しなかった
        {
            NN_LOG("[error] Param Path Doesnot exist!!\n");
            g_pDisplayData->isParamExist = false;
        }
    }
}

/* メニューに戻るときだけイベント待ちが必要 */
void MoveToMenu()
{
    g_GraphicMain.TransitToMenuMode();
    nn::os::WaitEvent(&g_MoveToMenuEvent);
    g_pDisplayData->currentWrittenSize = 0;
    SetWritableFilePath();
}

/* ボタン別の処理 */
void SetByButton()
{
    if(g_GraphicMain.IsExecuteButtonPushed() && IsTouchTrigger())
    {
        g_GraphicMain.TransitToScriptMode();
        nn::os::SignalEvent(&g_ExecuteEvent);
    }
    else if(g_GraphicMain.IsEraseButtonPushed() && IsTouchTrigger())
    {
        g_GraphicMain.SetGameStateWithTimeReset(CardState_Repairing);
        nn::os::SignalEvent(&g_ExecuteEvent);
    }
    else if(g_GraphicMain.IsBackButtonPushed() && IsTouchTrigger())
    {
        MoveToMenu();
    }
    else if(g_GraphicMain.IsIncreasePageCountButtonPushed() && IsTouchTrigger())
    {
        g_GraphicMain.IncreasePageCounter();
    }
    else if(g_GraphicMain.IsDecreasePageCountButtonPushed() && IsTouchTrigger())
    {
        g_GraphicMain.DecreasePageCounter();
    }
    else if(g_GraphicMain.IsWriteCancelButtonPushed() && IsTouchTrigger())
    {
        if(g_pDisplayData->isWriteCancelConfirm)
        {
            g_pDisplayData->isWriteCancelConfirm = false;
            g_GraphicMain.TransitToScriptEndMode();
            MoveToMenu();
        }
        else
        {
            g_pDisplayData->isWriteCancelConfirm = true;
        }
    }
    else if(g_GraphicMain.IsModeChangeButtonPushed() && IsTouchTrigger())
    {
        g_GraphicMain.ChangeMode();
    }
}

/* 画面描画スレッド */
void RunThreadFunction(void* pThreadNum)
{
    NN_LOG("Thread Start\n");
    nn::hid::TouchScreenState<1> touchScreenState;
    size_t counter = 0;

    while(FileManager::GetInstance()->Mount().IsFailure())
    {
        g_pDisplayData->isMountError = true;
        g_GraphicMain.Execute();
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(1));
    }
    g_pDisplayData->isMountError = false;

    while(1)
    {
        if(nn::os::TimedWaitEvent(&g_ThreadEndEvent, nn::TimeSpan::FromMilliSeconds(1)))
        {
            break;
        }
        if(counter % 30 == 0)
        {
            counter = 0;
            if(g_GraphicMain.IsMenuMode())
            {
                FileManager::GetInstance()->Initialize();
                SetWritableFilePath();
            }
        }
        nn::hid::GetTouchScreenState(&touchScreenState);
        g_GraphicMain.SetTouchScreenState(touchScreenState.touches[0].x, touchScreenState.touches[0].y);
        // 画面表示実行
        g_GraphicMain.Execute();
        // ボタンごとの処理
        SetByButton();
        g_PrevTouchScreenState = touchScreenState;
        counter++;
    }
}

//------------------------------------------------------------------------------
// gc Writeスレッド
//------------------------------------------------------------------------------

/* 書き込みスレッド */
void RunWriteThread(void* pWritePageInfo)
{
    nn::os::MultiWaitType waiter;
    nn::os::InitializeMultiWait(&waiter);
    nn::os::MultiWaitHolderType startEventHolder;
    nn::os::MultiWaitHolderType finishEventHolder;
    nn::os::InitializeMultiWaitHolder(&startEventHolder, &g_WriteStartEvent);
    nn::os::InitializeMultiWaitHolder(&finishEventHolder, &g_WriteFinishEvent);
    nn::os::LinkMultiWaitHolder(&waiter, &startEventHolder);
    nn::os::LinkMultiWaitHolder(&waiter, &finishEventHolder);
    while(1)
    {
        auto signaledHolder = nn::os::WaitAny(&waiter);
        if(signaledHolder == &finishEventHolder)
        {
            nn::os::ClearEvent(&g_WriteFinishEvent);
            nn::os::UnlinkAllMultiWaitHolder(&waiter);
            nn::os::FinalizeMultiWaitHolder(&finishEventHolder);
            nn::os::FinalizeMultiWaitHolder(&startEventHolder);
            nn::os::FinalizeMultiWait(&waiter);
            break;
        }
        else if(signaledHolder == &startEventHolder)
        {
            //8MB より小さい場合は 0xFF でパディングしてみる
            if(g_WritePageInfo.byteSize < WriteUnitSize)
            {
                memset(g_pDataBuffer + g_WritePageInfo.byteSize, 0xff, WriteUnitSize - g_WritePageInfo.byteSize);
            }
            NN_LOG("%lld / %lld (%.1llf%)", g_WritePageInfo.byteOffset / (1024 * 1024), g_pDisplayData->validDataSize / (1024 * 1024), (static_cast<double>(g_WritePageInfo.byteOffset) / g_pDisplayData->validDataSize) * 100.0);
            auto result = nn::fs::WriteToGameCard(g_WritePageInfo.byteOffset, g_pDataBuffer, WriteUnitSize);
            if(result.IsFailure())
            {
                NN_LOG("Failure\n nn::fs::WriteToGameCard() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
                g_pDisplayData->cardState = CardState_UnKnownError;
                ConfirmCardRemoved();
            }
            NN_LOG("   OK\n");
            nn::os::SignalEvent(&g_WriteEndEvent);
            nn::os::ClearEvent(&g_WriteStartEvent);
        }
        else
        {
            NN_ABORT("Write Thread catched unexpected event.\n");
        }
    }
}

//------------------------------------------------------------------------------
//  スレッド生成関連の処理
//------------------------------------------------------------------------------
void InitializeThread()
{
    NN_LOG("Initialize Thread\n");
    nn::os::InitializeEvent(&g_ThreadEndEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&g_ExecuteEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&g_MoveToMenuEvent, false, nn::os::EventClearMode_AutoClear);

    nn::Result result = nn::os::CreateThread(&g_Thread, RunThreadFunction, &g_Thread, g_ThreadStack, ThreadStackSize, ThreadPriority);
    if(!result.IsSuccess())
    {
        NN_ABORT("failed to create a thread\n");
    }
    nn::os::SetThreadNamePointer(&g_Thread, g_ThreadName);
    nn::os::StartThread(&g_Thread);
}
void FinalizeThread()
{
    // スレッドの終了
    nn::os::WaitThread(&g_Thread);
    nn::os::DestroyThread(&g_Thread);
    nn::os::FinalizeEvent(&g_ThreadEndEvent);
    nn::os::FinalizeEvent(&g_ExecuteEvent);
    nn::os::FinalizeEvent(&g_MoveToMenuEvent);
}

void InitializeWriteThread()
{
    nn::os::InitializeEvent(&g_WriteStartEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&g_WriteEndEvent, false, nn::os::EventClearMode_AutoClear);
    nn::os::InitializeEvent(&g_WriteFinishEvent, false, nn::os::EventClearMode_AutoClear);
    // 待ちイベントの初期設定
    nn::Result result = nn::os::CreateThread(&g_WriteThread, RunWriteThread, &g_WriteThread, g_WriteThreadStack, ThreadStackSize, ThreadPriority);
    if(!result.IsSuccess())
    {
        NN_ABORT("failed to create a thread\n");
    }
    nn::os::SetThreadNamePointer(&g_WriteThread, g_ThreadName2);
    nn::os::StartThread(&g_WriteThread);
}

void FinalizeWriteThread()
{
    nn::os::SignalEvent(&g_WriteFinishEvent);
    nn::os::WaitThread(&g_WriteThread);
    nn::os::DestroyThread(&g_WriteThread);
    nn::os::FinalizeEvent(&g_WriteStartEvent);
    nn::os::FinalizeEvent(&g_WriteEndEvent);
    nn::os::FinalizeEvent(&g_WriteFinishEvent);
}

//------------------------------------------------------------------------------
//  アプリの初期化関連の処理
//------------------------------------------------------------------------------

void InitializeGraphic()
{
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    // グラフィックスシステムのためのメモリ周りの初期化を行います。
    {
        const size_t GraphicsSystemMemorySize = 8 * 1024 * 1024;
        nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
        nv::InitializeGraphics(malloc(GraphicsSystemMemorySize), GraphicsSystemMemorySize);
    }
#endif
}

void InitializeHid()
{
    nn::hid::InitializeTouchScreen();
    nn::hid::InitializeDebugPad();
}

//------------------------------------------------------------------------------
//  書き込み実行部分
//------------------------------------------------------------------------------

bool ReadWriteImpl(nn::fs::FileHandle& file, int64_t unitIndex, size_t readWriteSize)
{
    int64_t byteOffset = unitIndex * WriteUnitSize;

    memset(g_pReadBufferFromFs, 0, readWriteSize);
    auto result = nn::fs::ReadFile(file, byteOffset, g_pReadBufferFromFs, readWriteSize);
    if(result.IsFailure())
    {
        NN_LOG("nn::fs::ReadFile() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        g_pDisplayData->isFileExist = false;
    }
    if(unitIndex == 0)
    {
        // 初回だけEncrypted Area を上書き
        if(g_pDisplayData->IsWritableFile())
        {
            memcpy(g_pReadBufferFromFs + EncryptedAreaStartAddress, g_EncryptedArea, EncryptedAreaLength);
            for(int i = 0; i < EncryptedKeyAreaLength; i++)
            {
                *(g_pReadBufferFromFs + EncryptedAreaStartAddress + EncryptedAreaLength + i) = i;
            }
        }
        // 初回だけ証明書を上書きしないように GetDeviceId で取得した証明書で上書き
        memcpy(g_pReadBufferFromFs + 0x40 * nn::gc::GcPageSize, g_CardCert, nn::gc::GcPageSize);
    }
    else
    {
        // 初回以外は前の WriteThread が残っている
        nn::os::WaitEvent(&g_WriteEndEvent);
        if(!g_pDisplayData->isFileExist)
        {
            NN_LOG("[error] File on PC is not Accessible.");
            return false;
        }
        if(!g_pDisplayData->IsCardWritable())
        {
            NN_LOG("[error] Card state : %d\n", static_cast<int>(g_pDisplayData->cardState));
            return false;
        }
    }
    memcpy(g_pDataBuffer, g_pReadBufferFromFs, readWriteSize);
    g_WritePageInfo.byteOffset = byteOffset;
    g_WritePageInfo.byteSize = readWriteSize;
    nn::os::SignalEvent(&g_WriteStartEvent);
    return true;
}

uint32_t GetAndSetRomSize(nn::gc::detail::DevCardParameter* pDevCardParam)
{
    uint32_t romSize = 0;
    switch(pDevCardParam->cardId1.memoryCapacity)
    {
        case nn::gc::detail::MemoryCapacity_1GB:
            pDevCardParam->romSize = nn::gc::detail::DevCardRomSize_1GB;
            romSize = 1;
            break;
        case nn::gc::detail::MemoryCapacity_2GB:
            pDevCardParam->romSize = nn::gc::detail::DevCardRomSize_2GB;
            romSize = 2;
            break;
        case nn::gc::detail::MemoryCapacity_4GB:
            pDevCardParam->romSize = nn::gc::detail::DevCardRomSize_4GB;
            romSize = 4;
            break;
        case nn::gc::detail::MemoryCapacity_8GB:
            pDevCardParam->romSize = nn::gc::detail::DevCardRomSize_8GB;
            romSize = 8;
            break;
        case nn::gc::detail::MemoryCapacity_16GB:
            pDevCardParam->romSize = nn::gc::detail::DevCardRomSize_16GB;
            romSize = 16;
            break;
        case nn::gc::detail::MemoryCapacity_32GB:
            pDevCardParam->romSize = nn::gc::detail::DevCardRomSize_32GB;
            romSize = 32;
            break;
        default:
            NN_LOG("[error] Memory Capacity = %x is wrong!\n", static_cast<uint8_t>(pDevCardParam->cardId1.memoryCapacity));
            g_pDisplayData->cardState = CardState_InvalidCardData;
            romSize = 0;
    }
    NN_LOG("romSize = %lld\n", romSize);
    return romSize;
}

bool CanWriteAndVerifyParam(nn::gc::detail::DevCardParameter* pDevCardParam)
{
    // Parameter 書き込み
    auto result = nn::fs::EraseAndWriteParamDirectly(pDevCardParam, sizeof(nn::gc::detail::DevCardParameter));
    if(result.IsFailure())
    {
        NN_LOG("nn::fs::EraseAndWriteParamDirectly() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        g_pDisplayData->cardState = CardState_InvalidCardType;
        return false;
    }
    nn::gc::detail::DevCardParameter readDevCardParam;
    // parameter Verify
    result = nn::fs::ReadParamDirectly(&readDevCardParam, sizeof(nn::gc::detail::DevCardParameter));
    if(result.IsFailure())
    {
        NN_LOG("nn::fs::ReadParamDirectly() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        g_pDisplayData->cardState = CardState_InvalidCardType;
        return false;
    }
    //    NN_LOG("Parameter...\n");
    //    PrintDebugArray(reinterpret_cast<char*>(&readDevCardParam), sizeof(nn::gc::detail::DevCardParameter));
    if(memcmp(pDevCardParam, &readDevCardParam, 73) != 0)
    {
        NN_LOG("Param Write comp Error!\n");
        g_pDisplayData->cardState = CardState_InvalidCardType;
        return false;
    }
    return true;
}

bool SetDevCardParameter()
{
    // カードヘッダコピー
    nn::gc::detail::CardHeader cardHeader;
    memcpy(&cardHeader, g_CardHeader + 0x100, 0x100);

    nn::gc::detail::DevCardParameter devCardParam;
    // param が設定されていなければ 0x00 埋め
    memcpy(&devCardParam, g_DevCardDefaultParam, sizeof(g_DevCardDefaultParam));
    if(g_pDisplayData->isParamExist)
    {
        NN_LOG("Param File: %s\n", g_pDisplayData->GetParamPath());
    }
    // 書き込みサイズ設定
    g_pDisplayData->validDataSize = (cardHeader.validDataEndAddress + 1) * nn::gc::GcPageSize + KeyAreaLength;

    // readOnly な param の設定
    nn::gc::detail::DevCardParameter readDevCardParam;
    auto result = nn::fs::ReadParamDirectly(&readDevCardParam, sizeof(nn::gc::detail::DevCardParameter));
    if(result.IsFailure())
    {
        NN_LOG("nn::fs::ReadParamDirectly() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        g_pDisplayData->cardState = CardState_InvalidCardType;
        return false;
    }
    devCardParam.cardId2.cardType = readDevCardParam.cardId2.cardType;
    devCardParam.nandSize = readDevCardParam.nandSize;

    // Rom 容量コピー
    devCardParam.cardId1.memoryCapacity = static_cast<nn::gc::detail::MemoryCapacity>(cardHeader.romSize);
    NN_LOG("MemoryCapacity : %x\n", static_cast<uint8_t>(devCardParam.cardId1.memoryCapacity));

    // memoryType 設定
    unsigned char memoryTypeValue = 0x20;
    devCardParam.backupAreaStartAddr = 0xFFFFFFFF;
    if(cardHeader.backupAreaStartPageAddress != 0xFFFFFFFF)  // R/W Type
    {
        memoryTypeValue |= 0x08;
        devCardParam.backupAreaStartAddr = cardHeader.backupAreaStartPageAddress / 64; // 32KB 単位
        NN_LOG("[warning] backupArea Set : 0x%llx. CONFIRM this application is R/W Type.\n", devCardParam.backupAreaStartAddr);
    }

    if(cardHeader.selSec == 0x01)  //T1
    {
        memoryTypeValue |= 0x01;
        memcpy(&(devCardParam.cardId2.cardSecurityNumber), &(cardHeader.selT1Key), 1);
        NN_LOG("T1 : securityNumber = %x\n", static_cast<uint8_t>(devCardParam.cardId2.cardSecurityNumber));
    }
    else if(cardHeader.selSec == 0x02)  //T2
    {
        memoryTypeValue |= 0x02;
        memcpy(&(devCardParam.cardId2.cardSecurityNumber), &(cardHeader.selKey), 1);
        NN_LOG("T2 : CertNumber = %x\n", static_cast<uint8_t>(devCardParam.cardId2.cardSecurityNumber));
    }
    else
    {
        NN_LOG("[error] SEL_SEC = 0x%x is wrong!\n", cardHeader.selSec);
        g_pDisplayData->cardState = CardState_InvalidCardData;
        return false;
    }

    memcpy(&(devCardParam.cardId1.memoryType), &memoryTypeValue, 1);
    NN_LOG("Memory Type = %x\n", memoryTypeValue);

    // RomAreaStartAddr 設定
    devCardParam.romAreaStartAddr = cardHeader.limArea / 64; // 32KB 単位

    // reserveAreaStartAddr 設定
    const uint32_t ReserveAddrSizeUnit = 8 * 1024 * 1024;
    uint32_t reservedAreaStartAddr = GetAndSetRomSize(&devCardParam) * nn::gc::detail::AvailableSizeBase / ReserveAddrSizeUnit;
    if(reservedAreaStartAddr == 0)
    {
        return false;
    }
    memcpy(devCardParam.reserveAreaStartAddr, &reservedAreaStartAddr, sizeof(devCardParam.reserveAreaStartAddr));

    return CanWriteAndVerifyParam(&devCardParam);
}

bool CanPrepareGameCardWrite()
{
    if(!ConfirmCardRemoved())
    {
        return false;
    }
    // Activate は GetDeviceID 終了時点でされているためこのまま
    return SetDevCardParameter();
}

bool ReadWrite(nn::fs::FileHandle& file, int64_t unitIndex, size_t readWriteSize)
{
    if(!ReadWriteImpl(file, unitIndex, readWriteSize))
    {
        ConfirmCardRemoved();
        return false;
    }
    g_pDisplayData->currentWrittenSize += readWriteSize;
    return true;
}

void FinishAndClose(nn::fs::FileHandle file)
{
    FinalizeWriteThread();
    nn::fs::CloseFile(file);
}

bool WriteFile()
{
    if(!(g_pDisplayData->IsWritableFile()) && !(g_pDisplayData->IsXciFile()))
    {
#if defined(NN_XCIE_WRITER_XCIE_MODE)
        NN_LOG("file must be 'xcie'\n");
#elif defined(NN_XCIE_WRITER_XCIR_MODE)
        NN_LOG("file must be 'xcir'\n");
#else
        NN_LOG("file must be 'xci'\n");
        return false;
#endif
    }
    if(!ConfirmCardRemoved())
    {
        NN_LOG("Card is not inserted\n");
    }

    nn::fs::FileHandle file;
    auto result = nn::fs::OpenFile(&file, g_pDisplayData->GetFileName(), nn::fs::OpenMode::OpenMode_Read);
    if(result.IsFailure())
    {
        NN_LOG("nn::fs::OpenFile() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return false;
    }
    g_pDisplayData->isFileExist = true; // OpenFile に成功しているので file は存在している

    nn::fs::ReadFile(file, 0x1000, g_CardHeader, 0x200);
    NN_LOG("Write Start %s\n", g_pDisplayData->GetFileName());

    // 要復号化ファイルなら復号化
    if(g_pDisplayData->IsWritableFile())
    {
        char keyAreaBuffer[EncryptedKeyAreaLength];
        nn::fs::ReadFile(file, EncryptedAreaStartAddress + EncryptedAreaLength, keyAreaBuffer, EncryptedKeyAreaLength);
        nn::fs::ReadFile(file, EncryptedAreaStartAddress, g_pReadBufferFromFs, EncryptedAreaLength);
        NN_ABORT_UNLESS(DecryptXcieToXci(g_EncryptedArea, EncryptedAreaLength, g_pReadBufferFromFs, EncryptedAreaLength, keyAreaBuffer, EncryptedKeyAreaLength).IsSuccess(), "Decrypt Xcie Failed.Abort\n");
    }

    // 書き込み準備
    if(!CanPrepareGameCardWrite())
    {
        ConfirmCardRemoved();
        return false;
    }
    InitializeWriteThread();
    g_pDisplayData->currentWrittenSize = 0;
    int64_t fileSize = g_pDisplayData->validDataSize;
    int64_t maxUnitNum = fileSize / WriteUnitSize;
    for(int64_t i = 0; i < maxUnitNum; i++)
    {
        // メニューモードならキャンセル済み
        if(g_GraphicMain.IsMenuMode() || !ReadWrite(file, i, WriteUnitSize))
        {
            FinishAndClose(file);
            return false;
        }
    }
    int64_t restSize = fileSize % WriteUnitSize;
    if(g_GraphicMain.IsMenuMode() || (restSize != 0 && !ReadWrite(file, maxUnitNum, restSize)))
    {
        FinishAndClose(file);
        return false;
    }

    // 最後の Write 終了待ち
    nn::os::WaitEvent(&g_WriteEndEvent);
    g_pDisplayData->currentWrittenSize = fileSize;

    FinishAndClose(file);
    NN_LOG("Write Finished\n");
    return true;
}

//------------------------------------------------------------------------------
// Parameter ファイルのパス取得
//------------------------------------------------------------------------------
nn::Result SetDefaultParamPath()
{
    nn::Result result = nn::fs::MountSdCardForDebug("sdcard");
    if(result.IsFailure())
    {
        NN_LOG("Mount Sdcard Failure\n");
        return result;
    }

    nn::fs::FileHandle fileHandle;
    result = nn::fs::OpenFile(&fileHandle, "sdcard:/paramPath.txt", nn::fs::OpenMode_Read);
    if(result.IsFailure())
    {
        NN_LOG("nn::fs::OpenFile() is failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        nn::fs::Unmount("sdcard");
        return result;
    }
    NN_LOG("[warning] If you don't want to Set Parameter, Remove SD card and Reboot Application!\n");
    int64_t fileSize;
    nn::fs::GetFileSize(&fileSize, fileHandle);
    // ファイルサイズがパスの許容サイズより長かった場合 ABORT
    if(fileSize < 1 || fileSize >= FilePathSize)
    {
        NN_ABORT("sdcard:/paramPath.txt : Size is too large. (file size is %lld, but file size must be lower than %d)\n", fileSize, FilePathSize);
    }
    memset(g_pReadBufferFromFs, 0x00, DataBufferSize);
    result = nn::fs::ReadFile(fileHandle, 0, g_pReadBufferFromFs, fileSize);
    nn::fs::CloseFile(fileHandle);
    nn::fs::Unmount("sdcard");
    if(result.IsFailure())
    {
        NN_LOG("ReadFile failure(Module:%d, Description:%d)\n", result.GetModule(), result.GetDescription());
        return result;
    }

    // SetFilePath
    g_pReadBufferFromFs[fileSize] = '\0';
    size_t actualLength = fileSize;
    // 複数行に渡っている場合、 1 行目だけが有効
    for(size_t i = 0; i < fileSize; i++)
    {
        if(g_pReadBufferFromFs[i] == '\n' || g_pReadBufferFromFs[i] == '\r')
        {
            actualLength = i;
            break;
        }
    }
    // 不要な部分を 0x00 埋め
    memset(g_pReadBufferFromFs + actualLength, 0x00, FilePathSize - 1 - actualLength);
    nn::util::Strlcpy(g_pDisplayData->GetParamPath(), g_pReadBufferFromFs, FilePathSize);
    *(g_pDisplayData->GetParamPath() + actualLength) = '\0';
    NN_RESULT_SUCCESS;
}

void DirectWrite(char* filePath, char* paramPath)
{
    if(FileManager::GetInstance()->Mount().IsFailure())
    {
        NN_LOG("Mount PC Failure\n");
        return;
    }

    if(!nn::fs::IsGameCardInserted())
    {
        NN_LOG("GameCard not Inserted\n");
        return;
    }

    GetDeviceId();

    // パラメータファイルにアクセスしてパラメータを取得、グローバルに設定
    if(paramPath != nullptr)
    {
        nn::fs::FileHandle file;
        if(nn::fs::OpenFile(&file, paramPath, nn::fs::OpenMode::OpenMode_Read).IsSuccess())
        {
            nn::util::Strlcpy(g_pDisplayData->GetParamPath(), paramPath, FilePathSize);
            int64_t fileSize;
            g_pDisplayData->isParamExist = true;
            nn::fs::GetFileSize(&fileSize, file);
            size_t readSize = (fileSize < 75) ? fileSize : 75;
            nn::fs::ReadFile(file, 0, g_DevCardDefaultParam, readSize);
            nn::fs::CloseFile(file);
        }
        else
        {
            NN_LOG("param file does not exist\n");
            return;
        }
    }
    g_GraphicMain.TransitToScriptMode();
    g_pDisplayData->SetFileName(filePath, 500);
    if(!WriteFile())
    {
        NN_LOG("Write Failure\n");
        return;
    }
    NN_LOG("Write Success\n");
    return;
}

void DirectWrite(char* filePath)
{
    DirectWrite(filePath, nullptr);
}

//------------------------------------------------------------------------------
//  メイン
//------------------------------------------------------------------------------

void MainLoop(const char* filePath)
{
    nn::os::MultiWaitType waiter;
    nn::os::InitializeMultiWait(&waiter);
    nn::os::MultiWaitHolderType executeEventHolder;
    nn::os::MultiWaitHolderType cardDetectEventHolder;
    nn::os::InitializeMultiWaitHolder(&executeEventHolder, &g_ExecuteEvent);
    nn::os::InitializeMultiWaitHolder(&cardDetectEventHolder, &g_CardDetectionEvent);
    nn::os::LinkMultiWaitHolder(&waiter, &executeEventHolder);
    nn::os::LinkMultiWaitHolder(&waiter, &cardDetectEventHolder);
    bool isDirectWrite = false;

    // 引数有りの場合はもう始める
    if(filePath[0] != '\0')
    {
        NN_LOG("Direct Write Mode\n");
        // カード挿入状態での開始以外許さない
        if(!nn::fs::IsGameCardInserted())
        {
            NN_LOG("[ERROR] GameCard is NOT inserted !!\nTry Again After Inserting GameCard.\n");
            return;
        }
        isDirectWrite = true;
        g_GraphicMain.TransitToScriptMode();
        g_pDisplayData->SetFileName(filePath, 500);
        nn::os::SignalEvent(&g_ExecuteEvent);
    }

    while(1)
    {
        auto signaledHolder = nn::os::WaitAny(&waiter);
        if(signaledHolder == &executeEventHolder)
        {
            auto tick = nn::os::GetSystemTick();
            // Repair モードなら修理が走る
            if(g_pDisplayData->cardState == CardState_Repairing && !isDirectWrite)
            {
                auto result = nn::fs::ForceEraseGameCard();
                GetDeviceId();
                AddToLogFile(false, result.IsSuccess());
            }
            // それ以外はファイル書き込み
            else
            {
                bool isWriteSuccess = WriteFile();
                if(isWriteSuccess)
                {
                    NN_LOG("Write Time = %llds\n", (nn::os::GetSystemTick() - tick).ToTimeSpan().GetSeconds());
                    g_GraphicMain.TransitToScriptEndMode();
                }
                else
                {
                    // メニューモードならキャンセルされた
                    if(g_GraphicMain.IsMenuMode())
                    {
                        NN_LOG("Write Canceled\n");
                    }
                    else
                    {
                        NN_LOG("Write Failure\n");
                    }
                }
                AddToLogFile(true, isWriteSuccess, g_GraphicMain.IsMenuMode());
            }

            nn::os::ClearEvent(&g_ExecuteEvent);
            nn::os::SignalEvent(&g_MoveToMenuEvent);
            if(isDirectWrite)
            {
                return;
            }
        }
        else if(signaledHolder == &cardDetectEventHolder)
        {
            if(nn::fs::IsGameCardInserted())
            {
                GetDeviceId();
            }
            else
            {
                g_pDisplayData->cardState = CardState_NotInserted;
            }
            nn::os::ClearSystemEvent(&g_CardDetectionEvent);
        }
        else
        {
            NN_ABORT("Unexpected Event catched!\n");
        }
    }

    // waiter 終了
    nn::os::UnlinkAllMultiWaitHolder(&waiter);
    nn::os::FinalizeMultiWaitHolder(&cardDetectEventHolder);
    nn::os::FinalizeMultiWaitHolder(&executeEventHolder);
    nn::os::FinalizeMultiWait(&waiter);
}

extern "C" void nnMain()
{
    // ABORT 殺す
    nn::fs::SetEnabledAutoAbort(false);
    nn::ns::Initialize();
    nn::ns::GameCardStopper stopper;
    nn::ns::GetGameCardStopper(&stopper);
    Option option(nn::os::GetHostArgc(), nn::os::GetHostArgv());
    char nspPath[MaxFilePathLength] = {}; // TORIAEZU

    NN_LOG("---- given parameter ---\n");
    // 入力の処理
    if(option.HasKey("-file")) //書き込むファイル
    {
        nn::util::Strlcpy(nspPath, option.GetValue("-file"), MaxFilePathLength);
        NN_LOG("File Path : %s\n", nspPath);
    }

    if(option.HasKey("-root"))
    {
        NN_LOG("Root path : %s\n", option.GetValue("-root"));
        FileManager::GetInstance()->SetRootPath(option.GetValue("-root"));
    }

    // -param が指定されていたら param はそこを設定
    if(option.HasKey("-param"))
    {
        // パラメータファイルにアクセスしてパラメータを取得、グローバルに設定
        nn::fs::FileHandle file;
        NN_LOG("Parameter : %s\n", option.GetValue("-param"));
        if(nn::fs::OpenFile(&file, option.GetValue("-param"), nn::fs::OpenMode::OpenMode_Read).IsSuccess())
        {
            nn::util::Strlcpy(g_pDisplayData->GetParamPath(), option.GetValue("-param"), FilePathSize);
            int64_t fileSize;
            g_pDisplayData->isParamExist = true;
            nn::fs::GetFileSize(&fileSize, file);
            size_t readSize = (fileSize < 75) ? fileSize : 75;
            if(nn::fs::ReadFile(file, 0, g_DevCardDefaultParam, readSize).IsFailure())
            {
                NN_LOG("[ERROR] param file read error\n");
                return;
            }
            nn::fs::CloseFile(file);
        }
        else
        {
            NN_LOG("[ERROR] param file does not exist\n");
            return;
        }
    }
    // -param が指定されておらず SD カードが刺さっている場合はデフォルトの param を参照
    else if(nn::fs::IsSdCardInserted() && SetDefaultParamPath().IsSuccess())
    {
        NN_LOG("Parameter : %s\n", g_pDisplayData->GetParamPath());
    }
    else
    {
        NN_LOG("Parameter : Default\n");
    }
    memset(g_CardHeader, 0x00, sizeof(g_CardHeader));
    memset(g_DevCardDefaultParam, 0x00, sizeof(g_DevCardDefaultParam));

    std::unique_ptr<char[]> readBuffer(new char[DataBufferSize]);
    std::unique_ptr<char[]> writeBuffer(new char[DataBufferSize]);
    g_pReadBufferFromFs = readBuffer.get();
    g_pDataBuffer = writeBuffer.get();

    InitializeGraphic();

    g_GraphicMain.Initialize(ApplicationHeapSize);
    InitializeThread();
    NN_LOG("Initialize Hid\n");
    InitializeHid();

    if(nn::fs::IsGameCardInserted())
    {
        GetDeviceId();
    }
    else if(option.HasKey("-file"))
    {
        // カード挿入状態での開始以外許さない
        NN_LOG("[ERROR] GameCard is NOT inserted !!\nTry Again After Inserting GameCard.\n");
        return;
    }

    if(nn::fs::OpenGameCardDetectionEventNotifier(&g_Notifier).IsFailure())
    {
        NN_ABORT("failed to open notifier.");
    }
    if(g_Notifier->BindEvent(&g_CardDetectionEvent, nn::os::EventClearMode_AutoClear).IsFailure())
    {
        NN_ABORT("failed to bind system event to notifier.\n");
    }
    NN_LOG("---- Start ---\n");
    MainLoop(nspPath);
    // 画面描画終了
    nn::os::SignalEvent(&g_ThreadEndEvent);
    FinalizeThread();
    return;
}
