﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once
#include <cstdio>
#include <cstring>

#include <nn/nn_Result.h>
#include <nn/nn_Log.h>

#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Code.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/fs/fs_Result.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Directory.h>

#include "DisplayData.h"

class MountPath
{
    friend class FileAccessor;
    std::string m_MountName;
    std::string m_TargetPath;

public:
    MountPath(std::string mountName = "", std::string targetPath = "")
    {
        UpdatePath(mountName, targetPath);
    }

    void UpdatePath(std::string mountName, std::string targetPath)
    {
        m_MountName = mountName;
        m_TargetPath = targetPath;
    }

    const std::string GetMountPath(std::string path = "")
    {
        return (m_MountName + ":/" + path);
    }
};

class FileAccessor
{
private:
    FileAccessor();
    bool m_IsMounted;

public:
    static MountPath defaultMountPath;

    static FileAccessor &GetInstance()
    {
        static FileAccessor FileAccessor;
        return FileAccessor;
    }
    nn::Result Mount();
    void Unmount();

    // Open して Read/Write, Close する
    void Read(char* outBuffer, size_t* outReadLength, const size_t bufferLength, const char* filePath);
    void Read(char* outBuffer, size_t* outReadLength, const size_t offset, const size_t bufferLength, const char* filePath);
    void Write(const char* buffer, const size_t bufferLength, const char* filePath);

    nn::Result Open(nn::fs::FileHandle* handle, const char* filePath, int64_t fileSize, nn::fs::OpenMode mode);

    nn::Result GetDirectoryList(nn::fs::DirectoryEntry* outDirectoryEntryBuffer, int64_t* outDirectoryEntryCount, const uint32_t directoryEntryBufferLength, const char* directoryPath)
    {
        return GetDirectoryOrFileList(outDirectoryEntryBuffer, outDirectoryEntryCount, directoryEntryBufferLength, directoryPath, nn::fs::OpenDirectoryMode_Directory);
    }
    nn::Result GetFileList(nn::fs::DirectoryEntry* outDirectoryEntryBuffer, int64_t* outDirectoryEntryCount, const uint32_t directoryEntryBufferLength, const char* directoryPath)
    {
        return GetDirectoryOrFileList(outDirectoryEntryBuffer, outDirectoryEntryCount, directoryEntryBufferLength, directoryPath, nn::fs::OpenDirectoryMode_File);
    }
    bool IsDirectoryExisted(nn::fs::DirectoryEntry* workDirectoryEntryBuffer, const uint32_t directoryEntryBufferLength, const char* directoryPath, const char* targetName)
    {
        return IsEntryExisted(workDirectoryEntryBuffer, directoryEntryBufferLength, directoryPath, targetName, nn::fs::OpenDirectoryMode_Directory);
    }
    bool IsFileExisted(nn::fs::DirectoryEntry* workDirectoryEntryBuffer, const uint32_t directoryEntryBufferLength, const char* directoryPath, const char* targetName)
    {
        return IsEntryExisted(workDirectoryEntryBuffer, directoryEntryBufferLength, directoryPath, targetName, nn::fs::OpenDirectoryMode_File);
    }
    bool IsEntryExisted(nn::fs::DirectoryEntry* workDirectoryEntryBuffer, const uint32_t directoryEntryBufferLength, const char* directoryPath, const char* targetName)
    {
        return IsEntryExisted(workDirectoryEntryBuffer, directoryEntryBufferLength, directoryPath, targetName, nn::fs::OpenDirectoryMode_All);
    }
    bool IsMounted()
    {
        return m_IsMounted;
    }

private:
    nn::Result GetDirectoryOrFileList(nn::fs::DirectoryEntry* outDirectoryEntryBuffer, int64_t* outDirectoryEntryCount, const uint32_t directoryEntryBufferLength, const char* directoryPath, int openDirectoryMode);
    bool IsEntryExisted(nn::fs::DirectoryEntry* workDirectoryEntryBuffer, const uint32_t directoryEntryBufferLength, const char* directoryPath, const char* targetName, int openDirectoryMode);
};

