﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ShopMonitoringTool_SysGraphics.h"
#include "ShopMonitoringTool_SysGraphicsDrawer.h"

#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

#include <nn/nn_Assert.h>
#include <nn/mem.h>
#include <nn/fs.h>
#include <nn/nn_Log.h>

const nn::util::Color4u8Type Drawer::DefaultColor = { { 255, 255, 255, 255 } };

void* Drawer::ReadResource(const char* filename) NN_NOEXCEPT
{
    nn::Result result;
    nn::fs::FileHandle hFile;

    int64_t fileSize = 0;
    result = nn::fs::OpenFile(&hFile, filename, nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());

    result = nn::fs::GetFileSize(&fileSize, hFile);
    NN_ASSERT(result.IsSuccess());

    nn::util::BinaryFileHeader fileHeader;
    size_t readSize;
    result = nn::fs::ReadFile(&readSize, hFile, 0, &fileHeader, sizeof(nn::util::BinaryFileHeader));
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT(readSize == sizeof(nn::util::BinaryFileHeader));
    size_t alignment = fileHeader.GetAlignment();

    m_pMemory.AlignUp(alignment);
    void* pBuffer = m_pMemory.Get();
    result = nn::fs::ReadFile(&readSize, hFile, 0, pBuffer, static_cast< size_t >(fileSize));
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT(readSize == static_cast< size_t >(fileSize));
    m_pMemory.Advance(static_cast< ptrdiff_t >(fileSize));

    nn::fs::CloseFile(hFile);

    return pBuffer;
}

Drawer& Drawer::GetInstance() NN_NOEXCEPT
{
    static Drawer instance;
    return instance;
}

void Drawer::DrawRect(float x, float y, float w, float h,
    const nn::util::Color4u8Type& color) NN_NOEXCEPT
{
    nns::gfx::GraphicsFramework* pFramework = (nns::gfx::GraphicsFramework*)gGraphics.GetFramework();
    nn::gfx::CommandBuffer* pCommand = pFramework->GetRootCommandBuffer(gGraphics.GetBufferIndex());
    nns::gfx::PrimitiveRenderer::Renderer* pRenderer = (nns::gfx::PrimitiveRenderer::Renderer*)gGraphics.GetRenderer();

    pRenderer->SetColor(color);
    pRenderer->Draw2DRect(pCommand, x, y, w, h);
}

void Drawer::DrawLine(float beginX, float beginY, float endX, float endY,
    const nn::util::Color4u8Type& color) NN_NOEXCEPT
{
    nns::gfx::GraphicsFramework* pFramework = (nns::gfx::GraphicsFramework*)gGraphics.GetFramework();
    nn::gfx::CommandBuffer* pCommand = pFramework->GetRootCommandBuffer(gGraphics.GetBufferIndex());
    nns::gfx::PrimitiveRenderer::Renderer* pRenderer = (nns::gfx::PrimitiveRenderer::Renderer*)gGraphics.GetRenderer();

    pRenderer->SetColor(color);
    pRenderer->Draw2DLine(pCommand, beginX, beginY, endX, endY);
}

void Drawer::DrawFrame(float x, float y, float w, float h,
    const nn::util::Color4u8Type& color) NN_NOEXCEPT
{
    nns::gfx::GraphicsFramework* pFramework = (nns::gfx::GraphicsFramework*)gGraphics.GetFramework();
    nn::gfx::CommandBuffer* pCommand = pFramework->GetRootCommandBuffer(gGraphics.GetBufferIndex());
    nns::gfx::PrimitiveRenderer::Renderer* pRenderer = (nns::gfx::PrimitiveRenderer::Renderer*)gGraphics.GetRenderer();

    pRenderer->SetColor(color);
    pRenderer->Draw2DFrame(pCommand, x, y, w, h);
}

void Drawer::InitializeTextureResource(const char* filename) NN_NOEXCEPT
{
    m_pMemoryHeap.Reset(malloc(32 * 1024 * 1024));
    m_pMemory = m_pMemoryHeap;

    nns::gfx::GraphicsFramework* pFramework = (nns::gfx::GraphicsFramework*)gGraphics.GetFramework();

    void *pTextureResource = ReadResource(filename);
    m_pResTextureFile = nn::gfx::ResTextureFile::ResCast(pTextureResource);
    m_pResTextureFile->Initialize(pFramework->GetDevice());
    for (int idxTexture = 0 ;idxTexture < GetTextureCount(); ++idxTexture)
    {
        m_pResTextureFile->GetResTexture(idxTexture)->Initialize(pFramework->GetDevice());
    }

    m_TextureDescriptorBaseIndex = pFramework->AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_TextureView, GetTextureCount());
    m_SamplerDescriptorBaseIndex = pFramework->AllocateDescriptorSlot(nn::gfx::DescriptorPoolType_Sampler, 1);

    for (int i = 0; i < GetTextureCount(); ++i)
    {
        //NN_LOG("dic[%d]:%s\n", i, m_pResTextureFile->GetTextureDic()->GetKey(i));
        pFramework->SetTextureViewToDescriptorPool(m_TextureDescriptorBaseIndex + i, m_pResTextureFile->GetResTexture(i)->GetTextureView());
    }
}

void Drawer::FinalizeTextureResource() NN_NOEXCEPT
{
    nns::gfx::GraphicsFramework* pFramework = (nns::gfx::GraphicsFramework*)gGraphics.GetFramework();

    for (int idxTexture = 0; idxTexture < GetTextureCount(); ++idxTexture)
    {
        m_pResTextureFile->GetResTexture(idxTexture)->Finalize(pFramework->GetDevice());
    }
    m_pResTextureFile->Finalize(pFramework->GetDevice());

    free(m_pMemoryHeap.Get());
}

int Drawer::GetTextureCount() NN_NOEXCEPT
{
    return m_pResTextureFile->GetTextureDic()->GetCount();
}

void Drawer::DrawTexture(int indexSlot, float x, float y, float w, float h, const nn::util::Color4u8Type& color) NN_NOEXCEPT
{

    NN_ASSERT(indexSlot < m_pResTextureFile->GetTextureDic()->GetCount());

    nns::gfx::GraphicsFramework* pFramework = (nns::gfx::GraphicsFramework*)gGraphics.GetFramework();
    nn::gfx::CommandBuffer* pCommand = pFramework->GetRootCommandBuffer(gGraphics.GetBufferIndex());
    nns::gfx::PrimitiveRenderer::Renderer* pRenderer = (nns::gfx::PrimitiveRenderer::Renderer*)gGraphics.GetRenderer();

    nn::gfx::DescriptorSlot texture;
    nn::gfx::DescriptorSlot sampler;

    pFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType_TextureView)->GetDescriptorSlot(&texture, m_TextureDescriptorBaseIndex + indexSlot);
    pFramework->GetDescriptorPool(nn::gfx::DescriptorPoolType_Sampler)->GetDescriptorSlot(&sampler,m_SamplerDescriptorBaseIndex);

    pRenderer->SetColor(color);
    pRenderer->Draw2DRect(pCommand,x, y, w, h, texture,sampler);
}
