﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdio>
#include <cstring>
#include <cmath>

#include <nn/nn_SdkText.h>

#include "qre.h"

#include "QREL.h"
#include "QREncode.h"

using namespace mw::qre;

typedef struct {
    uint16_t bfType;
    uint8_t padding[2];
    uint32_t bfSize;
    uint16_t bfReserved1;
    uint16_t bfReserved2;
    uint32_t bfOffBits;
} BITMAPFILEHEADER;

typedef struct {
    uint32_t biSize;
    int32_t biWidth;
    int32_t biHeight;
    uint16_t biPlanes;
    uint16_t biBitCount;
    uint32_t biCompression;
    uint32_t biSizeImage;
    int32_t biXPelsPerMeter;
    int32_t biYPelsPerMeter;
    uint32_t biClrUsed;
    uint32_t biClrImportant;
} BITMAPINFOHEADER;

/**
 * コンストラクタ
 */
CQREncode::CQREncode():
    m_divCount(1), m_version(0), m_size(0), create(false)
{
    for (int32_t n = 0; n < 16; n++) {
        memset(m_imageInfo + n, 0, sizeof(ImageInfo));
        m_imageInfo[n].rgbData = NULL;
    }

    memset(m_ErrorMessage, 0, 256);

    InitLastError();
}

/**
 * デストラクタ
 */
CQREncode::~CQREncode()
{
    ReleaseMemory();
}

void CQREncode::ReleaseMemory()
{
    for (int32_t n = 0; n < 16; n++) {
        if (m_imageInfo[n].rgbData) {
            QRE_Free(m_imageInfo[n].rgbData);
            m_imageInfo[n].rgbData = NULL;
        }
        memset(m_imageInfo + n, 0, sizeof(ImageInfo));
    }
}

/**
 * QRコード生成
 *
 * @param   qrInfo(in)  QRコード情報構造体
 * @return  生成結果（trueは正常：falseは失敗）
 */
bool CQREncode::QRE_EncodeData(EncodeData *qrinfo)
{
    if (!qrinfo) {
        strcpy(m_ErrorMessage, ERR_MESS_ARGUMENT_ILLEGAL);
        return false;
    }

    create = false;
    ReleaseMemory();
    InitLastError();
    InitHeap();
    bool status = true;

    QRCPARAM qrcparam;
    memset(&qrcparam, 0, sizeof(QRCPARAM));

    MODE tmp_mode = AnalyzeMode(qrinfo->data, qrinfo->size);

    m_divCount = qrinfo->total;
    m_version = qrinfo->version;
    // 自動分割チェック
    if (qrinfo->total == 0) {
        int32_t div = 0;
        int32_t ver = qrinfo->version;
        if (!CheckAutoDiv(qrinfo, &div, &ver, tmp_mode))
            return false;
        qrinfo->total = div;
        qrinfo->version = ver;
        m_divCount = div;
    }
    // データ設定
    if (!LoadData(qrcparam, qrinfo, tmp_mode))
        // エラー
        return false;
    if (qrcparam.divcount > 1 && qrcparam.howtodiv == DIV_EQUAL) {
        // 均等分割のとき型番を決定する。
        VERSIONINFO maxversion;
        maxversion.version = 1;
        for (int32_t div = 0; div < qrcparam.divcount; div++) {
            DIVDATA *divdata = &qrcparam.divdata[div];
            VERSIONINFO version = divdata->version[0];

            status = QREL_DataAnalyze(&version, &divdata->format[0].ecclevel,
                                      &divdata->mode, &divdata->userdata,
                                      divdata->chain.totalcount < 2
                                      ? NULL : &divdata->chain);
            if (!status) {
                SetErrorMessage("");
                FreeData(qrcparam);
                return false;
            }
            if (maxversion.version < version.version)
                maxversion.version = version.version;
        }
        for (int32_t div = 0; div < qrcparam.divcount; div++)
            qrcparam.divdata[div].version[0] = maxversion;
    }
    for (int32_t div = 0; div < qrcparam.divcount; div++) {
        if (!MakeOneSymbol(qrcparam.divdata[div], div)) {
            status = false;
            break;
        }
    }
    m_version = qrcparam.divdata[0].version[1].version;

    FreeData(qrcparam);
    create = true;
    return status;
}

bool CQREncode::QRE_EncodeWithImg(EncodeData *qrInfo, const ImageInfo *img)
{
    if (!qrInfo || !img) {
        strcpy(m_ErrorMessage, ERR_MESS_ARGUMENT_ILLEGAL);
        return false;
    }

    if (!QRE_EncodeData(qrInfo)) {
        return false;
    }

    if (img->width > m_imageInfo[0].width / 3
        || img->height > m_imageInfo[0].height / 3) {
        strcpy(m_ErrorMessage, ERR_MESS_IMAGE2BIG);
        create = false;
        return false;
    }

    for (int32_t i = 0; i < m_divCount; i++) {
        Rect pos;
        pos.top = (m_imageInfo[i].height - img->height) / 2;
        pos.bottom = pos.top + img->height;
        pos.left = (m_imageInfo[i].width - img->width) / 2;
        pos.right = pos.left + img->width;
        if (img->rgbData) {
            if (!CompositeImage(img, &pos, i)) {
                create = false;
                return false;
            }
        }
        else
            WhiteOutImage(&pos, i);

        m_imageInfo[i].imgPos.top = pos.top;
        m_imageInfo[i].imgPos.bottom = pos.bottom;
        m_imageInfo[i].imgPos.left = pos.left;
        m_imageInfo[i].imgPos.right = pos.right;
    }

    return true;
}

/**
 * BMPファイルサイズ取得
 *
 * @param   mode(in)    取得モード (mode 0: 保存 1: 表示)
 * @param   index(in)   分割番号
 * @return  BMPファイルサイズ
 */
int32_t CQREncode::QRE_GetBMPSize(int32_t index, bool is_bmp)
{
    if (!create)
        return 0;

    if (is_bmp)
        return (int32_t)round4(m_imageInfo[index].width * 3)
            * m_imageInfo[index].height + 54;

    return m_imageInfo[index].width * m_imageInfo[index].height * 3;
}

/**
 * BMPデータ取得
 *
 * @param   pData(out)  画像データ
 * @param   size(in)    画像サイズ
 * @param   mode(in)    取得モード (0: 保存 1: 表示)
 * @param   index(in)   分割番号
 * @return  取得結果（trueは正常：falseは失敗）
 */
bool CQREncode::QRE_GetBMPData(uint8_t *pData, int32_t size,
                               int32_t index, mw::qre::Rect *pos)
{
    if (pData == NULL || size < 1 || !create)
        return false;

    CreateBMPFile(pData, index);
    if (pos) {
        pos->top = m_imageInfo[index].imgPos.top;
        pos->bottom = m_imageInfo[index].imgPos.bottom;
        pos->left = m_imageInfo[index].imgPos.left;
        pos->right = m_imageInfo[index].imgPos.right;
    }

    return true;
}

bool CQREncode::QRE_GetBMPData(ImageInfo *img, int32_t index)
{
    if (!img || !img->rgbData || !create)
        return false;

    img->width = m_imageInfo[index].width;
    img->height = m_imageInfo[index].height;
    img->imgPos.top = m_imageInfo[index].imgPos.top;
    img->imgPos.bottom = m_imageInfo[index].imgPos.bottom;
    img->imgPos.left = m_imageInfo[index].imgPos.left;
    img->imgPos.right = m_imageInfo[index].imgPos.right;
    memcpy(img->rgbData, m_imageInfo[index].rgbData,
           img->width * img->height * 3);

    return true;
}

/**
 * 分割数取得
 *
 * @return  分割数
 */
int32_t CQREncode::QRE_GetDivCount()
{
    if (!create)
        return 0;
    return m_divCount;
}

/**
 * フォーマットバージョン取得
 *
 * @return  フォーマットバージョン
 */
int32_t CQREncode::QRE_GetVersion()
{
    if (!create)
        return 0;
    return m_version;
}

/**
 * エラー情報取得
 *
 * @param   pMess(out)  エラー情報文字列格納用
 * @return  取得結果（trueは正常：falseは失敗）
 */
bool CQREncode::QRE_GetErrorMessage(char *pMess, int32_t size)
{
    if (pMess == NULL)
        return false;
    strncpy(pMess, m_ErrorMessage, size);
    return true;
}

/**
 * データ作成
 *
 * @param   qrcparam(out)   QRコード情報
 * @param   qrinfo(in)      QRコード情報
 * @return  成功/失敗
 */
bool CQREncode::LoadData(QRCPARAM &qrcparam, const EncodeData *qrinfo, MODE tmp_mode)
{
    bool status = true;
    memset(&qrcparam, 0, sizeof(qrcparam));
    // データ数チェック
    if (qrinfo->size <= 0) {
        strcpy(m_ErrorMessage, NN_TEXT("データを入力してください。"));
        return false;
    }
    // 連結情報を設定する
    if (qrinfo->total <= 1) {
        // 分割しない
        qrcparam.howtodiv = DIV_NONE;
        qrcparam.divcount = 1;
    }
    else {
        // 均等分割
        qrcparam.howtodiv = DIV_EQUAL;
        qrcparam.divcount = qrinfo->total;
    }
    // QRコード生成用データ長
    qrcparam.userdata_all.length = qrinfo->size;

    qrcparam.userdata_all.data = (int8_t *)QRE_Malloc(qrinfo->size + 1);

    memset(qrcparam.userdata_all.data, 0x00, qrinfo->size + 1);
    memcpy(qrcparam.userdata_all.data, qrinfo->data, qrinfo->size);

    uint8_t parity_;
    QREL_CalcChainParity(&qrcparam.userdata_all, &parity_);

    // 各連結シンボルごとの処理を行う。
    qrcparam.divdata
        = (DIVDATA *)QRE_Malloc(qrcparam.divcount * sizeof(DIVDATA));
    memset(qrcparam.divdata, 0, qrcparam.divcount * sizeof(DIVDATA));
    for (int32_t div = 0; div < qrcparam.divcount; div++) {
        DIVDATA *divdata = &qrcparam.divdata[div];

        // 連結情報を設定する。
        divdata->chain.seqno = div + 1;
        divdata->chain.totalcount = qrcparam.divcount;
        divdata->rightparity = parity_;
        divdata->chain.parity = divdata->rightparity;

        // 型番を設定する。
        divdata->version[0].version = qrinfo->version;
        divdata->version[1].version = 0;

        // モード指示子を設定する。
#if 0
        divdata->mode = (tmp_mode == MODE_8BITSBYTE) ? MODE_ANY : tmp_mode;
#else
        divdata->mode = tmp_mode;
#endif

        // 形式情報を設定する。
        ECCLEVEL eclv;
        switch (qrinfo->ecc_level) {
        case ECC_LEVEL_M:
            eclv = ECCL_M;
            break;

        case ECC_LEVEL_Q:
            eclv = ECCL_Q;
            break;

        case ECC_LEVEL_H:
            eclv = ECCL_H;
            break;

        default:
            eclv = ECCL_L;
            break;
        }
        divdata->format[0].ecclevel = eclv;

        // マスクパターンを設定する。
        auto mask_any = MaskAny;
        divdata->format[0].mp = (int8_t)mask_any;
        divdata->format[1] = divdata->format[0];
        // データを設定する。
        switch (qrcparam.howtodiv) {
        case DIV_NONE:
            status = DivideNone(qrcparam, qrcparam.userdata_all);
            break;

        case DIV_EQUAL:
            status = DivideEqual(div, qrcparam, qrcparam.userdata_all);
            break;

        case DIV_FREE:
            break;

        default:
            strcpy(m_ErrorMessage, NN_TEXT("分割方法が不正です。"));
            status = false;
            break;
        }
        if (!status)
            return false;
        // セルのサイズ/クワイエットゾーンをロードする。
        divdata->qzone = DefaultQuietZoneSize;
        divdata->cellpitch = qrinfo->cell_size;
    }
    return true;
}

/**
 * 開放
 *
 * @param   qrcparam(in)    QRコード情報
 */
bool CQREncode::FreeData(QRCPARAM &qrcparam)
{
    if (qrcparam.divdata)
        QRE_Free(qrcparam.divdata);
    if (qrcparam.userdata_all.data)
        QRE_Free(qrcparam.userdata_all.data);
    return true;
}

/**
 * 分割情報設定(分割なし)
 *
 * @param   qrcparam(out)       QRコード情報
 * @param   userdata_all(out)   データ列
 */
bool CQREncode::DivideNone(QRCPARAM &qrcparam, USERDATA &userdata_all)
{
    qrcparam.divdata[0].userdata = userdata_all;
    return true;
}

/**
 * 分割情報設定(均等分割)
 *
 * @param   div(in)             分割数
 * @param   qrcparam(out)       QRコード情報
 * @param   userdata_all(out)   データ列
 */
bool CQREncode::DivideEqual(int32_t div, QRCPARAM &qrcparam, USERDATA &userdata_all)
{
    MODE mode = qrcparam.divdata[div].mode;
    double onesize = (double)userdata_all.length / qrcparam.divcount;
    int32_t this_size;

    int32_t sum = 0;
    for (int32_t i = 0; i < div; i++)
        sum += qrcparam.divdata[i].userdata.length;
    if (div != qrcparam.divcount - 1) {
        this_size = (int32_t)((div + 1) * onesize) - sum;
        if (mode == MODE_KANJI) {
            if (this_size % 2)
                // モード指示子が漢字でthis_sizeが奇数
                this_size--;    // this_sizeを-1する。
        }
    }
    else
        this_size = userdata_all.length - sum;

    qrcparam.divdata[div].userdata.data = userdata_all.data + sum;
    qrcparam.divdata[div].userdata.length = this_size;

    return true;
}

/**
 * QRシンボル生成
 *
 * @param   divdata(in)             分割情報
 * @param   divID(in)               分割番号
 * @return  成功/失敗
 */
// 手順
// 1. QREL_DataAnalyze              データ分析
// 2. QREL_DataEncode               コード生成
// 3. QREL_AddErrorCorrectData      誤り訂正コード追加
// 4. QREL_LoadToMatrix             マトリクス生成
// 5. QREL_SetMask                  マスク処理
// 6. QREL_GenerateQRCode           シンボル生成
// 7. CreateBWBMPFile               ビットマップデータ作成
bool CQREncode::MakeOneSymbol(DIVDATA &divdata, int32_t divID)
{
    bool status;
    int32_t ncode;
    uint8_t *code = NULL;
    uint8_t **matrix = NULL;
    int32_t cellsize;

    // データ分析を行う。
    VERSIONINFO version = divdata.version[0];

    status = QREL_DataAnalyze(&version, &divdata.format[0].ecclevel,
                              &divdata.mode, &divdata.userdata,
                              divdata.chain.totalcount < 2
                              ? NULL : &divdata.chain);
    if (!status) {
        SetErrorMessage("");
        return false;
    }
    divdata.version[1] = version;   // divdata.version[1]: 確定バージョン

    // ＱＲコードデータの要素数を獲得する
    VER_GetnCellNum(divdata.version[1].version, &cellsize);
    m_size = (cellsize + divdata.qzone * 2) * divdata.cellpitch;

    // コード生成を行う。
    status = QREL_DataEncode(&divdata.version[1], &divdata.format[0].ecclevel,
                             &divdata.mode, &divdata.userdata, &ncode, &code,
                             divdata.chain.totalcount < 2
                             ? NULL : &divdata.chain);
    if (!status) {
        SetErrorMessage("");
        return false;
    }
    status = QREL_AddErrorCorrectData(divdata.version[1],
                                      divdata.format[0].ecclevel, code);
    if (!status) {
        SetErrorMessage("");
        if (code)
            QRE_Free(code);
        return false;
    }
    // マトリクスを生成する。
    status = QREL_LoadToMatrix(divdata.version[1], ncode, code, &matrix);
    if (code)
        QRE_Free(code);
    if (!status) {
        SetErrorMessage("");
        return false;
    }

    // マスク処理
    divdata.format[1] = divdata.format[0];
    status = QREL_SetMask(divdata.version[1], &divdata.format[1].mp,
                          divdata.format[0].ecclevel, matrix);
    if (!status) {
        SetErrorMessage("");
        // matrixを解放する。
        FreeMatrix(matrix, cellsize);
        return false;
    }
    // シンボル完成
    status = QREL_GenerateQRCode(divdata.version[1], divdata.format[1],
                                 matrix);
    if (!status) {
        SetErrorMessage("");
        // matrixを解放する。
        FreeMatrix(matrix, cellsize);
        return false;
    }

    status = CreateRawData(cellsize, divdata.cellpitch,
                           divdata.qzone, matrix, divID);
    FreeMatrix(matrix, cellsize);
    if (!status) {
        strcpy(m_ErrorMessage, NN_TEXT("Rawdataを作成できません。"));
        // matrixを解放する。
        return false;
    }

    return true;
}

void CQREncode::FreeMatrix(uint8_t **matrix, int32_t cellsize)
{
    if (matrix == NULL)
        return;
    for (int32_t i = 0; i < cellsize; i++)
        if (matrix[i])
            QRE_Free(matrix[i]);
    if (matrix)
        QRE_Free(matrix);
}

bool CQREncode::CreateRawData(int32_t size, int32_t pitch, int32_t qzone,
                              uint8_t **matrix, int32_t divID)
{
    uint16_t width = (uint16_t)m_size * 3;
    uint16_t height = (uint16_t)m_size;
    uint8_t *ptr = reinterpret_cast<uint8_t *>(QRE_Malloc((uint32_t)width * height));
    if (!ptr)
        return false;

    uint8_t *pos = ptr;
    int32_t y_end = qzone * pitch;

    memset(pos, 0xFF, y_end * (uint32_t)width);
    pos += width * y_end;
    int32_t dy = y_end;

    y_end = (qzone + size) * pitch;
    int32_t cy = 0;
    for (int32_t sy = 0; dy < y_end; dy++, cy++) {
        int32_t x_end = qzone * pitch * 3;
        memset(pos, 0xFF, x_end);
        int32_t dx = x_end;

        x_end = (qzone + size) * pitch * 3;
        int32_t cx = 0;
        for (int32_t sx = 0; dx < x_end; dx += 3, cx++) {
            pos[dx] = pos[dx + 1] = pos[dx + 2] = ~matrix[sy][sx];
            if ((cx % pitch) == (pitch - 1))
                sx++;
        }

        memset(pos + dx, 0xFF, width - dx);
        pos += width;
        if ((cy % pitch) == (pitch - 1))
            sy++;
    }

    memset(pos, 0xFF, (height - dy) * (uint32_t)width);

    m_imageInfo[divID].width = (uint16_t)(width / 3);
    m_imageInfo[divID].height = (uint16_t)height;
    m_imageInfo[divID].imgPos.top = m_imageInfo[divID].imgPos.bottom
        = m_imageInfo[divID].imgPos.left = m_imageInfo[divID].imgPos.right = 0;

    if (m_imageInfo[divID].rgbData)
        QRE_Free(m_imageInfo[divID].rgbData);
    m_imageInfo[divID].rgbData = ptr;
    return true;
}

bool CQREncode::CreateBMPFile(uint8_t *ptr, int32_t index)
{
    uint16_t height = m_imageInfo[index].height;
    uint16_t width = m_imageInfo[index].width;
    int32_t bmp_width = (int32_t)round4(width * 3);
    BITMAPFILEHEADER header;// = (BITMAPFILEHEADER *)ptr;        // ビットマップヘッダー関連構造体
    memcpy(&header.bfType, "BM", 2);
    header.bfSize = 14 + 40 + (bmp_width * height);
    header.bfReserved1 = 0;
    header.bfReserved2 = 0;
    header.bfOffBits = 54;
    memcpy(ptr, &header.bfType, 2);
    memcpy(ptr + 2, &header.bfSize, 4);
    memcpy(ptr + 6, &header.bfReserved1, 2);
    memcpy(ptr + 8, &header.bfReserved2, 2);
    memcpy(ptr + 10, &header.bfOffBits, 4);

#if 0
    BITMAPINFOHEADER *info = (BITMAPINFOHEADER *)(ptr + 14);          // ビットマップヘッダー関連構造体
#else
    BITMAPINFOHEADER *info = (BITMAPINFOHEADER *)QRE_Malloc(40);          // ビットマップヘッダー関連構造体
#endif
    info->biSize = 40;
    info->biWidth = width;
    info->biHeight = height;
    info->biPlanes = 1;
    info->biBitCount = 24;
    info->biCompression = 0;
    info->biSizeImage = bmp_width * height;
    info->biXPelsPerMeter = info->biYPelsPerMeter = 3780;
    info->biClrUsed = 0;
    info->biClrImportant = 0;
#if 1
    memcpy(ptr + 14, info, 40);
    QRE_Free(info);
#endif

    int32_t raw_width = width * 3;
    int32_t rest = bmp_width - raw_width;
    for (int32_t y = 0; y < height; y++) {
        uint8_t *src_ptr = m_imageInfo[index].rgbData
            + (height - y - 1) * raw_width;
        uint8_t *dst_ptr = ptr + 54 + y * bmp_width;
        memcpy(dst_ptr, src_ptr, raw_width);
        if (rest)
            memset(dst_ptr + raw_width, 0, rest);
    }

    return true;
}

/**
 * 自動分割チェック
 * @param   div(out)        分割数
 * @param   ver(out)        バージョン
 * @return  成功/失敗
 */
bool CQREncode::CheckAutoDiv(const EncodeData *qrinfo, int32_t *div, int32_t *ver, MODE tmp_mode)
{
    int32_t all = 0;
    int32_t size = 0;
    int32_t tmpDiv = 0;
    int32_t tmpVer = 0;
    ERRCORUNIT  rsinfo;
    ECCLEVEL eclv;
    switch (qrinfo->ecc_level) {
    case ECC_LEVEL_M:
        eclv = ECCL_M;
        break;

    case ECC_LEVEL_Q:
        eclv = ECCL_Q;
        break;

    case ECC_LEVEL_H:
        eclv = ECCL_H;
        break;

    default:
        eclv = ECCL_L;
        break;
    }
    if (qrinfo->version == 0) {
        // 最小のバージョンで分割
        VER_GetAllDataWord(40, &all);
        VER_GetRSInfo(40, eclv, &rsinfo);
        //16:ヘッダー13バイト+α2～3バイト
        size = all - rsinfo.ErrcorCodeNum;
#if 0
        tmpDiv = (qrinfo->size + size - 1) / size;
        if (tmpDiv > 16) {
            sprintf(m_ErrorMessage, "%s", ERR_MESS_AUTO_DIV_VER);
            return false;
        }
        int32_t len = qrinfo->size / tmpDiv;
        for (int32_t n = 40; 0 < n; n--) {
            VER_GetAllDataWord(n, &all);
            VER_GetRSInfo(n, eclv, &rsinfo);
            size = all - rsinfo.ErrcorCodeNum - 16;
            if (size < len) {
                tmpVer = n + 1;
                break;
            }
        }
        *div = tmpDiv;
        *ver = tmpVer;
#else
        int32_t total_length = CalcDataBitLength(40, tmp_mode, qrinfo->size);
        if (((total_length + 7) / 8) < size) {
            // 分割無し
            tmpDiv = 1;
            tmpVer = 40;
            for (int32_t n = 39; n > 0; n--) {
                // 最小検索
                VER_GetAllDataWord(n, &all);
                VER_GetRSInfo(n, eclv, &rsinfo);
                size = all - rsinfo.ErrcorCodeNum;
                if (total_length > size * 8)
                    break;
                tmpVer = n;
            }
        }
        else {
            total_length -= (CalcCharLengthFieldLength(40, tmp_mode) + 4);
            // total_length = (total_length + 7) / 8;        // バイトサイズ
            // 20 ... 分割モード指示子
            int32_t div_work = size - ((20 + 4 + CalcCharLengthFieldLength(40, tmp_mode) + 7) / 8);
            tmpDiv = (((total_length + 7) / 8) + div_work - 1) / div_work;
#if 1
            if (tmp_mode == MODE_KANJI) {
                int32_t one_cnt = div_work * 8 / 13;
                total_length /= 13;
                tmpDiv = (total_length + one_cnt - 1) / one_cnt;
            }
#endif
            if (tmpDiv > 16) {
                sprintf(m_ErrorMessage, "%s", ERR_MESS_AUTO_DIV_VER);
                return false;
            }
            tmpVer = 40;
            int32_t one_length = (total_length + tmpDiv - 1) / tmpDiv;
            for (int32_t n = 39; n > 0; n--) {
                VER_GetAllDataWord(n, &all);
                VER_GetRSInfo(n, eclv, &rsinfo);
                size = all - rsinfo.ErrcorCodeNum - (20 + 4 + CalcCharLengthFieldLength(n, tmp_mode) + 7) / 8;
                if (one_length > size)
                    break;
                tmpVer = n;
            }
        }
        *div = tmpDiv;
        *ver = tmpVer;
#endif
    }
    else {
        // 指定したバージョンで分割
        VER_GetAllDataWord(qrinfo->version, &all);
        VER_GetRSInfo(qrinfo->version, eclv, &rsinfo);
        //16:ヘッダー13バイト+α2～3バイト
        size = all - rsinfo.ErrcorCodeNum;
        if (size < 1) {
            sprintf(m_ErrorMessage, "%s", ERR_MESS_AUTO_DIV_VER);
            return false;
        }
#if 0
        int32_t len = size - qrinfo->size;
        if (len > 0) {
            // 分割する必要なし
            *div = 1;
            return true;
        }
        tmpDiv = (qrinfo->size + size - 1) / size;
        if (tmpDiv > 16) {
            sprintf(m_ErrorMessage, "%s\n%s", ERR_MESS_AUTO_DIV_VER,
                    ERR_MESS_AUTO_DIV_SIZE);
            return false;
        }
        *div = tmpDiv;
#else
        int32_t total_length = CalcDataBitLength(qrinfo->version, tmp_mode, qrinfo->size);
        if (((total_length + 7) / 8) < size) {
            *div = 1;
            return true;
        }
        int32_t length_field = CalcCharLengthFieldLength(qrinfo->version, tmp_mode);
        total_length -= (4 + length_field);
        total_length = (total_length + 7) / 8;
        int32_t div_work = size - ((20 + 4 + length_field + 7) / 8);
        tmpDiv = (total_length + div_work - 1) / div_work;
#if 0
        if (tmp_mode == MODE_KANJI) {
            div_work /= 2;
            tmpDiv = ((total_length / 2) + div_work - 1) / div_work;
        }
#endif
        if (tmpDiv > 16) {
            sprintf(m_ErrorMessage, "%s\n%s", ERR_MESS_AUTO_DIV_VER,
                    ERR_MESS_AUTO_DIV_SIZE);
            return false;
        }
        *div = tmpDiv;
#endif
    }
    return true;
}// NOLINT(impl/function_size)

bool CQREncode::CompositeImage(const ImageInfo *img, Rect *pos, int32_t divID)
{
    int32_t swidth = img->width * 3;
    int32_t sheight = img->height;
    int32_t dwidth = m_imageInfo[divID].width * 3;
    int32_t dx = pos->left * 3;
    int32_t dy = pos->top;
    uint8_t *ptr = m_imageInfo[divID].rgbData;
    for (int32_t sy = 0; sy < sheight; sy++, dy++) {
        int32_t tsy = sy * swidth;
        int32_t tdy = dy * dwidth;
        memcpy(ptr + dx + tdy, img->rgbData + tsy, swidth);
    }

    return true;
}

bool CQREncode::WhiteOutImage(Rect *pos, int32_t divID)
{
    int32_t cwidth = (pos->right - pos->left) * 3;
    int32_t cheight = pos->bottom - pos->top;
    int32_t dwidth = m_imageInfo[divID].width * 3;
    int32_t dx = pos->left * 3;
    int32_t dy = pos->top;
    uint8_t *ptr = m_imageInfo[divID].rgbData;
    for (int32_t sy = 0; sy < cheight; sy++, dy++) {
        int32_t tdy = dy * dwidth;
        memset(ptr + dx + tdy, 0xFF, cwidth);
    }

    return true;
}

/**
 * エラーメッセージ設定
 * @param   mess(in)        追加メッセージ
 *
 */
void CQREncode::SetErrorMessage(const char *mess)
{
    switch (QREL_GetLastError()) {
    case ERR_ILLEGALPARAM:
        sprintf(m_ErrorMessage, "%s\n", ERR_MESS_ILLEGALPARAM);
        break;

    case ERR_VERSIONINFOERROR:
        sprintf(m_ErrorMessage, "%s\n", ERR_MESS_VERSIONINFO);
        break;

    case ERR_TOOSMALL:
        sprintf(m_ErrorMessage, "%s\n", ERR_MESS_TOOSMALL);
        break;

    case ERR_DATAFORMATERROR:
        sprintf(m_ErrorMessage, "%s\n", ERR_MESS_DATAFORMAT);
        break;

    default:
        sprintf(m_ErrorMessage, "%s\n", ERR_MESS_SYSTEM);
        break;
    }
    strcat(m_ErrorMessage, mess);
}
