﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "QCIT_RootSurface.h"

#include "QCIT_ApplicationInstaller.h"

namespace qcit
{
    InstallScene* g_InstallScene = nullptr;

    void QCITMain() NN_NOEXCEPT
    {
        //! GLV の共通カラースタイル定義
        glv::Style::standard().color.set(glv::StyleColor::BlackOnWhite);
        glv::Style::standard().color.fore.set(0.5);

        //! ユーザサーフェイスコンテキストの作成
        const int width = glv::glutGet(GLUT_SCREEN_WIDTH);
        const int height = glv::glutGet(GLUT_SCREEN_HEIGHT);

        RootSurfaceContext* context = new RootSurfaceContext(width, height);

        // メインループコールバックを登録.
        glv::ApplicationFrameworkRegisterLoopCallback(context);

        // メインループ.
        glv::Application::run();

        // GLV依存のユーザコンテキストの解放.
        delete context;

        // エンジンコンテキストの解放処理タイミングを明示的に呼び出すこともできます.
        // 呼び出さない場合では静的インスタンスのデストラクションのタイミングで呼ばれます.
        glv::Application::quit();
    }

    /**
    * @brief       コンストラクタです。
    *
    * @param[in]   width       幅
    * @param[in]   height      高
    */
    RootSurfaceContext::RootSurfaceContext(const unsigned width, const unsigned height) NN_NOEXCEPT
        : glv::Window(width, height, "QCIT"), glv::GLV(),
        m_pBackgroundDrawable(nullptr),
        m_pSurfaceTable(nullptr)
    {
        // メインサーフェイス
        const glv::space_t padding = 2;
        const glv::space_t surfaceWidth = width - (padding * 2);
        const glv::space_t surfaceHeight = height - (padding * 2);

        m_PageSurface.extent(surfaceWidth, surfaceHeight);
        m_PageSurface.cloneStyle().colors().set(glv::Color(0.098f, 0.27f, 1.0f, 0.5f), 0.7f);
        m_PageSurface.enable(glv::Property::DrawBack | glv::Property::DrawBorder | glv::Property::CropChildren);

        m_pSurfaceTable = new glv::Table("p", padding, padding, glv::Rect(surfaceWidth, surfaceHeight));
        auto& table = *m_pSurfaceTable;
        table << m_PageSurface;
        *this << table.arrange();

        g_InstallScene = new InstallScene(this);
        m_PageSurface << *g_InstallScene;

        // モーダル表示用ビュー
        *this << m_InvisibleWall;

        // 強制的にリスト部分にフォーカスを当てておく
        this->setFocus(g_InstallScene->GetFirstFocusedView());

        this->setGLV(*this);
    }

    /**
    * @brief       デストラクタです。
    */
    RootSurfaceContext::~RootSurfaceContext() NN_NOEXCEPT
    {
        delete m_pSurfaceTable;
    }

    /**
    * @brief       ページ共通背景画像描画オブジェクト取得.
    */
    glv::Drawable* RootSurfaceContext::GetBackgroundDrawable() NN_NOEXCEPT
    {
        return m_pBackgroundDrawable;
    }

    /**
    * @brief       ランタイムエンジンにアタッチされた際に呼ばれます。
    *
    * @param[in]   context     コンテキスト
    *
    * @see         glv::ApplicationLoopCallback::OnLoopAttached()
    */
    void RootSurfaceContext::OnLoopAttached(glv::ApplicationLoopContext& context) NN_NOEXCEPT
    {
        m_InvisibleWall.OnAttached(this);

        ApplicationLoopCallback::OnLoopAttached(context);
        GLV_LOG("OnLoopAttached\n");
        glv::Resources& res = glv::Resources::GetSharedResourceContext();
        auto drawablePath = std::string(RomMountName) + ":/drawable/viewBackground.bntx";
        m_pBackgroundDrawable = res.GetDrawable(drawablePath.c_str());
    }

    /**
    * @brief       ランタイムエンジンからデタッチされた際に呼ばれます。
    *
    * @param[in]   context     コンテキスト
    *
    * @see         glv::ApplicationLoopCallback::OnLoopDetached()
    */
    void RootSurfaceContext::OnLoopDetached(glv::ApplicationLoopContext& context) NN_NOEXCEPT
    {
        m_InvisibleWall.OnDetached(this);

        glv::Drawable* background = nullptr;
        if (nullptr != (background = m_pBackgroundDrawable))
        {
            m_pBackgroundDrawable = nullptr;
            background->Finish();
        }
        GLV_LOG("OnLoopDetached\n");
        ApplicationLoopCallback::OnLoopDetached(context);
    }

    /**
    * @brief       glv シーンレンダラ前に呼ばれます。
    *
    * @param[in]   context     コンテキスト
    * @param[in]   events      イベント
    *
    * @details     glv シーンレンダラへ hid 系イベントが通知される前に呼び出されます。
    *              この時点ではまだ glv コンテキストのレンダリングは開始していません。
    *              このメソッドが呼び出されるフレームは OnLoopAfterSceneRenderer() と同じです。
    *
    * @return      現在は RequiredRestoration::RequireRestrationNothing を返すようにしてください。
    *
    * @see         glv::ApplicationLoopCallback::OnLoopBeforeSceneRenderer()
    */
    const glv::RequiredRestoration RootSurfaceContext::OnLoopBeforeSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
    {
        NN_UNUSED(events);
        NN_UNUSED(context);

        m_InvisibleWall.OnLoopBeforeSceneRenderer(context, events);
        if (g_InstallScene != nullptr)
        {
            g_InstallScene->InitialCheckProcess();
            g_InstallScene->UpdateInstallProgress();
        }

        // ループ毎の前処理
        HandleLoopPre(context, events);

        // 戻り値は固定
        return glv::RequiredRestoration::RequireRestrationNothing;
    }

    /**
    * @brief       glv シーンレンダラ後に呼ばれます。
    *
    * @param[in]   context     コンテキスト
    * @param[in]   events      イベント
    *
    * @details     glv シーンレンダラのレンダリングが終わった後に呼び出されます。
    *              このメソッドが呼び出されるフレームは OnLoopBeforeSceneRenderer() と同じです。
    *
    * @return      現在は RequiredRestoration::RequireRestrationNothing を返すようにしてください。
    *
    * @see         glv::ApplicationLoopCallback::OnLoopAfterSceneRenderer()
    */
    const glv::RequiredRestoration RootSurfaceContext::OnLoopAfterSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
    {
        NN_UNUSED(events);
        NN_UNUSED(context);

        m_InvisibleWall.OnLoopAfterSceneRenderer(context, events);

        // ループ毎の後処理
        HandleLoopPost(context, events);

        // HID イベント処理
        HandleHidEvent(context, events);

        // 戻り値は固定
        return glv::RequiredRestoration::RequireRestrationNothing;
    }

    /**
    * @brief       ループ毎の前処理を行います。
    *
    * @param[in]   context     コンテキスト
    * @param[in]   events      イベント
    */
    void RootSurfaceContext::HandleLoopPre(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
    {
        NN_UNUSED(context);
        NN_UNUSED(events);
    }

    /**
    * @brief       ループ毎の後処理を行います。
    *
    * @param[in]   context     コンテキスト
    * @param[in]   events      イベント
    */
    void RootSurfaceContext::HandleLoopPost(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
    {
        NN_UNUSED(context);
        NN_UNUSED(events);
    }

    /**
    * @brief       HID イベントを処理します。
    *
    * @param[in]   context     コンテキスト
    * @param[in]   events      イベント
    */
    void RootSurfaceContext::HandleHidEvent(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
    {
        NN_UNUSED(context);
        NN_UNUSED(events);
    }

    glv::View* RootSurfaceContext::GetActivePage() NN_NOEXCEPT
    {
        return &m_PageSurface;
    }
};
