﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>

#include "QCIT_InstallContentInfo.h"
#include "QCIT_FsUtilities.h"
#include "QCIT_StringUtil.h"

namespace qcit
{

const char* CONTENT_TYPE = "Type";
const char* CONTENT_NSP_FILE_NAME = "NspFileName";
const char* CONTENT_ID = "Id";
const char* CONTENT_DIGEST = "Digest";
const char* CONTENT_DATA_SIZE = "DataSize";
const char* CONTENT_APPLICATION_ID = "ApplicationId";
const char* CONTENT_INDEX = "Index";

ContentDataContainer::ContentDataContainer() NN_NOEXCEPT
{
    this->InitializeTitleIndexMap();
}

void ContentDataContainer::InitializeTitleIndexMap() NN_NOEXCEPT
{
    // 冗長な表現となるためマクロ関数化しておく
    #define INIT_MAP(inTitleName) m_TitleIndexMap.insert(std::make_pair(std::string(inTitleName), 0))

    m_TitleIndexMap.clear();
    INIT_MAP(CONTENT_TYPE);
    INIT_MAP(CONTENT_NSP_FILE_NAME);
    INIT_MAP(CONTENT_ID);
    INIT_MAP(CONTENT_DIGEST);
    INIT_MAP(CONTENT_DATA_SIZE);
    INIT_MAP(CONTENT_APPLICATION_ID);
    INIT_MAP(CONTENT_INDEX);
}

ContentDataContainer::~ContentDataContainer() NN_NOEXCEPT
{
}

// テストデータの元になる一覧データの読み込み
// m_ContentDataList に展開する
nn::Result ContentDataContainer::ReadFromContentInfoCSV(const std::string& inInfoFilePath) NN_NOEXCEPT
{
    std::string dataStr;
    auto result = fsutil::GetDataOfFile(inInfoFilePath, dataStr);
    if (result.IsFailure())
    {
        NN_LOG("[QCIT][Error] ReadFromContentInfoCSV() : GetDataOfFile() Failed : path=%s", inInfoFilePath.c_str());
        return result;
    }

    // CSVデータの文字列としてパース処理を行う
    auto table = this->ParseTable(dataStr, ",");

    // パースしたCSVデータを本クラスのテストデータのコンテナ型へ変換する
    this->ToContentDataContainer(table);

    return result;
}

// Sync リストの設定値データの読み込み
// m_SyncList に展開する
nn::Result ContentDataContainer::ReadFromSyncListCSV(const std::string& inSyncListFilePath) NN_NOEXCEPT
{
    std::string dataStr;
    auto result = fsutil::GetDataOfFile(inSyncListFilePath, dataStr);
    if (result.IsFailure())
    {
        NN_LOG("[QCIT][Error] ReadFromSyncListCSV() : GetFileData() Failed : path=%s", inSyncListFilePath.c_str());
        return result;
    }

    // CSVデータの文字列としてパース処理を行う
    auto table = this->ParseTable(dataStr, ",");

    // パースしたCSVデータを本クラスのテストデータのコンテナ型へ変換する
    this->ToSyncListContainer(table);

    return result;
}

ContentDataContainer::CsvDataTable
ContentDataContainer::ParseTable(const std::string& inData, const std::string& delim) NN_NOEXCEPT
{
    std::stringstream ss(inData);
    CsvDataTable table;

    const auto delimSize = delim.size();
    std::string lineStr;
    while (getline(ss, lineStr))
    {
        // \r が末尾に存在する場合を考慮して、\r が存在すれば消しておく
        if (*lineStr.rbegin() == '\r')
        {
            lineStr.erase(lineStr.size() - 1);
        }

        if (lineStr.size() <= 1)
        {
            continue;
        }

        std::vector<std::string> tokens;

        size_t currentPos = 0;
        auto foundPos = lineStr.find(delim, currentPos);
        while (foundPos != std::string::npos)
        {
            tokens.emplace_back(lineStr, currentPos, foundPos - currentPos);
            currentPos = foundPos + delimSize;
            foundPos = lineStr.find(delim, currentPos);
        }
        tokens.emplace_back(lineStr, currentPos, lineStr.size() - currentPos);

        table.push_back(tokens);
    }

#if 0
    // デバッグ用のログ出力処理
    for (auto& record : table)
    {
        NN_LOG(":");
        for (auto& val : record)
        {
            NN_LOG("\"%s\":", val.c_str());
        }
        NN_LOG("\n");
        NN_LOG("----------------------------\n");
    }
#endif

    return table;
}

void ContentDataContainer::ToContentDataContainer(CsvDataTable& inCSVData) NN_NOEXCEPT
{
    if (inCSVData.size() < 2)
    {
        // CSVデータとしては不完全なデータ
        NN_LOG("[QCIT][Error] ToContentDataContainer() : Not CSV Data\n");
        return;
    }

    this->InitializeTitleIndexMap();

    auto iterRecord = std::begin(inCSVData);
    for (auto& titlePair : m_TitleIndexMap)
    {
        titlePair.second = this->FindIndexVector((*iterRecord), titlePair.first);
    }

    m_ContentDataList.clear();
    m_ContentInfoMap.clear();

    // 2行目以降のデータが各テストデータのレコードとなる
    ++iterRecord;
    for (; iterRecord != std::end(inCSVData); ++iterRecord)
    {
        ContentRecord record;
        this->SetCsvRecordData(record, *iterRecord);
        m_ContentDataList.push_back(record);
        m_ContentInfoMap.insert(std::make_pair(StringUtil::ToBit64(record.id), record));
    }
}

void ContentDataContainer::ToSyncListContainer(CsvDataTable& inCSVData) NN_NOEXCEPT
{
    if (inCSVData.size() < 2)
    {
        // CSVデータとしては不完全なデータ
        NN_LOG("[QCIT][Error] ToSyncListContainer() : Not CSV Data\n");
        return;
    }

    this->InitializeTitleIndexMap();

    auto iterRecord = std::begin(inCSVData);
    for (auto& titlePair : m_TitleIndexMap)
    {
        titlePair.second = this->FindIndexVector((*iterRecord), titlePair.first);
    }

    m_SyncList.clear();

    // 2行目以降のデータが各テストデータのレコードとなる
    ++iterRecord;
    for (; iterRecord != std::end(inCSVData); ++iterRecord)
    {
        SyncListRecord record;
        this->SetSyncListCsvRecordData(record, *iterRecord);
        m_SyncList.push_back(record);
    }
}

int ContentDataContainer::FindIndexVector(CsvRecord& inFindList, const std::string& inFindValue) NN_NOEXCEPT
{
    // 検索対象の値のイテレーターを取得
    auto iter = std::find(std::begin(inFindList), std::end(inFindList), inFindValue);
    // 取得したイテレーターのインデックス値を計算
    size_t index = std::distance(std::begin(inFindList), iter);
    if (index == inFindList.size())
    {
        //NN_LOG(" [Warning] Not Found CSV_Title : %s\n", inFindValue.c_str());
        // 該当なしであれば -1 を返す
        return -1;
    }
    return static_cast<int>(index);
}

std::string ContentDataContainer::SafeAt(CsvRecord& vec, const std::string& titleStr) NN_NOEXCEPT
{
    const int index = m_TitleIndexMap[titleStr];
    if (index < 0 || static_cast<int>(vec.size()) <= index)
    {
        //NN_LOG(" [Warning] Invalid Index : %d (%s)\n", index, titleStr.c_str());
        // リストの範囲外をアクセスしようとすると空文字を返しておく
        return std::string("");
    }
    return vec.at(index);
}

void ContentDataContainer::SetCsvRecordData(ContentRecord& outRecord, CsvRecord& inCSVRecordData) NN_NOEXCEPT
{
    auto& rec = outRecord;
    auto& rd = inCSVRecordData;

    rec.type = SafeAt(rd, CONTENT_TYPE);
    rec.nspFileName = SafeAt(rd, CONTENT_NSP_FILE_NAME);
    rec.digest = SafeAt(rd, CONTENT_DIGEST);
    rec.dataSize = SafeAt(rd, CONTENT_DATA_SIZE);
    rec.id = SafeAt(rd, CONTENT_ID);
    rec.applicationId = SafeAt(rd, CONTENT_APPLICATION_ID);
}

void ContentDataContainer::SetSyncListCsvRecordData(SyncListRecord& outRecord, CsvRecord& inCSVRecordData) NN_NOEXCEPT
{
    auto& rec = outRecord;
    auto& rd = inCSVRecordData;

    rec.type = SafeAt(rd, CONTENT_TYPE);
    rec.id = SafeAt(rd, CONTENT_APPLICATION_ID);
    rec.index = SafeAt(rd, CONTENT_INDEX);
}

} // ~namespace qcit
