﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#define _USE_MATH_DEFINES

#include <glv_core.h>
#include <glv_texture.h>

namespace qcit
{

/*********************************
 * struct Icon
 *********************************/

struct Icon
    : public glv::View
{
    mutable nn::os::Mutex m_Lock;
    glv::Texture2* m_pTexture;
    glv::space_t m_PaddingX;
    glv::space_t m_PaddingY;

    Icon( glv::GLsizei width, glv::GLsizei height, glv::space_t paddingX, glv::space_t paddingY ) NN_NOEXCEPT;
    virtual ~Icon() NN_NOEXCEPT;
    void SetRawTexture( const unsigned char* rawImageData, uint32_t rawImageSize ) NN_NOEXCEPT;
    void ClearTexture() NN_NOEXCEPT;
    virtual void onDraw( glv::GLV& g ) NN_NOEXCEPT;

private:
    void ActivateTexture() NN_NOEXCEPT;
};

/**
 * @brief       画像データ情報の構造体です。
 */
struct ImageDataInfo
{
    uint32_t        alignment;      //!< 一行のアライメント
    uint32_t        format;         //!< 画像のフォーマット
    uint32_t        type;           //!< 画素のデータタイプ
    std::int32_t    width;          //!< 画像の幅
    std::int32_t    height;         //!< 画像の高さ
    std::int32_t    stride;         //!< 画像のストライド（バイト）
    size_t          size;           //!< 画像のサイズ（バイト）
    uint8_t*        data;           //!< 画像のデータ
};

/**
 * @brief       JPEG ファイルからイメージを読み込んで、画像データ情報に設定します。
 *
 * @param[in]   image       画像データ情報
 * @param[in]   path        JPEG ファイルのパス
 *
 * @pre         画像データ情報の alignment に 1 以上の値を設定してください。
 */
void LoadJpegImage(ImageDataInfo& image, const char* path) NN_NOEXCEPT;

/**
 * @brief       画像データ情報に設定されているデータを解放します。
 *
 * @param[in]   image       画像データ情報
 */
void ReleaseImage(ImageDataInfo& image) NN_NOEXCEPT;

/**
 * @brief       画像データを扱うクラスです。
 */
class ImageData
{
    NN_DISALLOW_COPY( ImageData );
    NN_DISALLOW_MOVE( ImageData );

public:
    /**
     * @brief       コンストラクタです。
     */
    ImageData() NN_NOEXCEPT { InitializeImageDataInfo(1); }

    /**
     * @brief       コンストラクタです。
     *
     * @param[in]   alignment   一行のアライメント
     */
    explicit ImageData(uint32_t alignment) NN_NOEXCEPT { InitializeImageDataInfo(alignment); }

    /**
     * @brief       デストラクタです。
     */
    virtual ~ImageData() NN_NOEXCEPT {  ReleaseImage(); }

    /**
     * @brief       JPEG ファイルからイメージを読み込みます。
     *
     * @param[in]   path        JPEG ファイルのパス
     */
    void LoadJpegImage(const char* path) NN_NOEXCEPT { ::qcit::LoadJpegImage(m_ImageDataInfo, path); }

    /**
     * @brief       読み込んだイメージを解放します。
     */
    void ReleaseImage() NN_NOEXCEPT { ::qcit::ReleaseImage(m_ImageDataInfo); }


    operator const ImageDataInfo&() const { return m_ImageDataInfo; }

protected:
    /**
     * @brief       画像データ情報を初期化します。
     *
     * @param[in]   alignment   一行のアライメント
     */
    void InitializeImageDataInfo(uint32_t alignment) NN_NOEXCEPT
    {
        std::memset(&m_ImageDataInfo, 0, sizeof(m_ImageDataInfo));
        m_ImageDataInfo.alignment = alignment;
    }

private:
    ImageDataInfo m_ImageDataInfo;  //!< 画像データ情報
};


/**
 * @brief       画画像データ情報からテクスチャーを作成して表示するビューです。
 */
class TextureView : public glv::View
{
    NN_DISALLOW_COPY( TextureView );
    NN_DISALLOW_MOVE( TextureView );

public:
    /**
     * @brief       コンストラクタです。
     */
    explicit TextureView() NN_NOEXCEPT;

    /**
     * @brief       コンストラクタです。
     *
     * @param[in]   rect        ビューの矩形
     */
    explicit TextureView(const glv::Rect& rect) NN_NOEXCEPT;

    /**
     * @brief       デストラクタです。
     */
    virtual ~TextureView() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       画画像データ情報からテクスチャーを作成します。
     *
     * @param[in]   image       画像データ情報
     * @param[in]   fitting     ビューの幅高を画像に合わせるなら真を指定
     *
     * @return      自身のインスタンスを返します。
     *
     * @pre         テクスチャーが作成済みならば、先に破棄してください。
     */
    TextureView& CreateTexture(const ImageDataInfo& image, bool fitting) NN_NOEXCEPT;

    /**
     * @brief       テクスチャーを破棄します。
     *
     * @return      自身のインスタンスを返します。
     */
    TextureView& DestoryTexture() NN_NOEXCEPT;

    /**
     * @brief       テクスチャーを描画します。
     *
     * @param[in]   g           描画対象のルートビュー
     */
    virtual void onDraw(glv::GLV& g) NN_NOEXCEPT NN_OVERRIDE;

private:
    glv::Texture2* m_pTexture;      //!< テクスチャー
};

} // ~namespace qcit
