﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <array>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/util/util_StringView.h>

#include "QCIT_FsUtilities.h"

namespace qcit {
namespace fsutil {

bool IsExistPath(const char* inPath) NN_NOEXCEPT
{
    nn::fs::DirectoryEntryType entryType;

    // 指定パスの存在確認
    const nn::Result result = nn::fs::GetEntryType(&entryType, inPath);
    if(result.IsSuccess() == true)
    {
        // 指定されたファイルまたはディレクトリのパスが存在する
        return true;
    }

    if( nn::fs::ResultPathNotFound::Includes(result) == true )
    {
        // 指定パスが存在しない場合はこちらに入るはず。ログを出してもよいが・・
    }
    else
    {
        NN_LOG("Unexpected result\n");
    }

    // 存在していてもエラーとなった場合も存在しない扱いとする？
    // ひとまず存在しない(false)を返しておく
    return false;
}

bool IsAbsolutePath(const char* path) NN_NOEXCEPT
{
    nn::util::string_view view(path);
    return view.length() >= 2 && view[1] == ':';
}

bool IsNspFile(const char* path) NN_NOEXCEPT
{
    nn::util::string_view view(path);

    auto pos = view.find_last_of(".");

    // 拡張子がないファイル
    if (pos == nn::util::string_view::npos)
    {
        return false;
    }
    // 拡張子が .XYZ の形でない
    if (view.size() - pos != 4)
    {
        return false;
    }

    // 拡張子を取り出し、比較
    nn::util::string_view ext = view.substr(pos + 1, view.size() - pos);
    if (ext != "nsp")
    {
        return false;
    }

    return true;
}

nn::Result GetDataOfFile(const std::string& inFilePath, std::string& outFileData) NN_NOEXCEPT
{
    nn::Result result;

    File file;
    result = file.Open(inFilePath.c_str(), nn::fs::OpenMode_Read);
    if (result.IsFailure())
    {
        NN_LOG("[QCIT][Error] GetDataOfFile() : Open Failed : path=%s", inFilePath.c_str());
        return result;
    }

    const size_t fileSize = static_cast<size_t>(file.GetSize());
    std::unique_ptr<char[]> buf(new char[fileSize + 1]);

    result = file.Read(0, buf.get(), fileSize);
    if (result.IsFailure())
    {
        NN_LOG("[QCIT][Error] GetDataOfFile() : Read Failed : path=%s", inFilePath.c_str());
        return result;
    }
    // 念のため明示的にクローズしておく
    file.Close();

    // UTF-8 のファイルデータがそのまま読み込まれた場合の処理
    // 文字列として扱うため、末尾にヌル文字を入れておく
    buf[fileSize] = '\0';

    // BOM 付きデータにも対応するための判定処理
    char* strPtr = buf.get();
    const std::array<uint8_t, 3> utf8BomData = { { 0xEF, 0xBB, 0xBF } };
    if (memcmp(buf.get(), utf8BomData.data(), utf8BomData.size()) == 0)
    {
        // BOM付きデータであるためBOMのサイズ分ずらす
        strPtr += utf8BomData.size();
    }

    outFileData = strPtr;

    return result;
}

int64_t GetFileSize(const std::string& inFilePath) NN_NOEXCEPT
{
    if (IsExistPath(inFilePath.c_str()) == false)
    {
        // 存在しない場合は 0 を返す
        return 0;
    }

    File file;
    auto result = file.Open(inFilePath.c_str(), nn::fs::OpenMode_Read);
    if (result.IsFailure())
    {
        // ファイルオープンに失敗する場合は 0 を返す
        // (ディレクトリパスが渡された場合を考慮)
        NN_LOG("[QCIT][Error] GetFileSize() : Open Failed : path=%s", inFilePath.c_str());
        return 0;
    }

    return file.GetSize();
}

// -------------------------------------------------------------

nn::Result GetNspFileList(const std::string& inTargetDirectoryPath, FileInfoList& outFileInfoList) NN_NOEXCEPT
{
    fsutil::Directory dir;
    auto result = dir.Open(inTargetDirectoryPath.c_str(), nn::fs::OpenDirectoryMode_All);
    if (result.IsFailure())
    {
        return result;
    }

    while (NN_STATIC_CONDITION(true))
    {
        int64_t entryCount = 0;
        nn::fs::DirectoryEntry entry;

        result = dir.Read(&entryCount, &entry, 1);
        if (result.IsFailure())
        {
            NN_LOG("[QCIT][Error] GetNspFileList Failed : DirectoryEntry::Read() , result = 0x%08x, path = %s\n",
                result.GetInnerValueForDebug(), inTargetDirectoryPath.c_str());
            return result;
        }

        if (0 >= entryCount)
        {
            break;
        }

        const auto targetPath = inTargetDirectoryPath + entry.name;
        if (entry.directoryEntryType == nn::fs::DirectoryEntryType_File)
        {
            if (IsNspFile(targetPath.c_str()))
            {
                FileInfo info;
                info.path = targetPath;

                File file;
                file.Open(targetPath.c_str(), nn::fs::OpenMode_Read);
                info.size = file.GetSize();

                outFileInfoList.push_back(info);
            }
        }
        else if (entry.directoryEntryType == nn::fs::DirectoryEntryType_Directory)
        {
            auto chiledDirectoryPath = targetPath + "/";
            // 子ディレクトリに対して再帰的に関数を呼び出す
            result = GetNspFileList(chiledDirectoryPath, outFileInfoList);
            if (result.IsFailure())
            {
                return result;
            }
        }
    }

    return result;
}

// -------------------------------------------------------------

Directory::Directory() NN_NOEXCEPT
{
    m_Handle.handle = nullptr;
}

Directory::Directory(const char* inDirPath) NN_NOEXCEPT
{
    m_Handle.handle = nullptr;
    this->Open(inDirPath);
}

Directory::~Directory() NN_NOEXCEPT
{
    this->Close();
}

bool Directory::IsOpened() NN_NOEXCEPT
{
    return ( m_Handle.handle != nullptr ) ? true : false;
}

nn::Result Directory::Open(const char* inPath, int mode) NN_NOEXCEPT
{
    // すでにクラス内で開いているディレクトリがあれば閉じる(強制的に指定したパスをオープンする形となる)
    this->Close();

    //NN_LOG("Call nn::fs::OpenDirectory()\n");
    const auto result = nn::fs::OpenDirectory(&m_Handle, inPath, mode);
    if( result.IsFailure() )
    {
        SetResult(result);
        // オープンに失敗した場合は、念のためハンドルにNULLを設定しておく
        m_Handle.handle = nullptr;
    }

    return result;
}

void Directory::Close() NN_NOEXCEPT
{
    if( IsOpened() == true )
    {
        nn::fs::CloseDirectory(m_Handle);
        m_Handle.handle = nullptr;
    }
}

nn::Result Directory::Read(int64_t* outValue, nn::fs::DirectoryEntry* entryBuffer, int64_t entryBufferCount) NN_NOEXCEPT
{
    // Openしていない場合についてはどうするか？
    // ⇒無効なハンドルとしてエラーとなるはずなので、そのままFS関数を呼ぶ

    const auto result = nn::fs::ReadDirectory(outValue, entryBuffer, m_Handle ,entryBufferCount);
    if( result.IsFailure() )
    {
        SetResult(result);
    }

    return result;
}

int64_t Directory::GetEntryCount() NN_NOEXCEPT
{
    // Openしていない場合についてはどうするか？
    // ⇒ひとまず0を返しておく

    int64_t count = 0;
    const auto result = nn::fs::GetDirectoryEntryCount(&count, m_Handle);
    if( result.IsFailure() )
    {
        SetResult(result);
        return 0;
    }

    return count;
}

// -------------------------------------------------------------

File::File() NN_NOEXCEPT
{
    m_Handle.handle = nullptr;
}

File::File(const char* inFilePath) NN_NOEXCEPT
{
    m_Handle.handle = nullptr;
    this->Open(inFilePath);
}

File::~File() NN_NOEXCEPT
{
    this->Close();
}

bool File::IsOpened() NN_NOEXCEPT
{
    return ( m_Handle.handle != nullptr ) ? true : false;
}

nn::Result File::Open(const char* inPath, int mode) NN_NOEXCEPT
{
    // すでにクラス内で開いているディレクトリがあれば閉じる(強制的に指定したパスをオープンする形となる)
    this->Close();

    //NN_LOG("Call nn::fs::OpenFile()\n");
    const auto result = nn::fs::OpenFile( &m_Handle, inPath, mode);
    if( result.IsFailure() )
    {
        SetResult(result);
        // オープンに失敗した場合は、念のためハンドルにNULLを設定しておく
        m_Handle.handle = nullptr;
    }

    return result;
}

void File::Close() NN_NOEXCEPT
{
    if( IsOpened() == true )
    {
        //NN_LOG("Call nn::fs::CloseFile()\n");
        nn::fs::CloseFile(m_Handle);
        m_Handle.handle = nullptr;
    }
}

nn::Result File::Read(int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT
{
    // Openしていない場合についてはどうするか？
    // ⇒無効なハンドルとしてエラーとなるはずなので、そのままFS関数を呼ぶ

    const auto result = nn::fs::ReadFile( m_Handle, offset, buffer, size, option);
    if( result.IsFailure() )
    {
        SetResult(result);
    }

    return result;
}

nn::Result File::Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    // Openしていない場合についてはどうするか？
    // ⇒無効なハンドルとしてエラーとなるはずなので、そのままFS関数を呼ぶ

    const auto result = nn::fs::ReadFile( m_Handle, offset, buffer, size);
    if( result.IsFailure() )
    {
        SetResult(result);
    }

    return result;
}

nn::Result File::Read(size_t* outValue, int64_t offset, void* buffer, size_t size, const nn::fs::ReadOption& option) NN_NOEXCEPT
{
    // Openしていない場合についてはどうするか？
    // ⇒無効なハンドルとしてエラーとなるはずなので、そのままFS関数を呼ぶ

    const auto result = nn::fs::ReadFile(outValue, m_Handle, offset, buffer, size, option);
    if( result.IsFailure() )
    {
        SetResult(result);
    }

    return result;
}

nn::Result File::Read(size_t* outValue, int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
{
    // Openしていない場合についてはどうするか？
    // ⇒無効なハンドルとしてエラーとなるはずなので、そのままFS関数を呼ぶ

    const auto result = nn::fs::ReadFile(outValue, m_Handle, offset, buffer, size);
    if( result.IsFailure() )
    {
        SetResult(result);
    }

    return result;
}

nn::Result File::Write(int64_t offset, void* buffer, size_t size, const nn::fs::WriteOption& option) NN_NOEXCEPT
{
    // Openしていない場合についてはどうするか？
    // ⇒無効なハンドルとしてエラーとなるはずなので、そのままFS関数を呼ぶ

    const auto result = nn::fs::WriteFile( m_Handle, offset, buffer, size, option);
    if( result.IsFailure() )
    {
        SetResult(result);
    }

    return result;
}

nn::Result File::Flush() NN_NOEXCEPT
{
    // Openしていない場合についてはどうするか？
    // ⇒無効なハンドルとしてエラーとなるはずなので、そのままFS関数を呼ぶ

    const auto result = nn::fs::FlushFile(m_Handle);
    if( result.IsFailure() )
    {
        SetResult(result);
    }

    return result;
}

nn::Result File::SetSize(int64_t size) NN_NOEXCEPT
{
    // Openしていない場合についてはどうするか？
    // ⇒無効なハンドルとしてエラーとなるはずなので、そのままFS関数を呼ぶ

    const auto result = nn::fs::SetFileSize(m_Handle, size);
    if( result.IsFailure() )
    {
        SetResult(result);
    }

    return result;
}

int64_t File::GetSize() NN_NOEXCEPT
{
    // Openしていない場合についてはどうするか？
    // ⇒ひとまず0を返しておく

    int64_t size = 0;
    const auto result = nn::fs::GetFileSize(&size, m_Handle);
    if( result.IsFailure() )
    {
        SetResult(result);
        return 0;
    }

    return size;
}

int File::GetFileOpenMode() NN_NOEXCEPT
{
    // Openしていない場合についてはどうするか？
    // ⇒無効なハンドルとしてエラーとなるはずなので、そのままFS関数を呼ぶ (返り値はFSの仕様依存)

    return nn::fs::GetFileOpenMode(m_Handle);
}

} // namespace fsutil
} // namespace qcit

