﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <glv_CustomVerticalListView.h>
#include <glv_ScissorBoxView.h>

#include "QCIT_ModalView.h"

namespace qcit
{
    template< size_t TemporarySize, size_t OutBufferSize >
    glv::WideCharacterType* BuildUtf16(glv::WideCharacterType(&pOutBuffer)[OutBufferSize], const char* pFormat, ...) NN_NOEXCEPT
    {
        va_list vaList;
        va_start(vaList, pFormat);

        char pTemp[TemporarySize];
        std::vsprintf(pTemp, pFormat, vaList);
        nn::util::ConvertStringUtf8ToUtf16Native(reinterpret_cast< uint16_t* >(pOutBuffer), OutBufferSize, pTemp);

        va_end(vaList);
        return pOutBuffer;
    }

    class Button : public glv::Button
    {
    protected:
        glv::Label              m_Label;

    private:
        std::function< void() > m_Callback;
        glv::Style              m_StyleLabel[3];
        glv::Style              m_StyleButton[3];

        enum ButtonStyle
        {
            ButtonStyle_Focusable,
            ButtonStyle_NotFocusable,
            ButtonStyle_FocusableButInvalid,
        };

        /**
        *@brief 自身( Button )の表示領域を指定のラベル情報にフィットさせます.
        */
        void FitButton(float fontSize, glv::space_t paddingX, glv::space_t paddingY) NN_NOEXCEPT;

    public:
        Button(const glv::WideString& caption, std::function< void() > callback, const glv::Rect& r, glv::Place::t anchor = glv::Place::TL) NN_NOEXCEPT;

        Button(const char* caption, std::function< void() > callback, const glv::Rect& r, glv::Place::t anchor = glv::Place::TL) NN_NOEXCEPT;

        /**
        * @brief コンストラクタ.
        * @details 1バイト文字列( ASCII, Latin-1 )を対象にした初期化です.
        */
        explicit Button(const std::string& text, std::function< void() > callback = nullptr, float fontSize = CommonValue::InitialFontSize, glv::space_t paddingX = 8.0f, glv::space_t paddingY = 4.0f) NN_NOEXCEPT;

        /**
        * @brief コンストラクタ.
        * @details 2バイト文字列( UTF-16 )を対象にした初期化です.
        */
        explicit Button(const glv::WideString& text, std::function< void() > callback = nullptr, float fontSize = CommonValue::InitialFontSize, glv::space_t paddingX = 8.0f, glv::space_t paddingY = 4.0f) NN_NOEXCEPT;

        void Initialize() NN_NOEXCEPT;

        void UpdateFocusAndColor(bool isValid, bool isFocusable = false) NN_NOEXCEPT;

        void UpdateLabelText(const char* text) NN_NOEXCEPT;

        void EnableLabelView() NN_NOEXCEPT{ m_Label.enable(glv::Property::Visible); };

        void DisableLabelView() NN_NOEXCEPT{ m_Label.disable(glv::Property::Visible); };

        void IncreaseLabelSize(glv::space_t increasedX, glv::space_t increasedY) NN_NOEXCEPT;

        void ChangeLabelSize(glv::space_t width, glv::space_t height) NN_NOEXCEPT;
    };

/**
 * @brief インストール確認モーダルビュー
 */
class ConfirmView : public MessageView
{
    NN_DISALLOW_COPY( ConfirmView );
    NN_DISALLOW_MOVE( ConfirmView );

public:
    ConfirmView() NN_NOEXCEPT;
    virtual ~ConfirmView() NN_NOEXCEPT NN_OVERRIDE {};
};

struct ResultPropertyType
{
    glv::WideCharacterType fileName[48];
    glv::WideCharacterType result[64];
    bool isSyncFailure;

    ResultPropertyType() NN_NOEXCEPT : isSyncFailure(false) {}
};

class InstallResultListView : public glv::CustomVerticalListView< ResultPropertyType >
{
public:

    //----------------------------------------------------------

    typedef glv::CustomVerticalListView< ResultPropertyType > ParentType;

    /**
    * @brief コンストラクタです。
    */
    explicit InstallResultListView(const glv::Rect& parentClipRegion) NN_NOEXCEPT
        : CustomVerticalListView(parentClipRegion)
    {
        SetTouchAndGo(false);
        glv::Style* pStyle = new glv::Style();
        pStyle->color = glv::Style::standard().color;
        pStyle->color.selection.set(0.1f, 0.85f, 0.2f);
        style(pStyle);
        font().size(21.f);
    }

protected:
    //----------------------------------------------------------
    /**
    * @copydoc CustomVerticalListView<>::OnQueryBounds( const CustomVerticalListView<>::ItemType&, glv::space_t&, glv::space_t& )
    */
    virtual void OnQueryBounds(const ItemType& item, glv::space_t& outWidth, glv::space_t& outHeight) NN_NOEXCEPT NN_OVERRIDE
    {
        this->font().getBounds(outWidth, outHeight, item.fileName);
        outWidth = this->width();
    }

    //----------------------------------------------------------
    /**
    * @copydoc CustomVerticalListView<>::OnDrawItem( const CustomVerticalListView<>::ItemType&, const CustomVerticalListView<>::IndexType, const glv::Rect& )
    */
    virtual void OnDrawItem(const ItemType& item, const IndexType index, const glv::Rect& contentRegion) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(index);
        auto& font = this->font();

        if (item.isSyncFailure)
        {
            // 赤文字にする
            glv::Color clr(1.0f, 0.0f, 0.0f);
            glv::draw::color(clr);
        }

        auto pos = contentRegion.left() + 4.0f;
        font.render(item.fileName, pos, contentRegion.top());

        pos = contentRegion.left() + 600.0f;
        font.render(item.result, pos, contentRegion.top());
    }
};

/**
 * @brief 進捗表示モーダルビュー
 */
class InstallView : public MessageView
{
    NN_DISALLOW_COPY( InstallView );
    NN_DISALLOW_MOVE( InstallView );

    static const size_t       InitialActionCapacity; // 実行情報の初期数
    static const glv::space_t InstallViewWidth;
    static const glv::space_t ProgressBarMargineX;
    static const glv::space_t MessageButtonPaddingX;
    static const glv::space_t MessageButtonPaddingY;
    static const glv::space_t MessageViewPaddingX;
    static const glv::space_t MessageViewPaddingY;

public:
    /**
     * @brief       コンストラクタです。
     */
    InstallView( std::function< void() > closeCallback, std::function< void() > cancelCallback, std::function< void() > cleanupCallback, std::function< void() > nextCallback ) NN_NOEXCEPT;

    /**
     * @brief       デストラクタです。
     */
    virtual ~InstallView() NN_NOEXCEPT NN_OVERRIDE;

     /**
      * @brief      メッセージを追加します。
      *
      * @param[in]   pProgressCaption       状態を表示するためのラベル
      * @param[in]   pInstallingFileName    インストールしているファイル名を表示するためのラベル
      * @param[in]   pProgressLabel         インストール容量を表示するラベル
      */
    InstallView& AddProgressMessage( glv::Label* pProgressCaption, glv::Label* pProgressFileName, glv::Label* pProgressLabel ) NN_NOEXCEPT;

    /**
     * @brief       ボタンを追加します。
     *
     * @param[in]   pButton      ボタン
     * @param[in]   isExitModal  ボタン押下後にモーダルを終了させるかどうかのフラグ設定(デフォルトは有効)
     */
    InstallView& AddButton( glv::Button* pButton, bool isExitModal = true) NN_NOEXCEPT;

    /**
     * @brief       ボタンを追加します。
     *
     * @param[in]   pButton      ボタン
     * @param[in]   pFunc        ボタン押下時に実行される関数ポインタ
     * @param[in]   pParam       ボタン押下時に実行される関数への引数
     * @param[in]   isExitModal  ボタン押下後にモーダルを終了させるかどうかのフラグ設定(デフォルトは有効)
     */
    InstallView& AddButton(glv::Button* pButton, ActionFunc pFunc, void* pParam, bool isExitModal = true) NN_NOEXCEPT;

    /**
     * @brief       進捗状況バーを追加します。
     *
     * @param[in]   pProgressBar   進捗状況バーのスライダー
     */
    InstallView& AddProgressBar( glv::Slider* pProgressBar ) NN_NOEXCEPT;

    InstallView& AddProgressBar(glv::Slider* pProgressBar, glv::Label* pTotalProgressLabel, glv::Slider* pTotalProgressBar) NN_NOEXCEPT;

    /**
     * @brief       インストールしているファイル名を設定します。
     *
     * @param[in]   fileName        ファイル名
     */
    void SetProgressFileName( const char* fileName ) NN_NOEXCEPT
    {
        m_pProgressFileName->setValue( fileName );
    }

    /**
     * @brief       インストール進捗メッセージを設定します。
     *
     * @param[in]   message        表示内容
     */
    void SetProgressLabel( const char* message ) NN_NOEXCEPT
    {
        m_pProgressLabel->setValue( message );
    }

    /**
    * @brief       インストール全体進捗メッセージを設定します。
    *
    * @param[in]   message        表示内容
    */
    void SetTotalProgressLabel(const char* message) NN_NOEXCEPT
    {
        m_pTotalProgressLabel->setValue(message);
    }

    /**
     * @brief       直近でインストールしたファイルのサイズを取得します。
     * @return      ファイルのサイズ
     */
    int64_t GetLastNspTotalSize() NN_NOEXCEPT
    {
        return m_LastNspTotalSize;
    }

    /**
     * @brief       直近でインストールしたファイルのサイズを設定します。
     *
     * @param[in]   totalSize       ファイルサイズ
     */
    void SetLastNspTotalSize( int64_t totalSize ) NN_NOEXCEPT
    {
        m_LastNspTotalSize = totalSize;
    }

    /**
     * @brief       進捗状況バーの進捗度を設定します。
     *
     * @param[in]   value       進捗度( 0.0f - 1.0f )
     */
    void SetProgressBarValue( double value ) NN_NOEXCEPT
    {
        m_pProgressBar->setValue( value );
    }

    /**
    * @brief       全体進捗状況バーの進捗度を設定します。
    *
    * @param[in]   value       進捗度( 0.0f - 1.0f )
    */
    void SetTotalProgressBarValue(double value) NN_NOEXCEPT
    {
        m_pTotalProgressBar->setValue(value);
    }

    /**
     * @brief       インストール状態メッセージを設定します。
     *
     * @param[in]   message        表示内容
     */
    void SetProgressCaption( const char* message ) NN_NOEXCEPT
    {
        m_pProgressCaption->setValue( message );
    }

    /**
     * @brief       エラー画面を構築します。
     */
    void SetErrorMessageView( bool isLastTarget ) NN_NOEXCEPT;

    void SetInstallResultMessageView() NN_NOEXCEPT;

    /**
     * @brief       インストール結果を設定します。
     */
    void SetInstallResult( nn::Result result ) NN_NOEXCEPT
    {
        m_InstallResult = result;
    }

    /**
     * @brief       インストール完了ボタンを設定します。
     */
    void SetCloseButton() NN_NOEXCEPT;

    /**
     * @brief       ButtonsTable に追加された Button を全て削除します。
     */
    void RemoveButtons() NN_NOEXCEPT;

    void SetInstallFileNameList(std::set<std::string>& inSetList) NN_NOEXCEPT;
    void AddInstallResultList(const std::string& inFileName, nn::Result inResult, int64_t inElapsedSecond) NN_NOEXCEPT;
    void SetSyncFailIdList(std::set<nn::Bit64>& inFaildIdList) NN_NOEXCEPT;
    void SetSyncFailAocIndexList(nn::Bit64 inTargetAppId, std::set<int>& inFaildIndexList) NN_NOEXCEPT;

    void RecoverErrorToProgressMessage() NN_NOEXCEPT;

    void SetTotalElapsedSecond(int64_t inElapsedSecond) NN_NOEXCEPT;

protected:
    virtual void Initialize() NN_NOEXCEPT;

    glv::Table              m_ProgressBarsTable;    //!< スライダー配置テーブル
    glv::Slider*            m_pProgressBar;         //!< 進捗状況を表示するためのバー
    glv::Slider*            m_pTotalProgressBar;    //!< 全体進捗状況を表示するためのバー
    glv::Label*             m_pProgressCaption;     //!< 状態を表示するためのラベル
    glv::Label*             m_pProgressFileName;    //!< インストールしているファイル名を表示するためのラベル
    glv::Label*             m_pProgressLabel;       //!< インストール容量を表示するラベル
    glv::Label*             m_pTotalProgressLabel;  //!< インストール全体状況を表示するラベル
    Button*                 m_pCloseButton;         //!< インストール完了時に表示されるボタン
    Button*                 m_pCancelButton;        //!< InstallAll 中の失敗時に表示されるボタン（インストールをキャンセルする）
    Button*                 m_pNextButton;          //!< InstallAll 中の失敗時に表示されるボタン（次の nsp をインストールする）

    glv::ScissorBoxView*    m_pParentResultListView;
    InstallResultListView::CollectionType m_InstallResultList;
    InstallResultListView*  m_pInstallResultListView;

    struct ResultView
    {
        std::string resultString;
        bool isSuccess;

        ResultView(const std::string& inResult, bool inIsSuccess = false) NN_NOEXCEPT : resultString(inResult), isSuccess(inIsSuccess) {}
    };
    std::map<std::string, ResultView> m_FileResultMap;
    std::string             m_TotalElapsedTime;

    glv::Style              m_ResultStringStyle;

    nn::Result              m_InstallResult;        //!< インストール結果
    int64_t                 m_LastNspTotalSize;     //!< インストール完了表示用の直近でインストールしたファイルサイズ
};

} // ~namespace qcit
