﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/init.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <glv.h>
#include <glv_binding.h>
#include <glv_resources.h>
#include <nv/nv_MemoryManagement.h>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <nn/repair.h>
#include <nn/repair/repair_LabelText.h>
#include <nn/repair/repair_StreamTextView.h>
#include <nn/repair/repair_ShutdownButton.h>
#include <nn/repair/repair_Sdcard.h>
#include <nn/fs/fs_IEventNotifier.h>
#include <nn/fs/fs_SdCardPrivate.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_MultipleWait.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/util/util_FormatString.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiForSystem.h>
#include <nn/nifm/nifm_Api.h>
#include <nn/nifm/nifm_TemporaryNetworkProfile.h>
#include <nn/nifm/nifm_NetworkConnection.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nifm/nifm_ApiForTest.h>
#include <nn/nim/nim_Result.h>
#include <nn/repair/repair_NetworkSettingsParser.h>
#include <nn/nifm/nifm_ApiWirelessCommunicationControl.h>
#include <nn/time/time_ApiForRepair.h>
#include <nn/time/time_StandardNetworkSystemClock.h>
#include <nn/bpc/bpc_Rtc.h>
#include <nn/settings/system/settings_Boot.h>

#include <functional>

#include "RepairTimeReviser_LabelButton.h"

namespace {

static const char* ToolName = "NX Time Reviser Tool";
const int ToolMajorVersion = 6;
const int ToolMinorVersion = 0;

static const size_t GraphicsSystemReservedMemorySize = 8 * 1024 * 1024;
static glv::Label s_ToolInformationLabel;
static glv::Label s_ProgressLabel;
static nn::repair::StreamTextView* s_pStreamText;
LabelButton* s_pLabelButton;
nn::os::ThreadType s_WorkerThread;
nn::os::ThreadType s_SdCardDetectThread;
std::unique_ptr<nn::fs::IEventNotifier> s_Notifier;

nn::nifm::NetworkProfileData g_NetworkSettings;

bool g_AirplaneModeEnabled = false;

const int StackSize = 50 * 4096;
NN_ALIGNAS(nn::os::ThreadStackAlignment) uint8_t s_Stack[StackSize];

const int OperationStackSize = 50 * 4096;
NN_ALIGNAS(nn::os::ThreadStackAlignment) uint8_t s_OperationStack[OperationStackSize];

static void* Allocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, size);
}

static void Deallocate(void* address, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(address);
}

static void* Reallocate(void* address, size_t size, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(address, size);
}

static void SetupPeripherals() NN_NOEXCEPT
{
    nv::SetGraphicsAllocator(Allocate, Deallocate, Reallocate, nullptr);
    nv::InitializeGraphics(malloc(GraphicsSystemReservedMemorySize), GraphicsSystemReservedMemorySize);
    glslcSetAllocator(Allocate, Deallocate, Reallocate, nullptr);
}

void WaitNetworkConnectionAvailable(nn::nifm::TemporaryNetworkProfile* temporaryNetworkProfile,
    nn::nifm::NetworkConnection* networkConnection) NN_NOEXCEPT
{
    if (networkConnection->IsAvailable())
    {
        return;
    }

    nn::nifm::RequestHandle requestHandle = networkConnection->GetRequestHandle();
    nn::util::Uuid profileId = temporaryNetworkProfile->GetId();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::SetRequestNetworkProfileId(requestHandle, profileId));

    auto count = 0;
    while (NN_STATIC_CONDITION(1))
    {
        s_pStreamText->AppendValue("Connecting...\n");
        networkConnection->SubmitRequestAndWait();

        if (networkConnection->IsAvailable())
        {
            s_pStreamText->AppendValue("Network is available.\n");
            break;
        }
        else
        {
            std::string message = "Network is not available. retry ";
            message += std::to_string(count++);
            message += ". result = ";
            message += networkConnection->GetResult().GetInnerValueForDebug();
            message += "\n";
            s_pStreamText->AppendValue(message.c_str());
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
        }
    }
}

void EnableWifiConnection() NN_NOEXCEPT
{
    g_AirplaneModeEnabled = !nn::nifm::IsWirelessCommunicationEnabled();
    nn::nifm::SetWirelessCommunicationEnabled(true);
}

void RestoreAirplaneMode() NN_NOEXCEPT
{
    if (g_AirplaneModeEnabled) {
        nn::nifm::SetWirelessCommunicationEnabled(false);
    }
}

void CalibrateSystemClock(nn::nifm::TemporaryNetworkProfile* temporaryNetworkProfile,
    nn::nifm::NetworkConnection* networkConnection) NN_NOEXCEPT
{
    auto count = 0;
    while (NN_STATIC_CONDITION(1))
    {
        WaitNetworkConnectionAvailable(temporaryNetworkProfile, networkConnection);

        auto result = nn::time::CalibrateSystemClockWithInternalOffset();
        if (result.IsSuccess())
        {
            NN_LOG("CalibrateSystemClockWithInternalOffset succeeded\n");
            break;
        }

        std::string message = "Failed to Calibrate Clock. retry ";
        message += std::to_string(count++);
        message += ". result = ";
        message += std::to_string(result.GetInnerValueForDebug());
        message += "\n";
        s_pStreamText->AppendValue(message.c_str());
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(3));
    }
}

void OperationFunction() NN_NOEXCEPT
{
    //Wifi 接続を有効化する
    EnableWifiConnection();
    NN_UTIL_SCOPE_EXIT{
        RestoreAirplaneMode();
    };

    nn::nifm::TemporaryNetworkProfile temporaryNetworkProfile(g_NetworkSettings);
    nn::nifm::NetworkConnection networkConnection;

    // 時刻の特別補正
    CalibrateSystemClock(&temporaryNetworkProfile, &networkConnection);

    s_ProgressLabel.setValue("Progress: Finished");
    s_pStreamText->AppendValue("[[ SUCCESS ]]\n");
}

void StartRepairOperation() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(&s_WorkerThread,
                             (nn::os::ThreadFunction)OperationFunction,
                             NULL,
                             s_OperationStack,
                             OperationStackSize,
                             30));

    nn::os::StartThread(&s_WorkerThread);
}

static glv::GLV s_GlvRootView;

void DisplayToolInformation(glv::GLV* const pGlvRootView) NN_NOEXCEPT
{
    nn::repair::GetToolInformationLabel(&s_ToolInformationLabel, ToolName, ToolMajorVersion, ToolMinorVersion);

    *pGlvRootView << s_ToolInformationLabel;
}

nn::Result ImportNetworkSettingsFromSdCard(bool* pOutValue) NN_NOEXCEPT
{
    *pOutValue = false;

    const std::string MountName = "SdCard";
    NN_RESULT_DO(nn::fs::MountSdCard(MountName.c_str()));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount(MountName.c_str());
    };

    nn::fs::FileHandle handle;
    const std::string SerialFileName = (MountName + ":/settings.json");

    NN_RESULT_DO(nn::fs::OpenFile(&handle, SerialFileName.c_str(), nn::fs::OpenMode::OpenMode_Read));
    int64_t fileSize = 0;
    NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));

    int totalSize = static_cast<int>(fileSize);
    std::unique_ptr<char[]> storage(new char[totalSize]);

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, storage.get(), totalSize));

    auto text = std::string(storage.get(), totalSize);

    if (!nn::repair::ImportNetworkSettings(&g_NetworkSettings, text.c_str()))
    {
        s_pStreamText->AppendValue("Invaild NetworkSettings Format.\n");
        NN_RESULT_SUCCESS;
    }

    *pOutValue = true;

    NN_RESULT_SUCCESS;
}

nn::Result DetectSdCardInsertion() NN_NOEXCEPT
{
    // 刺し忘れに対応するために別スレッドで検出する
    NN_RESULT_DO(nn::os::CreateThread(&s_SdCardDetectThread, [](void *p)
    {
        for(;;)
        {
            std::unique_ptr<nn::repair::Sdcard> sdcard(new nn::repair::Sdcard());
            NN_ABORT_UNLESS_RESULT_SUCCESS(sdcard->WaitAttach());

            bool imported = false;
            auto result = ImportNetworkSettingsFromSdCard(&imported);
            if(result.IsFailure() || !imported)
            {
                s_pStreamText->AppendValue("Read settings.json in SD card failed. Please reinsert.\n");
                NN_ABORT_UNLESS_RESULT_SUCCESS(sdcard->WaitDetach());
                continue;
            }

            if(imported)
            {
                //進捗
                s_ProgressLabel.setValue("Progress: Waiting for the operation\n");
                s_pStreamText->AppendValue("Read settings.json succeeded\n");

                s_pLabelButton = new LabelButton("START", [&] { StartRepairOperation(); });
                s_GlvRootView << s_pLabelButton->pos(glv::Place::BC, 0, -30).anchor(glv::Place::BC);
                break;
            }
        }

    }, NULL, s_Stack, StackSize, 30));

    nn::os::StartThread(&s_SdCardDetectThread);

    NN_RESULT_SUCCESS;
}

bool IsTimeReviseRequired()
{
    nn::time::PosixTime time;

    auto result = nn::time::StandardNetworkSystemClock::GetCurrentTime(&time);
    return result.IsSuccess();
}

void ResetRtc()
{
    nn::bpc::InitializeRtc();
    nn::bpc::SetUpRtcResetOnShutdown();
    nn::bpc::FinalizeRtc();
}

nn::Result Initialize() NN_NOEXCEPT
{
    glv::GLV* const pGlvRootView = &s_GlvRootView;

    s_pStreamText = new nn::repair::StreamTextView(glv::Rect(1024, 440), 28.0f);
    s_pStreamText->pos(120, 164);
    *pGlvRootView << s_pStreamText;

    // シャットダウンボタン(FW 3.0.0 以降で有効)
    glv::Button* shutdownButton = new nn::repair::ShutdownButton();
    shutdownButton->pos(800,640);
    *pGlvRootView << shutdownButton;

    DisplayToolInformation(pGlvRootView);

    //進捗
    s_ProgressLabel.size(28);
    s_ProgressLabel.setValue("Progress: Waiting for insert the SdCard");
    *pGlvRootView << s_ProgressLabel.pos(glv::Place::TL, 120, 100);

    if (IsTimeReviseRequired())
    {
        NN_RESULT_DO(DetectSdCardInsertion());
    }
    else
    {
        ResetRtc();
        s_ProgressLabel.setValue("Progress: Finished");
        s_pStreamText->AppendValue("[[ SUCCESS(The correction is not necessary.) ]]\n");
    }

    nn::settings::system::SetRequiresRunRepairTimeReviser(false);
    NN_RESULT_SUCCESS;
}

void Main( glv::Window& window ) NN_NOEXCEPT
{
    glv::GLV* const pGlvRootView = &s_GlvRootView;

    nn::Result result = Initialize();
    if (result.IsFailure())
    {
        const int BufferSize = 200;
        char buffer[BufferSize];
        nn::util::SNPrintf(buffer, BufferSize, "Initialization Failed. Error code = 0x%08x.\n", result.GetInnerValueForDebug());

        s_pStreamText->AppendValue(std::string(buffer));
    }

    glv::Style::standard().color.set(glv::StyleColor::WhiteOnBlack);
    glv::Style::standard().color.fore.set( 0.5 );

    window.setGLV(*pGlvRootView);
    glv::Application::run();
}

} // namespace

extern "C" void nnMain()
{
    SetupPeripherals();

    glv::ApplicationFrameworkInitialize(glv::HidInitialConfiguration());

    const int width  = glv::glutGet(GLUT_SCREEN_WIDTH);
    const int height = glv::glutGet(GLUT_SCREEN_HEIGHT);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::InitializeSystem());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::SetExclusiveClient(nn::nifm::GetClientId()));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::InitializeForRepair());
    nn::time::SuspendAutonomicTimeCorrection();
    NN_UTIL_SCOPE_EXIT{
        nn::time::ResumeAutonomicTimeCorrection();
        nn::time::Finalize();
        nn::nifm::ClientId clientId = { nn::nifm::InvalidClientIdValue };
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::SetExclusiveClient(clientId));
    };

    glv::Window* window = new glv::Window(width, height, "Main Window");
    NN_ABORT_UNLESS_NOT_NULL(window);

    NN_UTIL_SCOPE_EXIT
    {
        delete window;
    };

    Main(*window);
}
