﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @page PageTargetToolRepairDeletePctl
 * @tableofcontents
 *
 * @brief
 * nn::repair ライブラリを用いて保護者設定を削除するプログラムの解説です。
 *
 * @section PageTargetToolRepairDeletePctl_SectionFileStructure ファイル構成
 * 本プログラムは @link ../../../Programs/Iris/Sources/TargetTools/RepairTools/RepairDeletePctl
 *  @endlink 以下にあります。
 *
 * @section PageTargetToolRepairDeletePctl_SectionNecessaryEnvironment 必要な環境
 * Ocean 実行環境
 *
 * @section PageTargetToolRepairDeletePctl_SectionHowToOperate 操作方法
 * 現在、保護者設定が有効か否かを右上に表示します。
 * 有効になっている場合、画面下のDelete ボタンを押すと保護者設定を削除します
 * リリース版では削除した設定を戻すことはできません。
 *
 * @section PageTargetToolRepairDeletePctl_SectionPrecaution 注意事項
 * 設定は Ocean で行ってください
 * このツールで保護者設定を行うことはできません
 *
 * @section PageTargetToolRepairDeletePctl_SectionHowToExecute 実行手順
 * ビルドし、実行してください。
 *
 * @section PageTargetToolRepairDeletePctl_SectionDetail 解説
 * 特記事項無し
 */

#include <iostream>
#include <iomanip>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <glv.h>
#include <glv_binding.h>
#include <glv_resources.h>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <sstream>
#include <nn/repair/repair_Api.h>
#include <nn/repair/repair_LabelButton.h>
#include <nn/repair/repair_ShutdownButton.h>
#include <nn/repair/repair_StreamTextView.h>
#include <nn/repair/repair_Sdcard.h>
#include <nn/repair/repair_LabelText.h>
#include <nn/repair/repair_NetworkSettingsParser.h>

// ペアレンタルコントロールサービス利用
#include <nn/pctl.h>
#include <nn/pctl/pctl_ApiSystem.h>
#include <nn/pctl/pctl_ApiWatcher.h>
#include <nn/pctl/pctl_ApiForAuthentication.h>

// デバイス情報取得
#include <nn/util/util_BitFlagSet.h>
#include <nn/settings/factory/settings_SerialNumber.h>
#include <nn/settings/factory/settings_DeviceCertificate.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nn/settings/system/settings_FirmwareVersion.h>
#include <nn/time/time_SteadyClockTimePoint.h>
#include <nn/time/time_StandardSteadyClock.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_Api.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#if defined( NN_SDK_BUILD_LIBRARY )
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#else
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#endif

//#define GLV_ENABLE_LOG_DEBUG

//!--------------------------------------------------------------------------------------
//! @brief SDKライブラリビルドでのアサート/ログマクロ制限.
//!--------------------------------------------------------------------------------------
#if defined( NN_SDK_BUILD_LIBRARY )
#define NN_ASSERT NN_SDK_ASSERT
#define GLV_LOG(...) NN_SDK_LOG( "[GLV] " __VA_ARGS__ )
#else
#define GLV_LOG(...) NN_LOG( "[GLV] " __VA_ARGS__ )
#endif // defined( NN_SDK_BUILD_LIBRARY )


namespace {
    const char* ToolName             = "NX Pre-repair check tool";
    const int   ToolMajorVersion     = 4;
    const int   ToolMinorVersion     = 0;

    const char* DispMsgCurrentDay    = "Current day setting\t\t";
    const char* DispMsgInitialDay    = "Assumed initial activation day\t";
    const char* DispMsgElapsedDay    = "Elapsed days\t\t\t";
    const char* DispMsgMigration[]   = {
        "[Attention!] \n",
        "This device is now on 'user migration'. \n",
        "Make sure the condition by the RepairAbortMigration Tool right away.\n",
    };

    const char* DispMsgPCtlEnable    = "Parental controls state\t\tEnabled";
    const char* DispMsgPCtlDisable   = "Parental controls state\t\tDisabled";

    const char* DispMsgButtonExec    = "Execute reset";

    static nn::repair::StreamTextView*    s_pTextView;
    static glv::Label*        s_isPctlValidLabel[2] = {nullptr,nullptr};

    static const size_t AppletHeapSize = 256 * 1024 * 1024;                 //!< アプリケーション予約ヒープメモリサイズ
    static const size_t AppletAllocatableSize = 256 * 1024 * 1024;          //!< アプリケーション稼働ヒープ上限メモリサイズ
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    static const size_t GraphicsSystemReservedMemorySize = 8 * 1024 * 1024; //!< NVNグラフィクス稼働予約メモリ領域サイズ
#endif

#if defined(NN_BUILD_TARGET_PLATFORM_NX)

    void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
    {
        NN_UNUSED(userPtr);
        return aligned_alloc(alignment, size);
    }

    void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
    {
        NN_UNUSED(userPtr);
        free(addr);
    }

    void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
    {
        NN_UNUSED(userPtr);
        return realloc(addr, newSize);
    }
#endif

    //!--------------------------------------------------------------------------------------
    //! @brief GLV用ユーザ定義アロケータ.
    //!--------------------------------------------------------------------------------------
    void* glvMemoryAllocator( const size_t size, const size_t beginAlignment ) NN_NOEXCEPT
    {
#if defined( NN_BUILD_CONFIG_OS_WIN )
        return ::_aligned_malloc( size, beginAlignment );
#else
        return ::aligned_alloc( beginAlignment, size );
#endif
    }

    //!--------------------------------------------------------------------------------------
    //! @brief GLV用ユーザ定義デアロケータ.
    //!--------------------------------------------------------------------------------------
    void glvMemoryDeallocator( void* address ) NN_NOEXCEPT
    {
#if defined( NN_BUILD_CONFIG_OS_WIN )
        ::_aligned_free( address );
#else
        ::free( address );
#endif
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ペリフェラルセットアップ
    //!--------------------------------------------------------------------------------------
    static void SetupPeripherals() NN_NOEXCEPT
    {
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
        // this memory allocation will be used from the nvn graphics systems at runtime.
        nv::SetGraphicsAllocator( NvAllocate, NvFree, NvReallocate, nullptr );
        nv::InitializeGraphics( ::malloc( GraphicsSystemReservedMemorySize ), GraphicsSystemReservedMemorySize );

#if NN_GFX_IS_TARGET_NVN
        // this memory allocation interface will be used when compiling of shader code at runtime.
        glslcSetAllocator( NvAllocate, NvFree, NvReallocate, nullptr );
#endif
#endif
    }

    //!--------------------------------------------------------------------------------------
    //! @brief HID設定初期値
    //!--------------------------------------------------------------------------------------
    static const glv::HidInitialConfiguration LocalHidConfiguration = glv::HidInitialConfiguration( glv::HidInitialConfiguration::PadAssetAssignRule_BasicPadPrimary );

} // namespace

//!--------------------------------------------------------------------------------------
//! nninitStartup() is invoked before calling nnMain().
//! 256MB確保
//!--------------------------------------------------------------------------------------
NN_OS_EXTERN_C void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::SetMemoryHeapSize( AppletHeapSize ) );
    uintptr_t address;
    const size_t MallocMemorySize = AppletAllocatableSize;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::AllocateMemoryBlock( &address, MallocMemorySize ) );
    nn::init::InitializeAllocator( reinterpret_cast< void* >( address ), MallocMemorySize );

    // GLV内部で利用されるルートメモリアロケータを上書きします.
    // 上書きしない場合は C++11 の std::aligned_alloc 及び std::free が利用されます.
    glv::detail::ApplicationMemoryAllocator::AttachUserAllocator( glvMemoryAllocator, glvMemoryDeallocator );
}

void MyMouseDownHandler() NN_NOEXCEPT
{
    s_pTextView->AppendValue("=== Delete Pairing ===\n");

    bool isPctl = nn::pctl::IsRestrictionEnabled();

    if(isPctl)
    {
        // pctl:r 権限必要
        nn::pctl::DeletePairing();
        nn::pctl::DeleteSettings();
        s_pTextView->AppendValue("[[ deleted ]]\n");
    }
    else
    {
        s_pTextView->AppendValue("[[ already none ]]\n");
    }
}

//!--------------------------------------------------------------------------------------
//! @brief ルートサーフェイスコンテキスト
//!--------------------------------------------------------------------------------------
class RootSurfaceContext : public glv::Window, public glv::GLV, public glv::ApplicationLoopCallback
{
private:

public:
    //!--------------------------------------------------------------------------------------
    //! @brief コンストラクタ.
    //! @details 指定の幅、高さの領域に対して、ページセレクトヘッダバーとページサーフェイスを追加します.
    //!--------------------------------------------------------------------------------------
    RootSurfaceContext( const unsigned width, const unsigned height ) NN_NOEXCEPT
         : glv::Window( width, height, "Main Window" ), glv::GLV()
    {
        // this->setGLV( *this );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief デストラクタ.
    //! @attention
    //! DropDownオブジェクトは内部で ListViewを利用していますが、親による自動解放が正しく動かないバグがある模様のため、
    //! 明示的に親( m_pSurfaceTable )が解放される前にデストラクションしています.
    //! m_pSurfaceTableもついでに.
    //! ※尚、GLVのViewは glv::SmartObjectクラスにより動的確保したオブジェクトは、親から外される際に自動的に delete されます.
    //!   明示的解放の場合には、親子関係の解除も含めるようにしてください.
    //!--------------------------------------------------------------------------------------
    virtual ~RootSurfaceContext() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ランタイムエンジンにアタッチされた際に呼ばれます.
    //! @see glv::ApplicationLoopCallback::OnLoopAttached( ApplicationLoopContext& )
    //!--------------------------------------------------------------------------------------
    virtual void OnLoopAttached( glv::ApplicationLoopContext& context ) NN_NOEXCEPT NN_OVERRIDE
    {
        ApplicationLoopCallback::OnLoopAttached( context );
        GLV_LOG( "OnLoopAttached\n" );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ランタイムエンジンからデタッチされた際に呼ばれます.
    //! @see glv::ApplicationLoopCallback::OnLoopDetached( ApplicationLoopContext& )
    //!--------------------------------------------------------------------------------------
    virtual void OnLoopDetached( glv::ApplicationLoopContext& context ) NN_NOEXCEPT NN_OVERRIDE
    {
        GLV_LOG( "OnLoopDetached\n" );
        ApplicationLoopCallback::OnLoopDetached( context );
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ループ中の呼び出し.@n
    //! @details glvシーンレンダラへ hid系イベントが通知される前に呼び出されます.@n
    //! この時点ではまだ glvコンテキストのレンダリングは開始していません.@n
    //! また、このメソッドが呼び出されるフレームは OnLoopAfterSceneRendererと同じです.@n
    //! @return reserved.@n
    //! 現在はRequiredRestoration::RequireRestrationNothing を返却してください.
    //! @see glv::ApplicationLoopCallback::OnLoopBeforeSceneRenderer( ApplicationLoopContext&, const HidEvents& )
    //!--------------------------------------------------------------------------------------
    virtual const glv::RequiredRestoration OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED( events );
        NN_UNUSED( context );
        if ( events.GetAvailableBasicPadCount() > 0 )
        {
            auto& bpad = events.GetBasicPad( 0 );
            if ( bpad.IsButtonDown<glv::BasicPadEventType::Button::R>() || bpad.IsButtonRepeat<glv::BasicPadEventType::Button::R>() )
            {
                GLV_LOG( "OnLoopBeforeSceneRenderer( frame [%zd] )\n", context.GetFrameCount() );
            }
        }
        return glv::RequiredRestoration::RequireRestrationNothing;
    }

    //!--------------------------------------------------------------------------------------
    //! @brief ループ中の呼び出し.@n
    //! @details glvシーンレンダラのレンダリングが終わった後に呼び出されます.@n
    //! また、このメソッドが呼び出されるフレームは OnLoopBeforeSceneRendererと同じです.@n
    //! @return reserved.@n
    //! 現在はRequiredRestoration::RequireRestrationNothing を返却してください.
    //! @see glv::ApplicationLoopCallback::OnLoopAfterSceneRenderer( ApplicationLoopContext&, const HidEvents& )
    //!--------------------------------------------------------------------------------------
    virtual const glv::RequiredRestoration OnLoopAfterSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED( events );
        NN_UNUSED( context );
        const glv::DebugPadEventType& dpad = events.GetDebugPad();
        if ( true == dpad.HasAnyEvent() )
        {
            // SELECT + START 同時押し( 後押し許可版 )
            const glv::DebugPadEventType::ButtonSetType exit( nn::hid::DebugPadButton::Start::Mask | nn::hid::DebugPadButton::Select::Mask );
            if ( exit == ( dpad.GetButtons() & exit )  )
            {
                // アプリケーションループ退場要求
                glv::ApplicationFrameworkExit();
            }
            if ( dpad.IsButtonDown<nn::hid::DebugPadButton::L>() )
            {
                GLV_LOG( "OnLoopAfterSceneRenderer( frame [%zd] )\n", context.GetFrameCount() );
            }
        }
        else if ( events.GetAvailableBasicPadCount() > 0 )
        {
            const glv::BasicPadEventType& bpad = events.GetBasicPad( 0 );
            // SELECT + START 同時押し( 後押し許可版 )
            const glv::BasicPadEventType::ButtonSetType exit( glv::BasicPadEventType::Button::Start::Mask | glv::BasicPadEventType::Button::Select::Mask );
            if ( exit == ( bpad.GetButtons() & exit )  )
            {
                // アプリケーションループ退場要求
                glv::ApplicationFrameworkExit();
            }

            if ( bpad.IsButtonDown<glv::BasicPadEventType::Button::L>() )
            {
                GLV_LOG( "OnLoopAfterSceneRenderer( frame [%zd] )\n", context.GetFrameCount() );
            }

            /*
            // X : ペアコンセット（デバッグ専用）
            if ( bpad.IsButtonDown<glv::BasicPadEventType::Button::X>() )
            {
                nn::pctl::SetPinCode("0000");
                s_pTextView->AppendValue("[[ parental settings generated ]]\n");
            }
            */

        }

        if ( s_isPctlValidLabel[0] != nullptr )
        {
            bool isPctl = nn::pctl::IsRestrictionEnabled();

            if(isPctl)
            {
                s_isPctlValidLabel[0] -> enable(glv::Property::Visible);
                s_isPctlValidLabel[1] -> disable(glv::Property::Visible);
            }
            else
            {
                s_isPctlValidLabel[0] -> disable(glv::Property::Visible);
                s_isPctlValidLabel[1] -> enable(glv::Property::Visible);
            }
        }

        return glv::RequiredRestoration::RequireRestrationNothing;
    }

private:
};

void ShowToolInformation(RootSurfaceContext* context)
{
    // タイトルラベル表示
    auto toolInformationLabel = new glv::Label;
    nn::repair::GetToolInformationLabel(toolInformationLabel, ToolName, ToolMajorVersion, ToolMinorVersion);
    *context << toolInformationLabel;
}

void ShowFirmwareVersion(nn::repair::StreamTextView* view)
{
    glv::Label firmwareVersionLabel;
    nn::repair::GetFirmwareVersionLabel(&firmwareVersionLabel);
    view->AppendValue(firmwareVersionLabel.getValue() + "\n");
}

void ShowSerialNumber(nn::repair::StreamTextView* view)
{
    glv::Label serialNumberLabel;
    nn::repair::GetSerialNumberLabel(&serialNumberLabel);
    view->AppendValue(serialNumberLabel.getValue() + "\n");
}

void ShowInitialLaunchTime(nn::repair::StreamTextView* view)
{
    ::nn::Result result;
    std::ostringstream oss;
    std::string txt;

    nn::time::CalendarTime calendarTime;
    nn::time::CalendarAdditionalInfo calendarAdditionalInfo;

    // time:u 権限必要
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Initialize());

    // 現在のposixTime(協定世界時 (UTC) の 西暦1970年1月1日午前0時0分0秒 からの経過秒数) を取得
    nn::time::PosixTime posixTime;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime));

    result = nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime);
    if (result.IsSuccess()) {
        view->AppendValue(DispMsgCurrentDay);

        NN_LOG("Current calendar(%s): %04d/%02d/%02d %02d:%02d:%02d\n",
            calendarAdditionalInfo.timeZone.standardTimeName,
            calendarTime.year, calendarTime.month, calendarTime.day, calendarTime.hour, calendarTime.minute, calendarTime.second);
        oss.str("");
        oss << std::setw(4) << calendarTime.year << "/";
        oss << std::setw(2) << std::setfill('0') << (int) calendarTime.month << "/";
        oss << std::setw(2) << std::setfill('0') << (int) calendarTime.day;
        oss << "(" << calendarAdditionalInfo.timeZone.standardTimeName << ")";
        oss << std::endl;
        txt = oss.str();
        view->AppendValue(txt);
    }

    // 初回起動時のRTC取得(set:sys 権限必要)
    view->AppendValue(DispMsgInitialDay);
    nn::settings::system::InitialLaunchSettings ils;
    nn::settings::system::GetInitialLaunchSettings(&ils);

    // 初回起動日時がセットされている？
    if( ils.flags.Test<nn::settings::system::InitialLaunchFlag::HasTimeStamp>() )
    {
        // 現在のRTC
        nn::time::SteadyClockTimePoint sctp;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::StandardSteadyClock::GetCurrentTimePoint(&sctp));

        // 経過秒を得る
        int64_t elapsedSeconds;
        result = nn::time::GetSpanBetween(&elapsedSeconds, ils.timeStamp, sctp);

        if(result.IsSuccess())
        {
            // 初回起動時の posix time
            posixTime.value -= elapsedSeconds;

            // PosixTime を CalendarTime へ変換
            // 計算に利用されるタイムゾーンはデバイスに設定されたものを利用。
            result = nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime);

            NN_LOG("Initial calendar(%s): %04d/%02d/%02d %02d:%02d:%02d\n",
                   calendarAdditionalInfo.timeZone.standardTimeName,
                   calendarTime.year, calendarTime.month, calendarTime.day, calendarTime.hour, calendarTime.minute, calendarTime.second);

            oss.str("");
            oss << std::setw(4) << calendarTime.year << "/";
            oss << std::setw(2) << std::setfill('0') << (int)calendarTime.month << "/";
            oss << std::setw(2) << std::setfill('0') << (int)calendarTime.day;
            oss << "(" << calendarAdditionalInfo.timeZone.standardTimeName << ")";
            oss << std::endl;

            // 経過日数を表示
            oss << DispMsgElapsedDay;
            oss << std::right;
            oss << elapsedSeconds / 60 / 60 / 24 << std::endl;

            txt = oss.str();
            view->AppendValue(txt);
            view->AppendValue("\n");
        }
        else
        {
            // debug 情報
            {
                // RTC リセットが発生すると SteadyClockTimePoint 内の UUID の値が変化する
                char fromString[64] = {};
                char toString[64] = {};
                ils.timeStamp.sourceId.ToString(fromString,64);
                sctp.sourceId.ToString(toString,64);
                NN_LOG("from : %s\n" , fromString );
                NN_LOG("to   : %s\n" , toString );
            }

            // 二つの RTC 間でリセットが発生した？
            view->AppendValue("Missed."); // RTC has been reseted

            // エラー情報の表示
            oss.str("");
            oss << "(";
            oss << std::hex << std::setw(8) << std::setfill('0') << (int)result.GetInnerValueForDebug()  << ")";
            oss << std::endl;
            txt = oss.str();
            view->AppendValue(txt);
        }
    }
    else
    {
        // 初回起動日時が未セットである
        view->AppendValue("Not Set.\n");
    }
}

void ShowDeviceId(nn::repair::StreamTextView* view)
{
    ::nn::Result result;

    // device id の定義 ： デバイス名 = prefix(2byte) + device id(16 byte 16進数文字表記) + "-" + ["1" or "0" (0がPROD)]
    glv::Label deviceIdLabel;
    nn::repair::GetDeviceIdLabel(&deviceIdLabel);

    view->AppendValue(deviceIdLabel.getValue() + "\n");
}

//!--------------------------------------------------------------------------------------
//! @brief メイン
//!--------------------------------------------------------------------------------------

void MyMain() NN_NOEXCEPT
{
    const int width  = glv::glutGet(GLUT_SCREEN_WIDTH);
    const int height = glv::glutGet(GLUT_SCREEN_HEIGHT);

    RootSurfaceContext* context = new RootSurfaceContext( width, height );

    // タイトルラベル表示
    ShowToolInformation(context);

    // ログビュー表示
    s_pTextView = new nn::repair::StreamTextView(glv::Rect(1024, 440), 28.f);
    s_pTextView->pos(120, 164);
    *context << s_pTextView;

    bool isMig = false;
    // ユーザ移行中断状態 の判定 (set:sys 権限必要)
    if (nn::repair::IsAvailableMigration())
    {
        isMig = true;
        nn::settings::system::AppletLaunchFlagSet flags;
        nn::settings::system::GetAppletLaunchFlags(&flags);
        isMig = flags.Test<nn::settings::system::AppletLaunchFlag::Migration>();

        if(isMig)
        {
            s_pTextView->Clear();
            const int num = sizeof(DispMsgMigration) / sizeof(DispMsgMigration[0]);
            for(int i=0; i<num; i++ )
            {
                s_pTextView->AppendValue(DispMsgMigration[i]);
            }
        }
    }

    if(!isMig)
    {
        // serial number (set:cal 権限必要)
        ShowSerialNumber(s_pTextView);

        // device id を得る
        ShowDeviceId(s_pTextView);

        // firmware version
        ShowFirmwareVersion(s_pTextView);

        // 初回起動日時
        ShowInitialLaunchTime(s_pTextView);

        // 有効ラベル表示
        s_isPctlValidLabel[0] = new glv::Label(DispMsgPCtlEnable, false);
        s_isPctlValidLabel[0]->pos(120, 100);
        s_isPctlValidLabel[0]->size(28.f);
        *context << s_isPctlValidLabel[0];
        s_isPctlValidLabel[1] = new glv::Label(DispMsgPCtlDisable, false);
        s_isPctlValidLabel[1]->pos(120, 100);
        s_isPctlValidLabel[1]->size(28.f);
        *context << s_isPctlValidLabel[1];

        // 削除／復活ボタン表示
        glv::Button* deleteButton = new nn::repair::LabelButton(DispMsgButtonExec, MyMouseDownHandler);
        deleteButton->pos(400, 640);
        *context << deleteButton;
    }

    // シャットダウンボタン(FW 3.0.0 以降で有効)
    glv::Button* shutdownButton = new nn::repair::ShutdownButton();
    shutdownButton->pos(800,640);
    *context << shutdownButton;

    glv::Style::standard().color.set(glv::StyleColor::WhiteOnBlack);
    glv::Style::standard().color.fore.set(0.5);

    context -> setGLV(*context);

    // メインループコールバックを登録.
    ApplicationFrameworkRegisterLoopCallback( context );

    glv::Application::run();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::time::Finalize());

    if ( nullptr != context )
    {
        delete context;
    }
}

//!--------------------------------------------------------------------------------------
//! @brief プロセスエントリポイント
//!--------------------------------------------------------------------------------------
NN_OS_EXTERN_C void nnMain() NN_NOEXCEPT
{
    SetupPeripherals();

    glv::ApplicationFrameworkInitialize(LocalHidConfiguration);

    MyMain();
}
