﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/srv/nim_HttpConnection.h>
#include <nn/nim/srv/nim_NetworkInstallUrl.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_TFormatString.h>
#include <nn/settings/factory/settings_SerialNumber.h>
#include <nn/repair.h>

#include "RepairAccountTransfer.h"

namespace {
    nn::nim::srv::DeviceContext g_DeviceContext;
}

// 前提として、ネットワークに接続しておく
nn::Result RequestRepairDeviceMigration() NN_NOEXCEPT
{
    nn::ssl::Context sslContext;
    NN_RESULT_DO(sslContext.Create(nn::ssl::Context::SslVersion_Auto));

    nn::ssl::CertStoreId certStoreId;
    NN_RESULT_DO(sslContext.RegisterInternalPki(&certStoreId, nn::ssl::Context::InternalPki_DeviceClientCertDefault));

    nn::nim::srv::HttpConnection httpConnection;
    httpConnection.Initialize(&sslContext);

    NN_UTIL_SCOPE_EXIT
    {
        sslContext.Destroy();
    };

    NN_UTIL_SCOPE_EXIT
    {
        httpConnection.Finalize();
    };

    //migrate
    nn::nim::srv::Url url;
    url.Assign("https://beach.hac.%.eshop.nintendo.net/v1/devices/hac/migrate?lang=en");

    // device_id: 移行先デバイス ID 16 進数文字列 16 文字
    // serial_number: 移行先シリアル番号 14桁
    const char* HeaderList[] = {
        "Accept: application/json",
        "Content-Type: application/x-www-form-urlencoded",
    };

    nn::settings::factory::SerialNumber serialNumber;
    nn::settings::factory::GetSerialNumber(&serialNumber);

    char deviceId[nn::repair::DeviceIdLength + 1];
    std::memset(deviceId, 0, sizeof(deviceId));
    nn::repair::GetDeviceIdHex(deviceId);
    deviceId[nn::repair::DeviceIdLength] = '\0';

    char postData[128];
    nn::util::TSNPrintf(postData, sizeof(postData), "device_id=%s&serial_number=%s", deviceId, serialNumber.string);

    const int ReadBufferSize = 1024;
    char readBuffer[ReadBufferSize];

    size_t readSize = 0;
    auto writeCallback = [&readBuffer, &readSize](const void* buffer, size_t bufferSize) -> nn::Result
    {
        NN_RESULT_THROW_UNLESS(readSize + bufferSize < ReadBufferSize, nn::nim::ResultUnexpectedResponseDeviceAuthenticationTokenTooLong());
        std::memcpy(&readBuffer[readSize], buffer, bufferSize);
        readSize += bufferSize;
        NN_RESULT_SUCCESS;
    };

    const int headerCount = sizeof(HeaderList) / sizeof(HeaderList[0]);

    return httpConnection.Post(url, postData, writeCallback, HeaderList, headerCount);
}
