﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "RepairAbortMigration.h"
#include <nn/settings/system/settings_Language.h>
#include <nn/settings/system/settings_Account.h>

namespace{
    static const char DefaultUserIconFilePath[] = "Contents:/SubstituteUserIcon.jpg";
    NN_ALIGNAS(64) char mountRomCacheBuffer[512];
}

//
// nn::ns::UnregisterNetworkServiceAccount(migration_account_user);
// と実装は同じ。
// 互換性保証の観点から修理での直接利用は不可
//

nn::Result UnregisterNetworkAccountAssociation(nn::account::Uid uid, bool isWithUserDeletion, bool *pIsDeviceLinkLeave) NN_NOEXCEPT
{
    nn::Result result_ec;
    nn::Result result_friend;
    nn::Result result_nsa;
    nn::Result result_npns;

    // このAPIをコールする前に
    // 対象のユーザーアカウントが存在し、ネットワークサービスアカウントが利用可能であることを確認しておくこと
    *pIsDeviceLinkLeave = false;

    // -----------------------------------------------------------------------------------------
    // 0. 機器認証解除 (ec)
    // (online じゃないとできない)
    {
        nn::ec::system::DeviceLinkStatus status;
        {
            nn::ec::system::AsyncDeviceLinkStatus async;
            result_ec = RequestDeviceLinkStatus(&async, uid);
            if( result_ec.IsSuccess() )
            {
                result_ec = async.Get(&status);
                NN_LOG("[ec] RequestDeviceLinkStatus(%s) %08x\n", status.hasDeviceLink ? "true":"false", result_ec.GetInnerValueForDebug());
            }
            else
            {
                NN_LOG("[ec] RequestDeviceLinkStatus Fail %08x\n", result_ec.GetInnerValueForDebug());
            }
        }

        if( result_ec.IsSuccess() && status.hasDeviceLink )
        {
            nn::ec::system::AsyncResult async;
            result_ec = nn::ec::system::RequestUnlinkDevice(&async, uid);
            if( result_ec.IsSuccess() )
            {
                result_ec = async.Get();
            }
            else
            {
                *pIsDeviceLinkLeave = true;
            }
            NN_LOG("[ec] Device unlink ensured %08x\n", result_ec.GetInnerValueForDebug());

            // エラーでも次に進む
        }
    }

    // -----------------------------------------------------------------------------------------
    // 1. NSA が消えた後に残っているとまずいものを消す

    result_friend = DeleteNetworkAccountRelatedInformation(uid);
    if( !isWithUserDeletion && !result_friend.IsSuccess() )
    {
        // UA を消さない場合はここでエラーとする
        return result_friend;
    }

    // -----------------------------------------------------------------------------------------
    // 2. NSA の登録解除処理
    // (online じゃないとできない)
    nn::account::AsyncContext ctx;
    nn::os::SystemEvent e;
    const int retry_max = 10;

    nn::account::NetworkServiceAccountAdministrator admin;

    for(int i=0;i<retry_max;i++) // 10 回
    {
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        result_nsa = nn::account::GetNetworkServiceAccountAdministrator(&admin, uid);
        if(result_nsa.IsSuccess())
        {
            NN_LOG("[account]GetNetworkServiceAccountAdministrator OK\n");

            result_nsa = admin.UnregisterAsync(&ctx);
            if(result_nsa.IsSuccess())
            {
                NN_LOG("[account]UnregisterAsync OK\n");
                NN_ABORT_UNLESS_RESULT_SUCCESS(ctx.GetSystemEvent(&e));
                e.Wait();
                result_nsa = ctx.GetResult();
            }

            // 内部的なアドレス解決の処理（nsd 絡み？）が終わらないとエラーが出る？
            // (TODO) もっときれいに待つ方法は無いか？
            // if( !ResultCurlErrorCouldntResolveHost::Includes(result_nsa) )
            if( result_nsa.IsSuccess() )
            {
                break;
            }
        }
        NN_LOG("[account] NSA Unregister Fail (%08x) \n", result_nsa.GetInnerValueForDebug());
    }

    // -----------------------------------------------------------------------------------------
    // 3. NSA を消した後に、できれば消したいものを消す
    //    - NPNS の通知トークンを削除
    //      (この処理はエラーを無視できる。)
    // (online じゃないとできない)

    // nn::npns::InitializeForSystem 済み
    result_npns = nn::npns::DestroyTokenForBaaS(uid);
    NN_LOG("[npns] destroy npns token (%08x)\n", result_npns.GetInnerValueForDebug() );

    // nsa を消せたかどうか
    return result_nsa;
}

// オフラインでもできるはず
nn::Result DeleteNetworkAccountRelatedInformation(nn::account::Uid uid) NN_NOEXCEPT
{
    nn::Result result_friends;
    nn::Result result_cruiser_shop;
    nn::Result result_cruiser_login;

    // 遊んだ記録の削除 (friends)
    nn::friends::SetOption(nn::friends::OptionAdmin_CheckUserStatus, 0);
    result_friends = nn::friends::DeletePlayHistory(uid);
    NN_LOG("[friends] PlayHistory removed (%08x)\n", result_friends.GetInnerValueForDebug() );

    // Cruiser のセーブデータの削除 (Shop, LoginShare)
    result_cruiser_shop  = nn::ns::DeleteUserSystemSaveData(uid, static_cast<nn::fs::SystemSaveDataId>(0x8000000000001061ull /* Shop */));
    result_cruiser_login = nn::ns::DeleteUserSystemSaveData(uid, static_cast<nn::fs::SystemSaveDataId>(0x8000000000001091ull /* LoginShare */));
    NN_LOG("[Cruiser] savedata removed for Shop and LoginShare (%08x,%08x)\n",
           result_cruiser_shop.GetInnerValueForDebug(), result_cruiser_login.GetInnerValueForDebug());

    if( !result_friends.IsSuccess() ) return result_friends;
    if( !result_cruiser_shop.IsSuccess() ) return result_cruiser_shop;
    if( !result_cruiser_login.IsSuccess() ) return result_cruiser_login;

    NN_RESULT_SUCCESS;
}

nn::Result DeleteUserAccountWithData(nn::account::Uid user) NN_NOEXCEPT
{
    bool isExist;

    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::account::GetUserExistence(&isExist, user) );

    // Eris/Include/nn/ns/
    nn::ns::UserSaveDataStatistics stat;
    nn::ns::ProgressMonitorForDeleteUserSaveDataAll monitor;
    nn::os::SystemEvent e;

    NN_RESULT_DO( nn::ns::CalculateUserSaveDataStatistics(&stat, user) );
    NN_RESULT_DO( nn::ns::DeleteUserSaveDataAll(&monitor, user));

    monitor.GetSystemEvent(&e);
    nn::os::TimerEvent timer(nn::os::EventClearMode_AutoClear);
    timer.StartPeriodic(nn::TimeSpan::FromMilliSeconds(500), nn::TimeSpan::FromMilliSeconds(500));
    while (NN_STATIC_CONDITION(true))
    {
        nn::os::WaitAny(e.GetBase(), timer.GetBase());
        if (e.TryWait())
        {
            break;
        }
        NN_ABORT_UNLESS(timer.TryWait());
        auto progress = monitor.GetStatistics();
        NN_UNUSED(progress);
        NN_LOG(" - %d of %d (%zu of %zu bytes)\n", progress.count, stat.count, progress.sizeInBytes, stat.sizeInBytes);
    }
    NN_ASSERT(e.TryWait());
    NN_ABORT_UNLESS_RESULT_SUCCESS( monitor.GetResult() );

    // UA 削除
    if(isExist)
    {
        NN_RESULT_DO( nn::account::DeleteUser(user) );
    }

    NN_RESULT_SUCCESS;
}

nn::settings::Language GetLanguage()
{
    nn::settings::LanguageCode langCode;
    nn::settings::GetLanguageCode( &langCode );

#define RETURN_IF_MATCH( LANG ) if( langCode == LANG ) { return LANG; }
    RETURN_IF_MATCH( nn::settings::Language_Japanese );
    RETURN_IF_MATCH( nn::settings::Language_AmericanEnglish );
    RETURN_IF_MATCH( nn::settings::Language_French );
    RETURN_IF_MATCH( nn::settings::Language_German );
    RETURN_IF_MATCH( nn::settings::Language_Italian );
    RETURN_IF_MATCH( nn::settings::Language_Spanish );
    RETURN_IF_MATCH( nn::settings::Language_Chinese );
    RETURN_IF_MATCH( nn::settings::Language_Korean );
    RETURN_IF_MATCH( nn::settings::Language_Dutch );
    RETURN_IF_MATCH( nn::settings::Language_Portuguese );
    RETURN_IF_MATCH( nn::settings::Language_Russian );
    RETURN_IF_MATCH( nn::settings::Language_Taiwanese );
    RETURN_IF_MATCH( nn::settings::Language_BritishEnglish );
    RETURN_IF_MATCH( nn::settings::Language_CanadianFrench );
    RETURN_IF_MATCH( nn::settings::Language_LatinAmericanSpanish );
    RETURN_IF_MATCH( nn::settings::Language_SimplifiedChinese );
    RETURN_IF_MATCH( nn::settings::Language_TraditionalChinese );
#undef RETURN_IF_MATCH

    return nn::settings::Language_AmericanEnglish;
}

// 固定のユーザー名。nn::settings::Language と並び順を一致させています。
static const nn::account::Nickname DefaultUserName[] = {
    { "ユーザー" },         // 日本語 (ja)
    { "User" },            // 米国英語 (en-US)
    { "Profil 1" },        // フランス語 (fr)
    { "Nutzer" },          // ドイツ語 (de)
    { "Utente" },          // イタリア語 (it)
    { "Invitado" },        // スペイン語 (es)
    { "User" },            // 中国語 (zh-CN)
    { "User" },            // 韓国語 (ko)
    { "Gebruiker" },       // オランダ語 (nl)
    { "Utilizador" },      // ポルトガル語 (pt)
    { "Временный" },       // ロシア語 (ru)
    { "User" },            // 台湾語 (zh-TW)
    { "User" },            // 英国英語 (en-GB)
    { "Utilisateur" },     // カナダフランス語 (fr-CA)
    { "Invitado" },        // 中南米スペイン語 (es-419)
    { "User" },            // 簡体字中国語 (zh-Hans)
    { "User" },            // 繁体字中国語 (zh-Hant)
};


// 参考：http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=107320203

nn::Result MountRom() NN_NOEXCEPT
{
    nn::Result result;
    size_t mountRomCacheBufferSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::fs::QueryMountRomCacheSize( &mountRomCacheBufferSize ) );
    NN_ABORT_UNLESS(sizeof(mountRomCacheBuffer) > mountRomCacheBufferSize , "too big cache size");
    {
        // abort 禁止
        nn::fs::ScopedAutoAbortDisabler scopedAbortDisabler;

        result = nn::fs::MountRom("Contents", mountRomCacheBuffer, mountRomCacheBufferSize);
        NN_LOG("MountRom (%08x) cachebuffersize:0x%x \n", result.GetInnerValueForDebug(), mountRomCacheBufferSize);

        if( result.IsSuccess() )
        {
            // 先にファイル存在のチェックしておく
            // （リソースデータファイル ROM取り込みの失敗)
            nn::fs::FileHandle file;
            result = nn::fs::OpenFile(&file, DefaultUserIconFilePath, nn::fs::OpenMode_Read);
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(file);
            };
        }
    }

    NN_LOG("[fs] MountRom() = %08x\n", result.GetInnerValueForDebug());

    return result;
}

// 事前に MountRom が必要
nn::Result GenerateDefaultUser() NN_NOEXCEPT
{
    static char icon_buf[128 * 1024];
    nn::fs::FileHandle file;

    NN_LOG("[fs] Read %s from Rom \n", DefaultUserIconFilePath);
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::fs::OpenFile(&file, DefaultUserIconFilePath, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(file);
    };
    int64_t size;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&size, file));
    NN_SDK_ASSERT(static_cast<size_t>(size) <= sizeof(icon_buf));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(file, 0, icon_buf, size));

    nn::account::Uid u;
    nn::account::ProfileEditor editor;

    const auto lang = GetLanguage();

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::BeginUserRegistration(&u));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::GetProfileEditor(&editor, u));

    editor.SetNickname(DefaultUserName[lang]);
    // editor.SetUserData(reinterpret_cast<const char*>(DefaultUseData), sizeof(DefaultUseData));

    NN_ABORT_UNLESS_RESULT_SUCCESS(editor.FlushWithImage(icon_buf, size));
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::account::CompleteUserRegistrationForcibly(u));

    // 新規UAを作ったのでアカウント設定の選択省略フラグをfalseにする
    {
        nn::settings::system::AccountSettings setting;
        nn::settings::system::GetAccountSettings( &setting );
        setting.userSelectorSettings.flags.Set( nn::settings::system::UserSelectorFlag::SkipsIfSingleUser::Index, false );
        nn::settings::system::SetAccountSettings( setting );
    }

    NN_LOG("[account] Generate default user \n");
    NN_RESULT_SUCCESS;
}

