﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <mutex>
#include <new>
#include <vector>

#include <nn/nn_Macro.h>
#include <nn/vi.h>
#include <nn/mem.h>
#include <nn/nn_Assert.h>
#include <nn/os/os_Mutex.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#include "NoftWriter_GraphicSystem.h"
#include "NoftWriter_Renderer.h"

namespace
{

const size_t ApplicationHeapSize = 128 * 1024 * 1024;

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
const size_t GraphicsMemorySize = 8 * 1024 * 1024;
#endif

const int FrameRate = 60;

nn::os::Mutex g_RendererMutex(false);

char g_RendererStorage[sizeof(noftwriter::graphics::Renderer)];
noftwriter::graphics::Renderer* g_pRendererInstance = nullptr;

}  // anonymous

namespace noftwriter { namespace graphics {

const Color Colors::Transparent = { {   0,   0,   0,   0 } };
const Color Colors::Black       = { {   0,   0,   0, 255 } };
const Color Colors::White       = { { 255, 255, 255, 255 } };
const Color Colors::Silver      = { { 128, 128, 128, 255 } };
const Color Colors::Gray        = { {  64,  64,  64, 255 } };
const Color Colors::DimGray     = { {   6,   6,   6, 255 } };
const Color Colors::Red         = { { 255,   0,   0, 255 } };
const Color Colors::Orange      = { { 255, 128,   0, 255 } };
const Color Colors::Gold        = { { 255, 215,   0, 255 } };
const Color Colors::Yellow      = { { 255, 255,   0, 255 } };
const Color Colors::Green       = { {   0, 255,   0, 255 } };
const Color Colors::Turquoise   = { {  64, 224, 208, 255 } };
const Color Colors::Cyan        = { {   0, 255, 255, 255 } };
const Color Colors::Blue        = { {   0,   0, 255, 255 } };
const Color Colors::DodgerBlue  = { {  20,  96, 255, 255 } };
const Color Colors::RoyalBlue   = { {  43,  70, 225, 255 } };
const Color Colors::Magenta     = { { 255,   0, 255, 255 } };
const Color Colors::Crimson     = { { 220,  20,  60, 255 } };

Renderer* Renderer::GetInstance() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_RendererMutex)> lock(g_RendererMutex);

    if (g_pRendererInstance == nullptr)
    {
        g_pRendererInstance = new(g_RendererStorage) Renderer();
        g_pRendererInstance->Initialize();
    }

    return g_pRendererInstance;
}

void Renderer::Shutdown() NN_NOEXCEPT
{
    std::lock_guard<decltype(g_RendererMutex)> lock(g_RendererMutex);

    if (g_pRendererInstance != nullptr)
    {
        g_pRendererInstance->Finalize();
        g_pRendererInstance->~Renderer();
        g_pRendererInstance = nullptr;
    }
}

void Renderer::Initialize() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, nullptr);
    nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, nullptr);
    nv::InitializeGraphics(std::malloc(GraphicsMemorySize), GraphicsMemorySize);
#endif

    m_ApplicationHeap.Initialize(ApplicationHeapSize);

    m_pGraphicsSystem = new(m_GraphicsSystemStorage) noftwriter::graphics::GraphicsSystem();
    m_pGraphicsSystem->SetApplicationHeap(&m_ApplicationHeap);
    m_pGraphicsSystem->Initialize();

    m_pTextWriter = &m_pGraphicsSystem->GetDebugFont();
}

void Renderer::Finalize() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    m_pGraphicsSystem->Finalize();
    m_pGraphicsSystem->~GraphicsSystem();

    m_ApplicationHeap.Finalize();
}

void Renderer::Render() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    m_pGraphicsSystem->BeginDraw();
    m_pTextWriter->Draw(&m_pGraphicsSystem->GetCommandBuffer());

    m_pGraphicsSystem->EndDraw();

    m_pGraphicsSystem->Synchronize(
        nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / FrameRate));
}

void Renderer::SetTextScale(float h, float v) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    m_pTextWriter->SetScale(h, v);
    m_FontContext.scaleHorizontal = h;
    m_FontContext.scaleVertical   = v;
}

void Renderer::GetTextScale(Point2D* pOutScale) const NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutScale);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    pOutScale->x = m_FontContext.scaleHorizontal;
    pOutScale->y = m_FontContext.scaleVertical;
}

void Renderer::SetTextColor(const Color& color) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    m_pTextWriter->SetTextColor(color);
    m_FontContext.color = color;
}

void Renderer::GetTextColor(Color* pOutColor) const NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutColor);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    *pOutColor = m_pTextWriter->GetTextColor();
}

void Renderer::DrawText(float x, float y, const char* format, ...) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(format);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    char printBuffer[512];
    va_list list;

    va_start(list, format);
    vsnprintf(printBuffer, sizeof(printBuffer), format, list);
    va_end(list);

    m_pTextWriter->SetCursor(x, y);
    m_pTextWriter->Print("%s", printBuffer);
}

void Renderer::DrawHorizontalLine(float x, float y, float length, const Color& color, float weight) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    m_pTextWriter->SetTextColor(color);

    nn::util::Float2 prevScale;
    GetTextScale(&prevScale);

    const float LineWidth = 16.0f;
    float dx = x;
    float dy = y - 11.0f * weight;

    SetTextScale(1.0f, weight);
    int count = static_cast<int>(length / LineWidth);
    for (int i = 0; i < count; i++)
    {
        DrawText(dx + LineWidth * i, dy, "─");
    }

    // 残り部分
    float remain = length - LineWidth * count;
    if (remain > 0.0f)
    {
        SetTextScale(remain / LineWidth, weight);
        DrawText(dx + LineWidth * count, dy, "─");
    }

    SetTextScale(prevScale.x, prevScale.y);

    // 色を元に戻す
    m_pTextWriter->SetTextColor(m_FontContext.color);
}

void Renderer::DrawVerticalLine(float x, float y, float length, const Color& color, float weight) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    m_pTextWriter->SetTextColor(color);

    nn::util::Float2 prevScale;
    GetTextScale(&prevScale);

    const float LineHeight = 16.0f;
    float dx = x - 7.0f * weight;
    float dy = y - 3.0f;
    float drawLength = length + 1.0f;

    SetTextScale(weight, 1.0f);
    int count = static_cast<int>(drawLength / LineHeight);
    for (int i = 0; i < count; i++)
    {
        DrawText(dx, dy + LineHeight * i, "│");
    }

    // 残り部分
    float remain = drawLength - LineHeight * count;
    if (remain > 0.0f)
    {
        SetTextScale(weight, remain / LineHeight);
        DrawText(dx, dy + LineHeight * count, "│");
    }

    SetTextScale(prevScale.x, prevScale.y);

    // 色を元に戻す
    m_pTextWriter->SetTextColor(m_FontContext.color);
}

void Renderer::DrawRect(float x, float y, float width, float height, const Color& color, float lineWeight) NN_NOEXCEPT
{
    DrawHorizontalLine(x, y, width, color, lineWeight);
    DrawHorizontalLine(x, y + height - 2, width, color, lineWeight);
    DrawVerticalLine(x, y, height, color, lineWeight);
    DrawVerticalLine(x + width - 1, y, height, color, lineWeight);
}

void Renderer::FillRect(float x, float y, float width, float height, const Color& color) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    m_pTextWriter->SetTextColor(color);

    nn::util::Float2 prevScale;
    GetTextScale(&prevScale);

    auto DrawSingleLine = [this](float dx, float dy, float length, float scaleH)
    {
        const float LineWidth = 13.5f;

        SetTextScale(1.0f, scaleH);

        int count = static_cast<int>(length / LineWidth);
        for (int i = 0; i < count; i++)
        {
            DrawText(dx + LineWidth * i, dy, "■");
        }

        // 残り部分
        float remain = length - LineWidth * count;
        if (remain > 0.0f)
        {
            SetTextScale(remain / LineWidth, scaleH);
            DrawText(dx + LineWidth * count, dy, "■");
        }
    };

    auto DrawMain = [&](float dx, float dy)
    {
        const float LineHeight = 13.5f;
        int count = static_cast<int>(height / LineHeight);
        for (int i = 0; i < count; i++)
        {
            DrawSingleLine(dx, dy + LineHeight * i, width, 1.0f);
        }

        // 残り部分
        float remain = height - LineHeight * count;
        if (remain > 0.0f)
        {
            float scale  = remain / LineHeight;
            float offset = 0.0f * scale;
            DrawSingleLine(dx, dy + LineHeight * count + offset, width, scale);
        }
    };

    DrawMain(x, y - 6.0f);
    SetTextScale(prevScale.x, prevScale.y);

    // 色を元に戻す
    m_pTextWriter->SetTextColor(m_FontContext.color);
}

}}  // noftwriter::graphics
