﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstring>
#include <mutex>
#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Result.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_Mutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Thread.h>
#include <nn/hid/hid_Npad.h>
#include <nn/nfp.h>
#include <nn/nfp/nfp_DebugTypes.h>

#include "npad/NoftWriter_NpadController.h"
#include "NoftWriter_Types.h"
#include "NoftWriter_Util.h"

namespace noftwriter {

/**
 * @brief   NFP API の処理結果です。
 */
enum class NfpResult
{
    Success,            //!< 成功
    Failed,             //!< 失敗 (汎用)
    DeviceLost,         //!< デバイス喪失
    TagLost,            //!< タグ喪失
    UnsupportedTag,     //!< 非対応タグ
    NotForeground,      //!< BG にいる
};

/**
 * @brief   検出した NFC に関する情報です。
 */
struct NfpInfo
{
    bool                    isValidTag;         //!< 有効なタグか

    nn::nfp::TagInfo        tagInfo;            //!< タグ情報

    /**
     * @brief   保持している情報をクリアします。
     */
    void Clear() NN_NOEXCEPT
    {
        isValidTag        = false;
        std::memset(&tagInfo,   0, sizeof(tagInfo));
    }

    /**
     * @brief   有効なタグかどうかを返します。
     */
    bool IsValidTag() const NN_NOEXCEPT
    {
        return isValidTag;
    }
};

// コールバック型
typedef void(*NfpTagDetectCallbackType)(void*);
typedef void(*NfpTagLostCallbackType)(void*);
typedef void(*NfpCancelCallbackType)(void*);
typedef void(*NfpResultCallbackType)(NfpResult, void*);

/**
 * @brief   NFP に関する処理を行うクラスです。
 */
class NfpProcessor final
{
    NN_DISALLOW_COPY(NfpProcessor);
    NN_DISALLOW_MOVE(NfpProcessor);

public:
    NfpProcessor() NN_NOEXCEPT :
        m_Mutex(true),
        m_IsRunning(false),
        m_IsActivated(false),
        m_Thread(),
        m_pThreadStack(nullptr),
        m_DetectEvent(),
        m_LostEvent(),
        m_SuspendEvent(),
        m_ResumeEvent(),
        m_DeviceHandle(),
        m_NfpInfo(),
        m_NextCommand(Command::None),
        m_DetectCallback(nullptr),
        m_pDetectCallbackArgument(nullptr),
        m_LostCallback(nullptr),
        m_pLostCallbackArgument(nullptr),
        m_CancelCallback(nullptr),
        m_pCancelCallbackArgument(nullptr),
        m_FinishCallback(nullptr),
        m_pFinishCallbackArgument(nullptr)
    {}

    ~NfpProcessor() NN_NOEXCEPT
    {
        Shutdown();
    }

    void Setup(char* pThreadStack, size_t stackSize) NN_NOEXCEPT;
    void Shutdown() NN_NOEXCEPT;

    bool Activate(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT;
    void Deactivate() NN_NOEXCEPT;

    void Suspend() NN_NOEXCEPT;
    void Resume() NN_NOEXCEPT;

    void UpdateInput(const npad::INpadController& controller) NN_NOEXCEPT;

    void SetNfpInfo(const NfpInfo& info) NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        m_NfpInfo = info;
    }

    void GetNfpInfo(NfpInfo* pOutInfo) const NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutInfo);

        std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

        *pOutInfo = m_NfpInfo;
    }

    void SetDetectCallback(NfpTagDetectCallbackType callback, void* pArgument) NN_NOEXCEPT
    {
        m_DetectCallback = callback;
        m_pDetectCallbackArgument = pArgument;
    }

    void SetLostCallback(NfpTagLostCallbackType callback, void* pArgument) NN_NOEXCEPT
    {
        m_LostCallback = callback;
        m_pLostCallbackArgument = pArgument;
    }

    void SetCancelCallback(NfpCancelCallbackType callback, void* pArgument) NN_NOEXCEPT
    {
        m_CancelCallback = callback;
        m_pCancelCallbackArgument = pArgument;
    }

    void SetFinishCallback(NfpResultCallbackType callback, void* pArgument) NN_NOEXCEPT
    {
        m_FinishCallback = callback;
        m_pFinishCallbackArgument = pArgument;
    }

    void ClearCallback() NN_NOEXCEPT
    {
        m_DetectCallback = nullptr;
        m_LostCallback   = nullptr;
        m_CancelCallback = nullptr;
        m_FinishCallback = nullptr;
    }

    void NotifyLostFocus() NN_NOEXCEPT;

    // タグ検出系
    nn::Result StartDetection() NN_NOEXCEPT;
    nn::Result StopDetection() NN_NOEXCEPT;

    // 書き込み系
    nn::Result WriteNtf(Ntf* ntf, nn::nfp::NtfWriteType ntfWriteType) NN_NOEXCEPT;

private:
    enum class Command
    {
        None,
        WriteNtf,
        Suspend
    };

private:
    static void ThreadFunction(void* pArg) NN_NOEXCEPT;

    void ThreadFunctionImpl() NN_NOEXCEPT;
    void CheckEvent() NN_NOEXCEPT;
    bool HandleCommand() NN_NOEXCEPT;
    void ReadNfc() NN_NOEXCEPT;
    nn::Result HandleCommonResult(nn::Result result) NN_NOEXCEPT;
    nn::Result HandleFlushResult(nn::Result result) NN_NOEXCEPT;

private:
    mutable nn::os::Mutex       m_Mutex;

    bool                        m_IsRunning;
    bool                        m_IsActivated;

    nn::os::ThreadType          m_Thread;
    char*                       m_pThreadStack;
    nn::os::SystemEventType     m_DetectEvent;
    nn::os::SystemEventType     m_LostEvent;
    nn::os::EventType           m_SuspendEvent;
    nn::os::EventType           m_ResumeEvent;
    nn::nfp::DeviceHandle       m_DeviceHandle;

    NfpInfo                     m_NfpInfo;
    Command                     m_NextCommand;

    NfpTagDetectCallbackType    m_DetectCallback;
    void*                       m_pDetectCallbackArgument;
    NfpTagLostCallbackType      m_LostCallback;
    void*                       m_pLostCallbackArgument;
    NfpCancelCallbackType       m_CancelCallback;
    void*                       m_pCancelCallbackArgument;
    NfpResultCallbackType       m_FinishCallback;
    void*                       m_pFinishCallbackArgument;

    Ntf*                        m_Ntf;
    nn::nfp::NtfWriteType       m_NtfWriteType;
};

}  // noftwriter
