﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Assert.h>
#include <nn/hid/hid_Npad.h>
#include <nn/nfp/nfp_DebugTypes.h>
#include <nn/util/util_CharacterEncoding.h>

#include "../NfpDebugTool_Util.h"
#include "../npad/NfpDebugTool_NpadController.h"
#include "NfpDebugTool_EditRegisterWindow.h"

namespace nfpdebug { namespace ui {

const int EditRegisterWindow::ItemCountMax = RegisterWindowItem_Count;

void EditRegisterWindow::UpdateInput(const npad::INpadController& controller) NN_NOEXCEPT
{
    if (!IsActive() || !IsVisible())
    {
        return;
    }

    if (controller.IsRepeated(nn::hid::NpadButton::Up::Mask))
    {
        m_CursorIndex = (m_CursorIndex + ItemCountMax - 1) % ItemCountMax;
    }
    else if (controller.IsRepeated(nn::hid::NpadButton::Down::Mask))
    {
        m_CursorIndex = (m_CursorIndex + 1) % ItemCountMax;
    }
    else if (controller.IsTriggered(nn::hid::NpadButton::A::Mask))
    {
        if (m_DecideHandler != nullptr)
        {
            m_DecideHandler(controller.GetNpadId(), 0, m_CursorIndex);
        }
    }

    Window::UpdateInput(controller);
}

void EditRegisterWindow::Render() NN_NOEXCEPT
{
    if (!IsVisible())
    {
        return;
    }

    Window::Render();

    auto& renderer = *GetRenderer();

    nn::util::Float2 prevScale;
    renderer.GetTextScale(&prevScale);

    graphics::Rectangle rect;
    GetClientRectangle(&rect);

    renderer.SetTextScale(1.2f, 1.2f);

    DrawCurrentInfo();

    renderer.SetTextScale(prevScale.x, prevScale.y);
}

void EditRegisterWindow::DrawCurrentInfo() NN_NOEXCEPT
{
    auto& renderer = *GetRenderer();

    graphics::Rectangle rect;
    GetClientRectangle(&rect);

    float dx = rect.x + 12;
    float dy = rect.y + 2;

    DrawCursor(
        dx - 8,
        dy + m_CursorIndex * renderer.GetTextLineHeight(),
        rect.width - 4,
        renderer.GetTextLineHeight() - 8,
        GetCursorFadeColor());

    renderer.SetTextColor(GetDisplayColor(graphics::Colors::White));

    float offsetX = 220;

#ifdef NFPDEBUG_SYSTEM
    const auto& date = m_RegisterInfo.registerDate;
    renderer.DrawText(dx, dy, "Register date");
    renderer.DrawText(dx + offsetX, dy, "%04d/%02d/%02d", date.year, date.month, date.day);
    dy += renderer.GetTextLineHeight();
#endif  // ifdef NFPDEBUG_SYSTEM

    char name[64] = {};
    nn::util::ConvertStringUtf16NativeToUtf8(
        name,
        sizeof(name),
        reinterpret_cast<const uint16_t*>(m_RegisterInfo.nickname));
    name[sizeof(name) - 1] = '\0';
    renderer.DrawText(dx, dy, "Nickname");
    renderer.DrawText(dx + offsetX, dy, "%s", name);
    dy += renderer.GetTextLineHeight();

#ifdef NFPDEBUG_SYSTEM
    renderer.DrawText(dx, dy, "Select Mii >");
#endif  // ifdef NFPDEBUG_SYSTEM
}

}}  // nfpdebug::ui
