﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <cstring>

#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/oe.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nfp/nfp_DebugApi.h>

#include "graphics/NfpDebugTool_Renderer.h"
#include "npad/NfpDebugTool_NpadController.h"
#include "ui/NfpDebugTool_CabinetResultWindow.h"
#include "ui/NfpDebugTool_EditAdminWindow.h"
#include "ui/NfpDebugTool_EditRegisterWindow.h"
#include "ui/NfpDebugTool_Footer.h"
#include "ui/NfpDebugTool_InputNumberWindow.h"
#include "ui/NfpDebugTool_MenuWindow.h"
#include "ui/NfpDebugTool_MessageBox.h"
#include "ui/NfpDebugTool_DeviceListWindow.h"
#include "ui/NfpDebugTool_TagDumpWindow.h"
#include "ui/NfpDebugTool_WaitDialog.h"
#include "NfpDebugTool_NfpProcessor.h"
#include "NfpDebugTool_RegisterInfoPreset.h"
#include "NfpDebugTool_Util.h"
#include "NfpDebugTool_Types.h"

//#define SHOW_UNSUPPORTED_COMMAND

 /**
  * @brief   メニューコマンドの定義
  */
#define NFPDEBUG_DEFINE_MENU_COMMAND(command) \
    void (command)(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT

/**
* @brief   メニュー項目をメニューオブジェクトに設定
*/
#define NFPDEBUG_SET_MENU_ITEMS(menu, items) \
    (menu).SetItems((items), NN_ARRAY_SIZE((items)));

namespace {

// プログラム名
const char ProgramName[] =
#ifdef NFPDEBUG_SYSTEM
    "NFP Debug Tool (System)";
#else
    "NFP Debug Tool";
#endif  // ifdef NFPDEBUG_SYSTEM

// ヒープサイズ
const size_t MemoryHeapSize = 256 * 1024 * 1024;
const size_t MallocHeapSize = 256 * 1024 * 1024;

NFPDEBUG_DEFINE_MENU_COMMAND(CommandEditTag);
NFPDEBUG_DEFINE_MENU_COMMAND(CommandDeleteInfo);
NFPDEBUG_DEFINE_MENU_COMMAND(CommandDeleteInfoAction);
NFPDEBUG_DEFINE_MENU_COMMAND(CommandBreakAmiibo);
NFPDEBUG_DEFINE_MENU_COMMAND(CommandBreakAmiiboAction);
NFPDEBUG_DEFINE_MENU_COMMAND(CommandFormatAmiibo);
NFPDEBUG_DEFINE_MENU_COMMAND(CommandRestoreAmiibo);
NFPDEBUG_DEFINE_MENU_COMMAND(CommandControlBackup);

NFPDEBUG_DEFINE_MENU_COMMAND(CommandDumpTag);
NFPDEBUG_DEFINE_MENU_COMMAND(CommandEditAdminInfo);
NFPDEBUG_DEFINE_MENU_COMMAND(CommandEditRegisterInfo);
NFPDEBUG_DEFINE_MENU_COMMAND(CommandEditApplicationArea);
NFPDEBUG_DEFINE_MENU_COMMAND(CommandFlushCurrentData);
NFPDEBUG_DEFINE_MENU_COMMAND(CommandCallCabinet);

NFPDEBUG_DEFINE_MENU_COMMAND(CommandSetNicknameJpUsEn);
NFPDEBUG_DEFINE_MENU_COMMAND(CommandSetNicknameCh);
NFPDEBUG_DEFINE_MENU_COMMAND(CommandSetNicknameKo);
NFPDEBUG_DEFINE_MENU_COMMAND(CommandSetNicknameTw);
NFPDEBUG_DEFINE_MENU_COMMAND(CommandSetNicknameGuideline);

NFPDEBUG_DEFINE_MENU_COMMAND(CommandSetMii);

NFPDEBUG_DEFINE_MENU_COMMAND(CommandSetApplicationAreaPreset);

NFPDEBUG_DEFINE_MENU_COMMAND(CommandSetCabinetMode);
NFPDEBUG_DEFINE_MENU_COMMAND(CommandSetCabinetFlags);

NFPDEBUG_DEFINE_MENU_COMMAND(CommandInitializeBackup);

// 優先実行アクション
enum class ForceAction
{
    None,           // なし
    CallCabinet     // cabinet 呼び出し
};

// cabinet 呼び出し形式
enum CabinetMode
{
    CabinetMode_NicknameAndOwner,
    CabinetMode_GameDataEraser,
    CabinetMode_Restorer,
    CabinetMode_Formatter
};

// cabinet 呼び出し設定
enum CabinetFlags
{
    CabinetFlags_None                   = 0,        // なし
    CabinetFlags_RequireNickname        = 1 << 0,   // ニックネーム必須
    CabinetFlags_UseCurrentTagInfo      = 1 << 1,   // 検出したタグの TagInfo を使う
    CabinetFlags_UseCurrentRegisterInfo = 1 << 2    // 検出したタグの RegisterInfo を使う
};

const nfpdebug::ui::MenuItem MainMenuItems[] =
{
    { "Edit tag data",          CommandEditTag },
    { "Delete info >",          CommandDeleteInfo },
#ifdef NFPDEBUG_SYSTEM
    { "Break amiibo >",         CommandBreakAmiibo },
    { "Format amiibo",          CommandFormatAmiibo },
    { "Restore amiibo",         CommandRestoreAmiibo },
#else
    { "Break amiibo",           CommandBreakAmiiboAction, nn::nfp::BreakType_Activation },
#endif  // ifdef NFPDEBUG_SYSTEM

    { "Backup data >",          CommandControlBackup },

#ifdef NFPDEBUG_SYSTEM
    { "Call cabinet >",         CommandCallCabinet },
#else
    { "Call amiibo settings >", CommandCallCabinet },
#endif  // ifdef NFPDEBUG_SYSTEM
};

const nfpdebug::ui::MenuItem DeleteInfoMenuItems[] =
{
    { "Delete application area",    CommandDeleteInfoAction, nfpdebug::DeleteInfoType_ApplicationArea },
    { "Delete register info",       CommandDeleteInfoAction, nfpdebug::DeleteInfoType_RegisterInfo },
};

const nfpdebug::ui::MenuItem BreakMenuItems[] =
{
    { "Break activation info",  CommandBreakAmiiboAction, nn::nfp::BreakType_Activation },
    { "Break HMAC",             CommandBreakAmiiboAction, nn::nfp::BreakType_Hmac },
};

const nfpdebug::ui::MenuItem ControlBackupMenuItems[] =
{
#ifdef SHOW_UNSUPPORTED_COMMAND
    { "Fill backup entries",    nullptr },
#endif  // ifdef SHOW_UNSUPPORTED_COMMAND
    { "Initialize backup data", CommandInitializeBackup },
};

const nfpdebug::ui::MenuItem TagMenuItems[] =
{
    { "Display current data",       CommandDumpTag },
    { "Edit admin info >",          CommandEditAdminInfo },
    { "Edit register info >",       CommandEditRegisterInfo },
    { "Set application area >",     CommandEditApplicationArea },
#ifdef SHOW_UNSUPPORTED_COMMAND
    { "Write classical format >",   nullptr },
#endif  // ifdef SHOW_UNSUPPORTED_COMMAND
    { "Flush current data",         CommandFlushCurrentData },
};

const nfpdebug::ui::MenuItem TagMenuItemsForNotAmiibo[] =
{
    { "Display current data",       CommandDumpTag },
};

const nfpdebug::ui::MenuItem SelectNicknameMenuItems[] =
{
    {
        "English",
        CommandSetNicknameJpUsEn,
        reinterpret_cast<uintptr_t>(nfpdebug::SampleNameEn)
    },
    {
        "Japanese",
        CommandSetNicknameJpUsEn,
        reinterpret_cast<uintptr_t>(nfpdebug::SampleNameJp)
    },
    {
        "Chinese",
        CommandSetNicknameCh,
        reinterpret_cast<uintptr_t>(nfpdebug::SampleNameCh)
    },
    {
        "Korean",
        CommandSetNicknameKo,
        reinterpret_cast<uintptr_t>(nfpdebug::SampleNameKo)
    },
    {
        "Taiwan",
        CommandSetNicknameTw,
        reinterpret_cast<uintptr_t>(nfpdebug::SampleNameTw)
    },
    {
        "For guideline check",
        CommandSetNicknameGuideline,
        0
    },
};

const nfpdebug::ui::MenuItem SelectMiiMenuItems[] =
{
    { "Test Mii 1", CommandSetMii,  0 },
    { "Broken Mii", CommandSetMii,  255 },
};

const nfpdebug::ui::MenuItem EditApplicationAreaMenuItems[] =
{
    { "Preset 1 (Sequential)",  CommandSetApplicationAreaPreset,    1 },
    { "Preset 2 (All 0xFF)",    CommandSetApplicationAreaPreset,    2 },
};

const nfpdebug::ui::MenuItem CallCabinetMenuItems[] =
{
#ifdef NFPDEBUG_SYSTEM
    { "Nickname & owner >", CommandSetCabinetMode,  CabinetMode_NicknameAndOwner },
    { "Erase gamedata >",   CommandSetCabinetMode,  CabinetMode_GameDataEraser },
    { "Restore >",          CommandSetCabinetMode,  CabinetMode_Restorer },
#else
    { "Nickname & owner",   CommandSetCabinetMode,  CabinetMode_NicknameAndOwner },
    { "Erase gamedata",     CommandSetCabinetMode,  CabinetMode_GameDataEraser },
    { "Restore",            CommandSetCabinetMode,  CabinetMode_Restorer },
#endif  // ifdef NFPDEBUG_SYSTEM
    { "Format",             CommandSetCabinetMode,  CabinetMode_Formatter },
};

const nfpdebug::ui::MenuItem SetCabinetFlagsMenuItems[] =
{
    { "Use detected tag",   CommandSetCabinetFlags, CabinetFlags_UseCurrentTagInfo },
    { "Not use tag info",   CommandSetCabinetFlags, CabinetFlags_None },
};

const nfpdebug::ui::MenuItem SetCabinetFlagsMenuItemsForNickname[] =
{
    {
        "Use detected tag",
        CommandSetCabinetFlags,
        CabinetFlags_RequireNickname
            | CabinetFlags_UseCurrentTagInfo
            | CabinetFlags_UseCurrentRegisterInfo
    },
    {
        "Use detected tag (without RegisterInfo)",
        CommandSetCabinetFlags,
        CabinetFlags_RequireNickname | CabinetFlags_UseCurrentTagInfo
    },
    {
        "Not use tag info",
        CommandSetCabinetFlags,
        CabinetFlags_RequireNickname
    },
    {
        "Use detected tag (Optional nickname)",
        CommandSetCabinetFlags,
        CabinetFlags_UseCurrentTagInfo | CabinetFlags_UseCurrentRegisterInfo
    },
    {
        "Use detected tag (without RegisterInfo, optional nickname)",
        CommandSetCabinetFlags,
        CabinetFlags_UseCurrentTagInfo
    },
    {
        "Not use tag info (Optional nickname)",
        CommandSetCabinetFlags,
        CabinetFlags_None
    },
};

const char HelpNone[] = "";
const char HelpForMainMenu[] =
    "[↑↓] Select  [A] Decide";
const char HelpForSubMenu[] =
    "[↑↓] Select  [A] Decide  [B] Cancel";
const char HelpForTagDump[] =
    "[←→] Change page  [B] Close";
const char HelpForInputNumber[] =
    "[←→] Move cursor  [↑↓] Change digit  [A] Decide  [B] Cancel";
const char HelpForMessageBox[] =
    "[A][B] Close";
const char HelpForSearchTag[] =
    "[B] Cancel";

nfpdebug::ui::DeviceListWindow      g_DeviceListWindow;
nfpdebug::ui::CabinetResultWindow   g_CabinetResultWindow;
nfpdebug::ui::TagDumpWindow         g_TagDumpWindow;
nfpdebug::ui::EditAdminWindow       g_EditAdminWindow;
nfpdebug::ui::EditRegisterWindow    g_EditRegisterWindow;
nfpdebug::ui::InputNumberWindow     g_InputNumberWindow;
nfpdebug::ui::MessageBox            g_MessageBox;
nfpdebug::ui::WaitDialog            g_WaitDialog;
nfpdebug::ui::Footer                g_Footer;

nfpdebug::ui::MenuWindow  g_MainMenu;
nfpdebug::ui::MenuWindow  g_DeleteInfoMenu;
nfpdebug::ui::MenuWindow  g_BreakMenu;
nfpdebug::ui::MenuWindow  g_ControlBackupMenu;
nfpdebug::ui::MenuWindow  g_TagMenu;
nfpdebug::ui::MenuWindow  g_SelectNicknameMenu;
nfpdebug::ui::MenuWindow  g_SelectMiiMenu;
nfpdebug::ui::MenuWindow  g_EditApplicationAreaMenu;
nfpdebug::ui::MenuWindow  g_CallCabinetMenu;
nfpdebug::ui::MenuWindow  g_SetCabinetFlagsMenu;

nfpdebug::ui::Window* g_pActiveWindow = nullptr;

nfpdebug::ui::Window* g_AllWindows[] =
{
    &g_DeviceListWindow,
    &g_MainMenu,
    &g_DeleteInfoMenu,
    &g_BreakMenu,
    &g_ControlBackupMenu,
    &g_TagMenu,
    &g_EditApplicationAreaMenu,
    &g_CallCabinetMenu,
    &g_SetCabinetFlagsMenu,
    &g_CabinetResultWindow,
    &g_TagDumpWindow,
    &g_EditAdminWindow,
    &g_EditRegisterWindow,
    &g_SelectNicknameMenu,
    &g_SelectMiiMenu,
    &g_InputNumberWindow,
    &g_MessageBox,
    &g_WaitDialog,
};

nfpdebug::NfpProcessor g_Nfp;

NN_OS_ALIGNAS_THREAD_STACK char g_NfpThreadStack[nn::os::ThreadStackAlignment * 4];
NN_OS_ALIGNAS_THREAD_STACK char g_HidThreadStack[nn::os::ThreadStackAlignment * 4];

nn::os::EventType   g_RenderEvent;
nn::os::ThreadType  g_HidThread;

nfpdebug::NfpInfo    g_SavedNfpInfo;

ForceAction  g_ForceAction = ForceAction::None;
CabinetMode  g_CabinetMode;
CabinetFlags g_CabinetFlags;

/**
 * @brief   アプリがフォーカス状態にあるか判定する
 */
bool IsFocused() NN_NOEXCEPT
{
    return nn::oe::GetCurrentFocusState() == nn::oe::FocusState_InFocus;
}

/**
 * @brief   キーヘルプを更新
 */
void UpdateHelp() NN_NOEXCEPT
{
    if (g_pActiveWindow == &g_MainMenu)
    {
        g_Footer.SetHelpText(HelpForMainMenu);
    }
    else if (g_pActiveWindow == &g_TagDumpWindow)
    {
        g_Footer.SetHelpText(HelpForTagDump);
    }
    else if (g_pActiveWindow == &g_InputNumberWindow)
    {
        g_Footer.SetHelpText(HelpForInputNumber);
    }
    else
    {
        g_Footer.SetHelpText(HelpForSubMenu);
    }
}

/**
 * @brief   指定したウィンドウをアクティブ化
 */
void ActivateWindow(nfpdebug::ui::Window* pWindow, bool isVisibleLastWindow) NN_NOEXCEPT
{
    if (g_pActiveWindow != nullptr)
    {
        g_pActiveWindow->Deactivate();
        if (!isVisibleLastWindow)
        {
            g_pActiveWindow->Hide();
        }
    }

    g_pActiveWindow = pWindow;
    g_pActiveWindow->Show();
    g_pActiveWindow->Activate();

    UpdateHelp();
}

/**
 * @brief   指定したウィンドウをサブメニューとしてアクティブ化
 */
void ActivateSubMenu(nfpdebug::ui::Window* pWindow, bool isVisibleLastWindow) NN_NOEXCEPT
{
    if (g_pActiveWindow != nullptr && isVisibleLastWindow)
    {
        auto* pActiveWindow = reinterpret_cast<nfpdebug::ui::SelectableWindow*>(g_pActiveWindow);
        pWindow->SetPosition(
            pActiveWindow->GetX() + 160,
            pActiveWindow->GetSelectedItemY());
    }

    ActivateWindow(pWindow, isVisibleLastWindow);
}

/**
 * @brief   メッセージダイアログを表示
 */
void DisplayMessage(
    const char* message,
    const char* caption,
    nfpdebug::ui::MessageBox::Icon icon,
    nfpdebug::ui::WindowHandlerType cancelHandler) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(message);
    NN_ASSERT_NOT_NULL(caption);
    NN_ASSERT_NOT_NULL(cancelHandler);

    ActivateWindow(&g_MessageBox, false);
    g_WaitDialog.Hide();
    g_MessageBox.SetIcon(icon);
    g_MessageBox.SetCaption(caption);
    g_MessageBox.SetText(message);
    g_MessageBox.SetCancelHandler(cancelHandler);
    g_DeviceListWindow.ClearActiveNpadId();
    g_Footer.SetHelpText(HelpForMessageBox);
}

/**
 * @brief   情報メッセージを表示
 */
void DisplayInformationMessage(
    const char* message,
    nfpdebug::ui::WindowHandlerType cancelHandler) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(message);
    NN_ASSERT_NOT_NULL(cancelHandler);

    DisplayMessage(
        message,
        "Information",
        nfpdebug::ui::MessageBox::Icon::Information,
        cancelHandler);
}

/**
 * @brief   エラーメッセージを表示
 */
void DisplayErrorMessage(
    const char* message,
    nfpdebug::ui::WindowHandlerType cancelHandler) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(message);
    NN_ASSERT_NOT_NULL(cancelHandler);

    DisplayMessage(
        message,
        "Error",
        nfpdebug::ui::MessageBox::Icon::Error,
        cancelHandler);
}

/**
 * @brief   amiibo 以外を検出した場合のメッセージを表示
 */
void DisplayNotAmiiboMessage(nfpdebug::ui::WindowHandlerType cancelHandler) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(cancelHandler);

    DisplayErrorMessage("Detected tag is not an amiibo.", cancelHandler);
}

/**
 * @brief   汎用エラーメッセージを表示
 */
void DisplayCommonError(
    nn::Result result,
    nfpdebug::ui::WindowHandlerType cancelHandler) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(cancelHandler);

    const char* message;
    if (nn::nfp::ResultNfcDeviceNotFound::Includes(result))
    {
        message = "Device not found.";
    }
    else if (nn::nfp::ResultTagNotFound::Includes(result) ||
        nn::nfp::ResultUidMisMatch::Includes(result))
    {
        message = "Target tag is not found.";
    }
    else
    {
        message = "Unexpected error.";
    }

    DisplayErrorMessage(message, cancelHandler);
}

/**
 * @brief   Flush 系の操作に失敗した場合のメッセージを表示
 */
void DisplayFlushFailMessage(
    const char* message,
    nfpdebug::ui::WindowHandlerType cancelHandler) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(message);
    NN_ASSERT_NOT_NULL(cancelHandler);

    DisplayErrorMessage(message, cancelHandler);
}

/**
 * @brief   メインメニューに戻る
 */
void ReturnToMainMenu(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(npadId);
    NN_UNUSED(argument);
    ActivateWindow(&g_MainMenu, false);
}

/**
 * @brief   cabinet モード選択メニューに戻る
 */
void ReturnToCabinetMenu(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(npadId);
    NN_UNUSED(argument);
    ActivateWindow(&g_CallCabinetMenu, false);
}

/**
 * @brief   タグ検出後のメニューに戻る
 */
void ReturnToTagMenu(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(npadId);
    NN_UNUSED(argument);
    ActivateWindow(&g_TagMenu, false);
}

/**
 * @brief   情報削除メニューに戻る
 */
void ReturnToDeleteInfoMenu(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(npadId);
    NN_UNUSED(argument);
    ActivateWindow(&g_DeleteInfoMenu, false);
};

/**
 * @brief   タグ破壊メニューに戻る
 */
void ReturnToBreakMenu(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(npadId);
    NN_UNUSED(argument);
    ActivateWindow(&g_BreakMenu, false);
};

/**
 * @brief   Register info 編集ウィンドウに戻る
 */
void ReturnToRegisterInfoMenu(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(npadId);
    NN_UNUSED(argument);
    ActivateWindow(&g_EditRegisterWindow, false);
}

/**
 * @brief   タグ検出処理の実行
 */
void ExecuteSearchAction(
    const nn::hid::NpadIdType& npadId,
    nfpdebug::NfpTagDetectCallbackType detectCallback,
    nfpdebug::NfpResultCallbackType finishCallback) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(detectCallback);

    auto* pPreviousWindow = g_pActiveWindow;

    auto Cleanup = [&]
    {
        g_Nfp.Deactivate();
        pPreviousWindow->Activate();
    };

    pPreviousWindow->Deactivate();
    g_Nfp.Deactivate();

    if (!g_Nfp.Activate(npadId))
    {
        NN_LOG("FAILED: Activate NFP\n");
        Cleanup();
        return;
    }

    // タグサーチ中断
    auto CancelCallback = [](void* pArg)
    {
        g_Nfp.Deactivate();
        g_Nfp.SetNfpInfo(g_SavedNfpInfo);
        ActivateWindow(reinterpret_cast<nfpdebug::ui::Window*>(pArg), false);
        g_WaitDialog.Hide();
        g_DeviceListWindow.ClearActiveNpadId();
    };

    g_Nfp.ClearCallback();
    g_Nfp.SetDetectCallback(detectCallback, pPreviousWindow);
    g_Nfp.SetCancelCallback(CancelCallback, pPreviousWindow);
    g_Nfp.SetFinishCallback(finishCallback, pPreviousWindow);

    if (g_Nfp.StartDetection().IsFailure())
    {
        NN_LOG("FAILED: StartDetection\n");
        Cleanup();
        return;
    }

    g_DeviceListWindow.SetActiveNpadId(npadId);

    auto dialogCancelHandler = [](const nn::hid::NpadIdType& npadId, uintptr_t argument)
    {
        g_Nfp.StopDetection();
    };

    g_WaitDialog.SetCancelHandler(dialogCancelHandler);
    g_WaitDialog.SetText(
        "  Please touch an amiibo!\n"
        "(Press B button to cancel)");
    ActivateWindow(&g_WaitDialog, true);

    g_Footer.SetHelpText(HelpForSearchTag);
}

/**
 * @brief   コマンド: Edit tag data
 */
void CommandEditTag(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(argument);

    // タグ検出
    auto DetectCallback = [](void* pArg)
    {
        NN_UNUSED(pArg);

        g_Nfp.Deactivate();

        // TODO: NeedRestore 等でメッセージを出す

        nfpdebug::NfpInfo info;
        g_Nfp.GetNfpInfo(&info);
        if (info.IsAmiibo() && info.HasValidRamArea())
        {
            NFPDEBUG_SET_MENU_ITEMS(g_TagMenu, TagMenuItems);
        }
        else
        {
            NFPDEBUG_SET_MENU_ITEMS(g_TagMenu, TagMenuItemsForNotAmiibo);
        }

        // 諸々のカーソル位置を初期化
        g_TagMenu.SetCursorIndex(0);
        g_SelectNicknameMenu.SetCursorIndex(0);
        g_SelectMiiMenu.SetCursorIndex(0);
        g_EditApplicationAreaMenu.SetCursorIndex(0);

        // ウィンドウ位置調整のために一度 MainMenu を経由する
        ActivateWindow(&g_MainMenu, false);
        ActivateSubMenu(&g_TagMenu, true);
        g_DeviceListWindow.ClearActiveNpadId();
    };

    ExecuteSearchAction(npadId, DetectCallback, nullptr);
}

/**
 * @brief   コマンド: Delete info
 */
void CommandDeleteInfo(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(npadId);
    NN_UNUSED(argument);

    ActivateSubMenu(&g_DeleteInfoMenu, true);
}

/**
 * @brief   消去タイプに応じたコマンドの実処理
 */
template <nfpdebug::DeleteInfoType InfoType>
void CommandDeleteInfoImpl(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT
{
    // 処理完了
    auto FinishCallback = [](nfpdebug::NfpResult result, void* pArg)
    {
        g_Nfp.Deactivate();

        if (result == nfpdebug::NfpResult::Success)
        {
            DisplayInformationMessage("Finished!!", ReturnToDeleteInfoMenu);
        }
        else
        {
            DisplayFlushFailMessage("Delete failed...", ReturnToDeleteInfoMenu);
        }
    };

    // タグ検出
    auto DetectCallback = [](void* pArg)
    {
        nfpdebug::NfpInfo info;
        g_Nfp.GetNfpInfo(&info);
        if (info.isAmiibo)
        {
            g_Footer.SetHelpText(HelpNone);
            g_WaitDialog.SetText(
                "  Delete in progress...\n"
                "Keep touching the amiibo!");
            auto result = g_Nfp.DeleteInfo(InfoType);
            if (result.IsFailure())
            {
                DisplayCommonError(result, ReturnToDeleteInfoMenu);
            }
            else
            {
                g_Footer.SetHelpText(HelpNone);
            }
        }
        else
        {
            g_Nfp.Deactivate();
            DisplayNotAmiiboMessage(ReturnToDeleteInfoMenu);
        }
    };

    ExecuteSearchAction(npadId, DetectCallback, FinishCallback);
}

/**
 * @brief   コマンド: amiibo の情報消去
 */
void CommandDeleteInfoAction(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    switch (static_cast<nfpdebug::DeleteInfoType>(argument))
    {
    case nfpdebug::DeleteInfoType_ApplicationArea:
        CommandDeleteInfoImpl<nfpdebug::DeleteInfoType_ApplicationArea>(npadId);
        break;
    case nfpdebug::DeleteInfoType_RegisterInfo:
        CommandDeleteInfoImpl<nfpdebug::DeleteInfoType_RegisterInfo>(npadId);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

/**
 * @brief   コマンド: Break amiibo
 */
void CommandBreakAmiibo(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(npadId);
    NN_UNUSED(argument);

    ActivateSubMenu(&g_BreakMenu, true);
}

/**
 * @brief   BreakType に応じた amiibo 破壊コマンドの実処理
 */
template <nn::nfp::BreakType BreakType>
void CommandBreakAmiiboImpl(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT
{
    // 処理完了
    auto FinishCallback = [](nfpdebug::NfpResult result, void* pArg)
    {
        g_Nfp.Deactivate();

        auto returnFunction =
#ifdef NFPDEBUG_SYSTEM
            ReturnToBreakMenu;
#else
            ReturnToMainMenu;
#endif  // ifdef NFPDEBUG_SYSTEM

        if (result == nfpdebug::NfpResult::Success)
        {
            DisplayInformationMessage("Finished!!", returnFunction);
        }
        else
        {
            DisplayFlushFailMessage("Break failed...", returnFunction);
        }
    };

    // タグ検出
    auto DetectCallback = [](void* pArg)
    {
        nfpdebug::NfpInfo info;
        g_Nfp.GetNfpInfo(&info);
        if (info.isAmiibo)
        {
            g_Footer.SetHelpText(HelpNone);
            g_WaitDialog.SetText(
                "  Break in progress...\n"
                "Keep touching the amiibo!");
            auto result = g_Nfp.BreakTag(BreakType);
            if (result.IsFailure())
            {
                DisplayCommonError(result, ReturnToBreakMenu);
            }
            else
            {
                g_Footer.SetHelpText(HelpNone);
            }
        }
        else
        {
            g_Nfp.Deactivate();
            DisplayNotAmiiboMessage(ReturnToBreakMenu);
        }
    };

    ExecuteSearchAction(npadId, DetectCallback, FinishCallback);
}

/**
 * @brief   コマンド: amiibo の情報破壊
 */
void CommandBreakAmiiboAction(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    switch (static_cast<nn::nfp::BreakType>(argument))
    {
    case nn::nfp::BreakType_Activation:
        CommandBreakAmiiboImpl<nn::nfp::BreakType_Activation>(npadId);
        break;
    case nn::nfp::BreakType_Hmac:
        CommandBreakAmiiboImpl<nn::nfp::BreakType_Hmac>(npadId);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

/**
 * @brief   コマンド: Format
 */
void CommandFormatAmiibo(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(argument);

    // 処理完了
    auto FinishCallback = [](nfpdebug::NfpResult result, void* pArg)
    {
        g_Nfp.Deactivate();

        if (result == nfpdebug::NfpResult::Success)
        {
            DisplayInformationMessage("Finished!!", ReturnToMainMenu);
        }
        else
        {
            DisplayFlushFailMessage("Format failed...", ReturnToMainMenu);
        }
    };

    // タグ検出
    auto DetectCallback = [](void* pArg)
    {
        nfpdebug::NfpInfo info;
        g_Nfp.GetNfpInfo(&info);
        if (info.isAmiibo)
        {
            ActivateWindow(&g_WaitDialog, true);
            g_WaitDialog.SetText(
                "  Format in progress...\n"
                "Keep touching the amiibo!");

            auto result = g_Nfp.Format();
            if (result.IsFailure())
            {
                DisplayCommonError(result, ReturnToMainMenu);
            }
            else
            {
                g_Footer.SetHelpText(HelpNone);
            }
        }
        else
        {
            g_Nfp.Deactivate();
            DisplayNotAmiiboMessage(ReturnToMainMenu);
        }
    };

    ExecuteSearchAction(npadId, DetectCallback, FinishCallback);
}

/**
 * @brief   コマンド: Restore
 */
void CommandRestoreAmiibo(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(argument);

    // 処理完了
    auto FinishCallback = [](nfpdebug::NfpResult result, void* pArg)
    {
        g_Nfp.Deactivate();

        switch (result)
        {
        case nfpdebug::NfpResult::Success:
            DisplayInformationMessage("Finished!!", ReturnToMainMenu);
            break;
        case nfpdebug::NfpResult::NotBroken:
            DisplayInformationMessage("This amiibo is not broken.", ReturnToMainMenu);
            break;
        default:
            DisplayFlushFailMessage("Restore failed...", ReturnToMainMenu);
            break;
        }
    };

    // タグ検出
    auto DetectCallback = [](void* pArg)
    {
        nfpdebug::NfpInfo info;
        g_Nfp.GetNfpInfo(&info);
        if (info.isAmiibo)
        {
            ActivateWindow(&g_WaitDialog, true);
            g_WaitDialog.SetText(
                "  Restore in progress...\n"
                "Keep touching the amiibo!");

            auto result = g_Nfp.Restore();
            if (result.IsFailure())
            {
                DisplayCommonError(result, ReturnToMainMenu);
            }
            else
            {
                g_Footer.SetHelpText(HelpNone);
            }
        }
        else
        {
            g_Nfp.Deactivate();
            DisplayNotAmiiboMessage(ReturnToMainMenu);
        }
    };

    ExecuteSearchAction(npadId, DetectCallback, FinishCallback);
}

/**
 * @brief   コマンド: Control backup data
 */
void CommandControlBackup(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(npadId);
    NN_UNUSED(argument);

    ActivateSubMenu(&g_ControlBackupMenu, true);
}

/**
 * @brief   コマンド: Initialize backup data
 */
void CommandInitializeBackup(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(npadId);
    NN_UNUSED(argument);

    nn::nfp::BackupDataHeader header;
    size_t readSize;
    if (nn::nfp::ReadBackupData(&header, &readSize, sizeof(header)).IsFailure())
    {
        DisplayErrorMessage("Failed to read backup data.", ReturnToMainMenu);
        return;
    }

    NN_ASSERT_EQUAL(readSize, sizeof(header));
    NN_UNUSED(readSize);

    // ヘッダを破壊する
    header.entryNum = 0;

    if (nn::nfp::WriteBackupData(&header, sizeof(header)).IsFailure())
    {
        DisplayErrorMessage("Failed to write backup data.", ReturnToMainMenu);
        return;
    }

    DisplayInformationMessage("Backup data is initialized.", ReturnToMainMenu);
}

/**
 * @brief   コマンド: Dump
 */
void CommandDumpTag(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(npadId);
    NN_UNUSED(argument);

    nfpdebug::NfpInfo info;
    g_Nfp.GetNfpInfo(&info);
    g_TagDumpWindow.SetNfpInfo(info);
    g_TagDumpWindow.InitializeDisplayPage();

    ActivateWindow(&g_TagDumpWindow, true);
}

/**
 * @brief   Admin info の Application ID (NX) を設定
 */
void ApplyAdminInfoApplicationIdNx(const nn::hid::NpadIdType& npadId, uint64_t value) NN_NOEXCEPT
{
    NN_UNUSED(npadId);

    nfpdebug::NfpInfo info;
    g_Nfp.GetNfpInfo(&info);

    auto& adminInfo = info.allData.adminInfo;
    nfpdebug::GetApplicationIdForAmiibo(
        &adminInfo.applicationId,
        &adminInfo.applicationIdExt,
        value);
    g_Nfp.SetAdminInfo(adminInfo);
    g_EditAdminWindow.SetAdminInfo(adminInfo);

    NN_LOG(
        "Application ID    : 0x%08X%08X\n"
        "Application ID ext: 0x%02X\n",
        static_cast<uint32_t>(adminInfo.applicationId.value >> 32),
        static_cast<uint32_t>(adminInfo.applicationId.value & 0xFFFFFFFF),
        adminInfo.applicationIdExt);

    ActivateWindow(&g_EditAdminWindow, false);
}

/**
 * @brief   Admin info の Application ID (Classic) を設定
 */
void ApplyAdminInfoApplicationIdClassic(const nn::hid::NpadIdType& npadId, uint64_t value) NN_NOEXCEPT
{
    NN_UNUSED(npadId);

    nfpdebug::NfpInfo info;
    g_Nfp.GetNfpInfo(&info);

    auto& adminInfo = info.allData.adminInfo;
    adminInfo.applicationId.value = static_cast<nn::Bit64>(value);
    adminInfo.applicationIdExt    = 0;
    g_Nfp.SetAdminInfo(adminInfo);
    g_EditAdminWindow.SetAdminInfo(adminInfo);

    NN_LOG("Title ID: 0x%08X%08X\n",
        static_cast<uint32_t>(value >> 32),
        static_cast<uint32_t>(value & 0xFFFFFFFF));

    ActivateWindow(&g_EditAdminWindow, false);
}

/**
 * @brief   Admin info の Access ID を設定
 */
void ApplyAdminInfoAccessId(const nn::hid::NpadIdType& npadId, uint64_t value) NN_NOEXCEPT
{
    NN_UNUSED(npadId);

    nfpdebug::NfpInfo info;
    g_Nfp.GetNfpInfo(&info);

    auto& adminInfo = info.allData.adminInfo;
    adminInfo.accessId = static_cast<nn::Bit32>(value);
    g_Nfp.SetAdminInfo(adminInfo);
    g_EditAdminWindow.SetAdminInfo(adminInfo);

    NN_LOG("Access ID: 0x%08X\n", static_cast<nn::Bit32>(value));
    ActivateWindow(&g_EditAdminWindow, false);
}

/**
 * @brief   Admin info の編集を開始
 */
void ExecuteEditAdminInfo(int selectedIndex) NN_NOEXCEPT
{
    nfpdebug::NfpInfo info;
    g_Nfp.GetNfpInfo(&info);

    nfpdebug::ui::InputWindowHandlerType decideHandler;
    switch (selectedIndex)
    {
    case nfpdebug::ui::AdminWindowItem_NxApplicationId:
        {
            decideHandler = ApplyAdminInfoApplicationIdNx;
            g_InputNumberWindow.SetNumberRestriction(
                16,
                nfpdebug::ui::NumberBase::Hex);
            g_InputNumberWindow.SetNumber(
                nfpdebug::GetActualApplicationId(
                    info.allData.adminInfo.applicationId,
                    info.allData.adminInfo.applicationIdExt));
            g_InputNumberWindow.SetText("Please input application ID.");
        }
        break;
    case nfpdebug::ui::AdminWindowItem_ClassicApplicationId:
        {
            decideHandler = ApplyAdminInfoApplicationIdClassic;
            g_InputNumberWindow.SetNumberRestriction(
                16,
                nfpdebug::ui::NumberBase::Hex);
            g_InputNumberWindow.SetNumber(
                info.allData.adminInfo.applicationId.value);
            g_InputNumberWindow.SetText("Please input application ID.");
        }
        break;
    case nfpdebug::ui::AdminWindowItem_AccessId:
        {
            decideHandler = ApplyAdminInfoAccessId;
            g_InputNumberWindow.SetNumberRestriction(
                8,
                nfpdebug::ui::NumberBase::Hex);
            g_InputNumberWindow.SetNumber(info.allData.adminInfo.accessId);
            g_InputNumberWindow.SetText("Please input access ID.");
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    auto cancelHandler = [](const nn::hid::NpadIdType& npadId, uintptr_t arg)
    {
        NN_UNUSED(npadId);
        NN_UNUSED(arg);
        ActivateWindow(&g_EditAdminWindow, false);
    };

    g_InputNumberWindow.SetCancelHandler(cancelHandler);
    g_InputNumberWindow.SetDecideHandler(decideHandler);

    ActivateWindow(&g_InputNumberWindow, true);
}

/**
 * @brief   コマンド: Edit admin info
 */
void CommandEditAdminInfo(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(npadId);
    NN_UNUSED(argument);

    nfpdebug::NfpInfo info;
    g_Nfp.GetNfpInfo(&info);
    g_EditAdminWindow.SetAdminInfo(info.allData.adminInfo);
    g_EditAdminWindow.SetCursorIndex(0);

    auto decideHandler = [](const nn::hid::NpadIdType& npadId, uintptr_t arg, int index)
    {
        NN_UNUSED(npadId);
        NN_UNUSED(arg);
        ExecuteEditAdminInfo(index);
    };

    g_EditAdminWindow.SetDecideHandler(decideHandler);

    ActivateSubMenu(&g_EditAdminWindow, true);
}

/**
 * @brief   Register info の register date を設定
 */
void ApplyRegisterInfoRegisterDate(const nn::hid::NpadIdType& npadId, uint64_t value) NN_NOEXCEPT
{
    NN_UNUSED(npadId);

    nfpdebug::NfpInfo info;
    g_Nfp.GetNfpInfo(&info);

    // 入力値から register date を復元
    // (デバッグ用のため、範囲外の値も特に制限しない)
    auto& registerInfo = info.allData.registerInfo;
    auto& date = registerInfo.registerDate;
    date.year  = value / 10000;
    date.month = value / 100 % 100;
    date.day   = value % 100;

    g_Nfp.SetRegisterInfo(registerInfo);
    g_EditRegisterWindow.SetRegisterInfo(registerInfo);

    NN_LOG("Register date: %4d/%02d/%02d\n",
        date.year,
        date.month,
        date.day);

    ActivateWindow(&g_EditRegisterWindow, false);
}

/**
 * @brief   Register info の編集を開始
 */
void ExecuteEditRegisterInfo(int selectedIndex) NN_NOEXCEPT
{
    nfpdebug::NfpInfo info;
    g_Nfp.GetNfpInfo(&info);

    switch (selectedIndex)
    {
    case nfpdebug::ui::RegisterWindowItem_Date:  // Register date
        {
            g_InputNumberWindow.SetNumberRestriction(
                8,
                nfpdebug::ui::NumberBase::Decimal);

            auto& date = info.allData.registerInfo.registerDate;
            g_InputNumberWindow.SetNumber(
                date.year * 10000 + date.month * 100 + date.day);
            g_InputNumberWindow.SetText("Please input register date.");

            auto cancelHandler = [](const nn::hid::NpadIdType& npadId, uintptr_t arg)
            {
                NN_UNUSED(npadId);
                NN_UNUSED(arg);
                ActivateWindow(&g_EditRegisterWindow, false);
            };

            g_InputNumberWindow.SetCancelHandler(cancelHandler);
            g_InputNumberWindow.SetDecideHandler(ApplyRegisterInfoRegisterDate);

            ActivateWindow(&g_InputNumberWindow, true);
        }
        break;
    case nfpdebug::ui::RegisterWindowItem_Nickname:
        ActivateSubMenu(&g_SelectNicknameMenu, true);
        break;
    case nfpdebug::ui::RegisterWindowItem_Mii:
        ActivateSubMenu(&g_SelectMiiMenu, true);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

/**
 * @brief   コマンド: Edit register info
 */
void CommandEditRegisterInfo(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(npadId);
    NN_UNUSED(argument);

    nfpdebug::NfpInfo info;
    g_Nfp.GetNfpInfo(&info);
    g_EditRegisterWindow.SetRegisterInfo(info.allData.registerInfo);
    g_EditRegisterWindow.SetCursorIndex(0);

    auto decideHandler = [](const nn::hid::NpadIdType& npadId, uintptr_t arg, int index)
    {
        NN_UNUSED(arg);
        ExecuteEditRegisterInfo(index);
    };

    g_EditRegisterWindow.SetDecideHandler(decideHandler);

    ActivateSubMenu(&g_EditRegisterWindow, true);
}

/**
 * @brief   指定したニックネームを読み込み済みのタグ情報に反映
 */
void SetNicknameForCurrentTag(const char* nickname, nn::nfp::FontRegion fontRegion) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(nickname);

    nfpdebug::NfpInfo info;
    g_Nfp.GetNfpInfo(&info);

    // ニックネームを UTF-16 に変換
    char16_t utf16Name[nn::nfp::NicknameLengthMax + 1] = {};
    auto result = nn::util::ConvertStringUtf8ToUtf16Native(
        reinterpret_cast<uint16_t*>(utf16Name),
        nn::nfp::NicknameLengthMax,
        nickname,
        strnlen(nickname, nn::nfp::NicknameLengthMax * 4 + 1));
    if (result != nn::util::CharacterEncodingResult_Success)
    {
        NN_LOG("FAIL: Convert to UTF-16 (%d)\n", result);
    }

    // 読み込み済みの register info に反映
    auto& registerInfo = info.allData.registerInfo;
    std::memcpy(registerInfo.nickname, utf16Name, sizeof(registerInfo.nickname));
    registerInfo.fontRegion = static_cast<nn::Bit8>(fontRegion);

    g_Nfp.SetRegisterInfo(registerInfo);
    g_EditRegisterWindow.SetRegisterInfo(registerInfo);
}

/**
 * @brief   コマンド: ニックネーム を設定
 */
void CommandSetNicknameJpUsEn(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    const auto* name = reinterpret_cast<const char*>(argument);
    NN_ASSERT_NOT_NULL(name);

    SetNicknameForCurrentTag(name, nn::nfp::FontRegion_JpUsEu);
    ReturnToRegisterInfoMenu(npadId, 0);
}

/**
 * @brief   コマンド: ニックネーム (中国語) を設定
 */
void CommandSetNicknameCh(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    const auto* name = reinterpret_cast<const char*>(argument);
    NN_ASSERT_NOT_NULL(name);

    SetNicknameForCurrentTag(name, nn::nfp::FontRegion_China);
    ReturnToRegisterInfoMenu(npadId, 0);
}

/**
 * @brief   コマンド: ニックネーム (韓国語) を設定
 */
void CommandSetNicknameKo(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    const auto* name = reinterpret_cast<const char*>(argument);
    NN_ASSERT_NOT_NULL(name);

    SetNicknameForCurrentTag(name, nn::nfp::FontRegion_Korea);
    ReturnToRegisterInfoMenu(npadId, 0);
}

/**
 * @brief   コマンド: ニックネーム (台湾) を設定
 */
void CommandSetNicknameTw(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    const auto* name = reinterpret_cast<const char*>(argument);
    NN_ASSERT_NOT_NULL(name);

    SetNicknameForCurrentTag(name, nn::nfp::FontRegion_Taiwan);
    ReturnToRegisterInfoMenu(npadId, 0);
}

/**
 * @brief   コマンド: ニックネーム (ガイドライン確認用) を設定
 */
void CommandSetNicknameGuideline(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(argument);

    // 読み込み済みの register info に反映
    nfpdebug::NfpInfo info;
    g_Nfp.GetNfpInfo(&info);

    auto& registerInfo = info.allData.registerInfo;
    std::memcpy(
        registerInfo.nickname,
        nfpdebug::NicknameForGuidelineCheck,
        sizeof(registerInfo.nickname));
    registerInfo.fontRegion = static_cast<nn::Bit8>(nn::nfp::FontRegion_JpUsEu);

    g_Nfp.SetRegisterInfo(registerInfo);
    g_EditRegisterWindow.SetRegisterInfo(registerInfo);

    ReturnToRegisterInfoMenu(npadId, 0);
}

/**
 * @brief   コマンド: Mii を設定
 */
void CommandSetMii(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(argument);

    nfpdebug::NfpInfo info;
    g_Nfp.GetNfpInfo(&info);

    auto& registerInfo = info.allData.registerInfo;

    switch (argument)
    {
    case 255:  // 無効な Mii データ
        {
            std::memset(&registerInfo.miiDataCore, 0, sizeof(registerInfo.miiDataCore));
            std::memset(&registerInfo.miiDataExtention, 0, sizeof(registerInfo.miiDataExtention));
            registerInfo.miiExtentionVersion = 0x00;
        }
        break;
    default:
        {
            nn::mii::StoreData miiData;
            nfpdebug::CreateSampleMiiData(&miiData, static_cast<int>(argument));
            NN_ABORT_UNLESS_RESULT_SUCCESS(
                nfpdebug::ConvertStoreDataToNfpMiiData(
                    &registerInfo.miiDataCore,
                    &registerInfo.miiDataExtention,
                    miiData));
            registerInfo.miiExtentionVersion = 0x00;
        }
        break;
    }

    g_Nfp.SetRegisterInfo(registerInfo);

    ReturnToRegisterInfoMenu(npadId, 0);
}

/**
 * @brief   コマンド: Edit application area
 */
void CommandEditApplicationArea(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(npadId);
    NN_UNUSED(argument);

    ActivateSubMenu(&g_EditApplicationAreaMenu, true);
}

/**
 * @brief   コマンド: AppArea preset n
 */
void CommandSetApplicationAreaPreset(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    char applicationArea[nn::nfp::ApplicationAreaSizeV2];

    switch (argument)
    {
    case 1:  // 0x00 からのインクリメント
        for (int i = 0; i < sizeof(applicationArea); i++)
        {
            applicationArea[i] = static_cast<char>(i & 0xFF);
        }
        break;
    case 2:  // All 0xFF
        std::memset(applicationArea, 0xFF, sizeof(applicationArea));
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    g_Nfp.SetApplicationArea(applicationArea, sizeof(applicationArea));

    ReturnToTagMenu(npadId, 0);
}

/**
 * @brief   コマンド: Flush current  data
 */
void CommandFlushCurrentData(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(npadId);
    NN_UNUSED(argument);

    // 処理完了
    auto FinishCallback = [](nfpdebug::NfpResult result, void* pArg)
    {
        g_Nfp.Deactivate();

        if (result == nfpdebug::NfpResult::Success)
        {
            DisplayInformationMessage("Finished!!", ReturnToTagMenu);
        }
        else
        {
            DisplayFlushFailMessage("Flush failed...", ReturnToTagMenu);
        }
    };

    // タグ検出
    auto DetectCallback = [](void* pArg)
    {
        nfpdebug::NfpInfo info;
        g_Nfp.GetNfpInfo(&info);

        // 一致確認
        if (!nfpdebug::IsSameTag(g_SavedNfpInfo.tagInfo.tagId, info.tagInfo.tagId))
        {
            DisplayCommonError(nn::nfp::ResultUidMisMatch(), ReturnToTagMenu);
            return;
        }

        if (info.isAmiibo)
        {
            ActivateWindow(&g_WaitDialog, true);
            g_WaitDialog.SetText(
                "  Flush in progress...\n"
                "Keep touching the amiibo!");

            g_Nfp.SetNfpInfo(g_SavedNfpInfo);
            auto result = g_Nfp.FlushDebug();
            if (result.IsFailure())
            {
                DisplayCommonError(result, ReturnToTagMenu);
            }
            else
            {
                g_Footer.SetHelpText(HelpNone);
            }
        }
        else
        {
            g_Nfp.Deactivate();
            DisplayNotAmiiboMessage(ReturnToTagMenu);
        }
    };

    g_Nfp.GetNfpInfo(&g_SavedNfpInfo);

    ExecuteSearchAction(npadId, DetectCallback, FinishCallback);
}

/**
 * @brief   コマンド: Call cabinet
 */
void CommandCallCabinet(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(npadId);
    NN_UNUSED(argument);

    ActivateSubMenu(&g_CallCabinetMenu, true);
}

/**
 * @brief   cabinet の呼び出し実処理
 */
nn::Result CallCabinetImpl(
    nfpdebug::CabinetResult* pOutCabinetResult,
    CabinetMode mode,
    CabinetFlags flags) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutCabinetResult);

    bool isRegistered;
    pOutCabinetResult->Initialize();

    // ニックネーム & オーナー登録呼び出し
    auto callRegisterer = [&]
    {
        bool isNicknameRequired = (flags & CabinetFlags_RequireNickname) != 0;
        nfpdebug::NfpInfo info;
        g_Nfp.GetNfpInfo(&info);
        if ((flags & CabinetFlags_UseCurrentRegisterInfo) != 0)
        {
            // 検出済みタグの RegisterInfo を使用
            NN_ASSERT_NOT_EQUAL(flags & CabinetFlags_UseCurrentTagInfo, 0);
            return g_Nfp.CallNicknameAndOwner(
                &pOutCabinetResult->deviceHandle,
                &isRegistered,
                &pOutCabinetResult->registerInfo,
                isNicknameRequired,
                info.tagInfo,
                info.publicRegisterInfo);
        }
        else if ((flags & CabinetFlags_UseCurrentTagInfo) != 0)
        {
            // 検出済みタグの TagInfo を使用
            return g_Nfp.CallNicknameAndOwner(
                &pOutCabinetResult->deviceHandle,
                &isRegistered,
                &pOutCabinetResult->registerInfo,
                isNicknameRequired,
                info.tagInfo);
        }

        // TagInfo, RegisterInfo 共に指定しない
        pOutCabinetResult->infoFlags.Set<
            nfpdebug::CabinetResultFlags::TagInfo>();
        return g_Nfp.CallNicknameAndOwner(
            &pOutCabinetResult->deviceHandle,
            &pOutCabinetResult->tagInfo,
            &isRegistered,
            &pOutCabinetResult->registerInfo,
            isNicknameRequired);
    };

    // ゲームデータ消去呼び出し
    auto callEraser = [&]
    {
        if ((flags & CabinetFlags_UseCurrentTagInfo) != 0)
        {
            // 検出したタグを使用
            return g_Nfp.CallGameDataEraser(&pOutCabinetResult->deviceHandle);
        }

        // 検出済みのタグを使用しない
        pOutCabinetResult->infoFlags.Set<
            nfpdebug::CabinetResultFlags::TagInfo>();
        return g_Nfp.CallGameDataEraser(
            &pOutCabinetResult->deviceHandle,
            &pOutCabinetResult->tagInfo);
    };

    // 復旧モード呼出し
    auto callRestorer = [&]
    {
        if ((flags & CabinetFlags_UseCurrentTagInfo) != 0)
        {
            // 検出したタグを使用
            return g_Nfp.CallRestorer(&pOutCabinetResult->deviceHandle);
        }

        // 検出済みのタグを使用しない
        pOutCabinetResult->infoFlags.Set<
            nfpdebug::CabinetResultFlags::TagInfo>();
        return g_Nfp.CallRestorer(
            &pOutCabinetResult->deviceHandle,
            &pOutCabinetResult->tagInfo);
    };

    nn::Result result;
    switch (mode)
    {
    case CabinetMode_NicknameAndOwner:
        result = callRegisterer();
        break;
    case CabinetMode_GameDataEraser:
        result = callEraser();
        break;
    case CabinetMode_Restorer:
        result = callRestorer();
        break;
    case CabinetMode_Formatter:
        result = g_Nfp.CallFormatter(
            &pOutCabinetResult->deviceHandle,
            &pOutCabinetResult->tagInfo);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    if (result.IsFailure())
    {
        pOutCabinetResult->Initialize();
        return result;
    }

    // 成功時、DeviceHandle と RegisterInfo は常に有効
    pOutCabinetResult->infoFlags.Set<
        nfpdebug::CabinetResultFlags::DeviceHandle>();
    pOutCabinetResult->infoFlags.Set<
        nfpdebug::CabinetResultFlags::RegisterInfo>();

    NN_RESULT_SUCCESS;

}  // NOLINT(impl/function_size)

/**
 * @brief   cabinet の呼び出し
 */
void CallCabinet(const nn::hid::NpadIdType& npadId) NN_NOEXCEPT
{
    // タグ検出の要否判定
    auto NeedsTagDetection = []()
    {
        if (g_CabinetMode == CabinetMode_Formatter)
        {
            return false;
        }
        else if ((g_CabinetFlags & CabinetFlags_UseCurrentTagInfo) == 0)
        {
            return false;
        }

        return true;
    };

    // タグ検出不要な場合はすぐに呼ぶ
    if (!NeedsTagDetection())
    {
        g_Footer.SetHelpText(HelpNone);
        g_WaitDialog.SetText(
            "Launching cabinet...");
        g_ForceAction = ForceAction::CallCabinet;
        return;
    }

    // タグ検出
    auto DetectCallback = [](void* pArg)
    {
        g_DeviceListWindow.ClearActiveNpadId();

        nfpdebug::NfpInfo info;
        g_Nfp.GetNfpInfo(&info);
        if (info.isAmiibo)
        {
            g_Footer.SetHelpText(HelpNone);
            g_WaitDialog.SetText(
                "Launching cabinet...");
            g_ForceAction = ForceAction::CallCabinet;
        }
        else
        {
            g_Nfp.Deactivate();
            DisplayNotAmiiboMessage(ReturnToCabinetMenu);
        }
    };

    ExecuteSearchAction(npadId, DetectCallback, nullptr);
}

/**
 * @brief   コマンド: cabinet モード設定
 */
void CommandSetCabinetMode(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(npadId);

    g_CabinetMode = static_cast<CabinetMode>(argument);
    switch (g_CabinetMode)
    {
    case CabinetMode_Formatter:
        CallCabinet(npadId);
        return;
#ifdef NFPDEBUG_SYSTEM
    case CabinetMode_NicknameAndOwner:
        {
            NFPDEBUG_SET_MENU_ITEMS(g_SetCabinetFlagsMenu, SetCabinetFlagsMenuItemsForNickname);
            g_SetCabinetFlagsMenu.SetSize(620, 128);
        }
        break;
    default:
        {
            NFPDEBUG_SET_MENU_ITEMS(g_SetCabinetFlagsMenu, SetCabinetFlagsMenuItems);
            g_SetCabinetFlagsMenu.SetSize(320, 128);
        }
        break;
#else
    default:
        {
            NN_UNUSED(SetCabinetFlagsMenuItemsForNickname);
            NN_UNUSED(CommandFormatAmiibo);
            NN_UNUSED(CommandRestoreAmiibo);
            g_CabinetFlags = CabinetFlags_None;
            CallCabinet(npadId);
        }
        return;
#endif  // ifdef NFPDEBUG_SYSTEM
    }

#ifdef NFPDEBUG_SYSTEM
    g_SetCabinetFlagsMenu.SetCursorIndex(0);
    ActivateSubMenu(&g_SetCabinetFlagsMenu, true);
#endif  // ifdef NFPDEBUG_SYSTEM
}

/**
 * @brief   コマンド: cabinet 詳細設定
 */
void CommandSetCabinetFlags(const nn::hid::NpadIdType& npadId, uintptr_t argument) NN_NOEXCEPT
{
    NN_UNUSED(npadId);

    ActivateWindow(&g_CallCabinetMenu, false);
    g_CabinetFlags = static_cast<CabinetFlags>(argument);
    CallCabinet(npadId);
}

/**
 * @brief   ウィンドウの初期化
 */
void InitializeWindow(nfpdebug::graphics::Renderer* pRenderer) NN_NOEXCEPT
{
    for (auto* pWindow : g_AllWindows)
    {
        pWindow->Initialize();
        pWindow->SetRenderer(pRenderer);
    }

    g_DeviceListWindow.SetPosition(960, 330);
    g_DeviceListWindow.SetSize(300, 320);
    g_DeviceListWindow.SetCaption("Connected devices");
    g_DeviceListWindow.SetCaptionColor(
        nfpdebug::ui::DefaultActiveCaptionBackColor,
        nfpdebug::ui::DefaultActiveCaptionBackColor,
        nfpdebug::ui::DefaultCaptionColor);
    g_DeviceListWindow.Show();

    g_TagDumpWindow.SetPosition(120, 100);
    g_TagDumpWindow.SetCancelHandler(ReturnToTagMenu);
    g_TagDumpWindow.SetCaption("Tag data");

    g_CabinetResultWindow.SetPosition(120, 100);
    g_CabinetResultWindow.SetCancelHandler(ReturnToCabinetMenu);
    g_CabinetResultWindow.SetCaption("Cabinet result");

    g_EditAdminWindow.SetCancelHandler(ReturnToTagMenu);
    g_EditAdminWindow.SetCaption("＜ Edit admin info");

    g_EditRegisterWindow.SetCancelHandler(ReturnToTagMenu);
    g_EditRegisterWindow.SetCaption("＜ Edit register info");

    g_InputNumberWindow.SetPosition(400, 320);
    g_InputNumberWindow.SetCaption("Input number");

    g_MessageBox.SetPosition(380, 304);
    g_MessageBox.SetSize(520, 128);

    g_WaitDialog.SetPosition(380, 256);
    g_WaitDialog.SetSize(520, 224);
    g_WaitDialog.SetCaption("Processing...");

    NFPDEBUG_SET_MENU_ITEMS(g_MainMenu, MainMenuItems);
    g_MainMenu.SetPosition(32, 108);
    g_MainMenu.SetCaption("Main menu");

    NFPDEBUG_SET_MENU_ITEMS(g_DeleteInfoMenu, DeleteInfoMenuItems);
    g_DeleteInfoMenu.SetCancelHandler(ReturnToMainMenu);
    g_DeleteInfoMenu.SetCaption("＜ Delete info");

    NFPDEBUG_SET_MENU_ITEMS(g_BreakMenu, BreakMenuItems);
    g_BreakMenu.SetCancelHandler(ReturnToMainMenu);
    g_BreakMenu.SetCaption("＜ Break amiibo");

    NFPDEBUG_SET_MENU_ITEMS(g_ControlBackupMenu, ControlBackupMenuItems);
    g_ControlBackupMenu.SetCancelHandler(ReturnToMainMenu);
    g_ControlBackupMenu.SetCaption("＜ Backup data");

    NFPDEBUG_SET_MENU_ITEMS(g_TagMenu, TagMenuItems);
    g_TagMenu.SetCancelHandler(ReturnToMainMenu);
    g_TagMenu.SetCaption("＜ Tag menu");

    NFPDEBUG_SET_MENU_ITEMS(g_SelectNicknameMenu, SelectNicknameMenuItems);
    g_SelectNicknameMenu.SetCancelHandler(ReturnToRegisterInfoMenu);
    g_SelectNicknameMenu.SetCaption("＜ Select nickname");

    NFPDEBUG_SET_MENU_ITEMS(g_SelectMiiMenu, SelectMiiMenuItems);
    g_SelectMiiMenu.SetCancelHandler(ReturnToRegisterInfoMenu);
    g_SelectMiiMenu.SetCaption("＜ Select Mii");

    NFPDEBUG_SET_MENU_ITEMS(g_EditApplicationAreaMenu, EditApplicationAreaMenuItems);
    g_EditApplicationAreaMenu.SetCancelHandler(ReturnToTagMenu);
    g_EditApplicationAreaMenu.SetCaption("＜ Select application area");

    NFPDEBUG_SET_MENU_ITEMS(g_CallCabinetMenu, CallCabinetMenuItems);
    g_CallCabinetMenu.SetCancelHandler(ReturnToMainMenu);
    g_CallCabinetMenu.SetCaption(
#ifdef NFPDEBUG_SYSTEM
        "＜ Call cabinet"
#else
        "＜ Call amiibo settings"
#endif  // ifdef NFPDEBUG_SYSTEM
    );

    NFPDEBUG_SET_MENU_ITEMS(g_SetCabinetFlagsMenu, SetCabinetFlagsMenuItems);
    g_SetCabinetFlagsMenu.SetCancelHandler(ReturnToCabinetMenu);
    g_SetCabinetFlagsMenu.SetCaption("＜ Select parameter");

    g_Footer.SetRenderer(pRenderer);
    g_Footer.Show();

    ActivateWindow(&g_MainMenu, false);
}

/**
 * @brief   cabinet 呼び出しの実行
 */
void ProcessCallCabinet() NN_NOEXCEPT
{
    nfpdebug::CabinetResult cabinetResult;
    auto result = CallCabinetImpl(&cabinetResult, g_CabinetMode, g_CabinetFlags);
    g_Nfp.Deactivate();

#ifdef NFPDEBUG_SYSTEM
    // 結果表示
    if (result.IsFailure())
    {
        DisplayInformationMessage("Not changed.", ReturnToCabinetMenu);
        return;
    }

    g_CabinetResultWindow.SetCabinetResult(cabinetResult);
    ActivateWindow(&g_CabinetResultWindow, false);
    g_Footer.SetHelpText(HelpForMessageBox);
#else
    NN_UNUSED(result);
    ReturnToMainMenu(0, 0);
#endif  // ifdef NFPDEBUG_SYSTEM
}

/**
 * @brief   優先実行アクションの更新
 */
void UpdateForceAction() NN_NOEXCEPT
{
    if (g_ForceAction == ForceAction::None)
    {
        return;
    }

    switch (g_ForceAction)
    {
    case ForceAction::CallCabinet:
        {
            ProcessCallCabinet();
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    g_ForceAction = ForceAction::None;
}

/**
 * @brief   パッド入力の更新
 */
void UpdateInput() NN_NOEXCEPT
{
    for (int i = 0; i < nfpdebug::npad::NpadIdCountMax; i++)
    {
        auto id = nfpdebug::npad::NpadIds[i];
        auto* pController = nfpdebug::npad::GetNpadController(id);
        if (pController == nullptr)
        {
            continue;
        }

        auto time1 = nn::os::GetSystemTick().ToTimeSpan();
        pController->Update();
        auto time2 = nn::os::GetSystemTick().ToTimeSpan();
        g_pActiveWindow->UpdateInput(*pController);
        auto time3 = nn::os::GetSystemTick().ToTimeSpan();
        g_Nfp.UpdateInput(*pController);
        auto time4 = nn::os::GetSystemTick().ToTimeSpan();

#if 0
        if (time4.GetMilliSeconds() - time1.GetMilliSeconds() >= 100)
        {
            auto d1 = time2 - time1;
            NN_LOG("  T1: %2d.%03d\n", d1.GetSeconds(), d1.GetMilliSeconds() % 1000);
            auto d2 = time3 - time2;
            NN_LOG("  T2: %2d.%03d\n", d2.GetSeconds(), d2.GetMilliSeconds() % 1000);
            auto d3 = time4 - time3;
            NN_LOG("  T3: %2d.%03d\n", d3.GetSeconds(), d3.GetMilliSeconds() % 1000);
        }
#else
        NN_UNUSED(time1);
        NN_UNUSED(time2);
        NN_UNUSED(time3);
        NN_UNUSED(time4);
#endif
    }

    UpdateForceAction();
}

/**
 * @brief   HID 関連の処理を行うスレッド関数
 */
void HidThreadFunction(void* pArg) NN_NOEXCEPT
{
    bool* pRuns = reinterpret_cast<bool*>(pArg);
    while (*pRuns)
    {
        nn::os::WaitEvent(&g_RenderEvent);
        UpdateInput();
    }
}

/**
 * @brief   ウィンドウを更新
 */
void UpdateWindows() NN_NOEXCEPT
{
    for (auto* pWindow : g_AllWindows)
    {
        pWindow->Update();
    }
}

/**
 * @brief   システムの状態変化監視
 */
void CheckSystemMessage() NN_NOEXCEPT
{
    nn::oe::Message message;
    if (!nn::oe::TryPopNotificationMessage(&message))
    {
        return;
    }

    if (message != nn::oe::MessageFocusStateChanged)
    {
        return;
    }

    switch (nn::oe::GetCurrentFocusState())
    {
    case nn::oe::FocusState_Background:
    case nn::oe::FocusState_OutOfFocus:
        g_Nfp.NotifyLostFocus();
        break;
    default:
        // 何もしない
        break;
    }
}

/**
 * @brief   描画処理
 */
void Render() NN_NOEXCEPT
{
    NN_UTIL_SCOPE_EXIT
    {
        nn::os::SignalEvent(&g_RenderEvent);
    };

    if (!IsFocused())
    {
        // フォーカス喪失中は描画せず 1 フレーム程度の時間待つだけ
        nn::os::SleepThread(nn::TimeSpan::FromMicroSeconds(16666));
        return;
    }

    auto& renderer = *nfpdebug::graphics::Renderer::GetInstance();

    // 初期描画設定
    renderer.SetTextScale(1.2f, 1.2f);
    renderer.SetTextColor(nfpdebug::graphics::Colors::White);

    nfpdebug::PrintHeader(&renderer, ProgramName);

    for (auto* pWindow : g_AllWindows)
    {
        pWindow->Render();
    }

    g_Footer.Render();
    renderer.Render();
}

}  // anonymous

// アプリケーションのメモリ管理機構を初期化
extern "C" void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::SetMemoryHeapSize(MemoryHeapSize));

    uintptr_t address = uintptr_t();
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::AllocateMemoryBlock(&address, MallocHeapSize));

    nn::init::InitializeAllocator(
        reinterpret_cast<void*>(address), MallocHeapSize);
}

extern "C" void nnMain()
{
#ifndef NFPDEBUG_SYSTEM
    // 一般用で未使用になる定義に対する警告を抑止
    NN_UNUSED(CommandFormatAmiibo);
    NN_UNUSED(CommandRestoreAmiibo);
    NN_UNUSED(CommandBreakAmiibo);
    NN_UNUSED(SetCabinetFlagsMenuItemsForNickname);
#endif  // ifndef NFPDEBUG_SYSTEM

    InitializeWindow(nfpdebug::graphics::Renderer::GetInstance());

    nfpdebug::npad::Initialize();

    nn::nfp::InitializeDebug();
    g_Nfp.Setup(g_NfpThreadStack, sizeof(g_NfpThreadStack));

    bool runs = true;

    nn::os::InitializeEvent(&g_RenderEvent, false, nn::os::EventClearMode_AutoClear);

    // フォーカス状態変化通知を受け取る
    nn::oe::SetFocusHandlingMode(nn::oe::FocusHandlingMode_Notify);

    nn::os::CreateThread(
        &g_HidThread,
        HidThreadFunction,
        &runs,
        g_HidThreadStack,
        sizeof(g_HidThreadStack),
        nn::os::DefaultThreadPriority);
    nn::os::StartThread(&g_HidThread);

    NN_LOG("<< %s Started >>\n", ProgramName);

    while (NN_STATIC_CONDITION(runs))
    {
        UpdateWindows();
        CheckSystemMessage();
        Render();
    }

    nn::os::WaitThread(&g_HidThread);
    nn::os::DestroyThread(&g_HidThread);

    NN_LOG("Finished\n");

    g_Nfp.Shutdown();
    nn::nfp::FinalizeDebug();
    nfpdebug::graphics::Renderer::Shutdown();
}
