﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <functional>
#include <string>
#include <sstream>
#include <vector>
#include <algorithm>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/crypto/crypto_Sha1Generator.h>
#include <nn/fs.h>
#include <nn/fs_Base.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>

#include <glv.h>
#include <glv_binding.h>
#include <glv_resources.h>
#include <glv_ScissorBoxView.h>
#include <nv/nv_MemoryManagement.h>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <nn/util/util_FormatString.h>
#include "NandVerifier_Viewer.h"
#include "NandVerifier_LabelButton.h"

namespace
{
    bool CheckExtension(std::string fileName, std::string extension) NN_NOEXCEPT
    {
        if (fileName.size() < extension.size())
        {
            return false;
        }

        return fileName.substr(fileName.size() - extension.size()) == extension;
    }

    glv::Scroll* CreateScroll()
    {
        auto scroll = new glv::Scroll(glv::Rect(1000.f, 550.f), 40.0f, 5.0f, 5.0f);
        scroll->disable(glv::Property::FocusHighlight);

        return scroll;
    }
}

Viewer::Viewer(HashList** ppInHashList) NN_NOEXCEPT
    : glv::ScissorBoxView(5.f, 5.f, 1000.f, 550.f), m_ppHashList(ppInHashList), m_Scroll(nullptr), m_ResultNum(0)
{}

void Viewer::Initialize(std::function<void(std::string)> func)
{
    m_Scroll = CreateScroll();
    m_pTable = new glv::Table("<", 3, 1);

    for (int i = 0; i < PartitionNum; i++)
    {
        m_SkipCompared[i] = new ToggleButton(m_ppHashList[i]->GetPartitionName(), "verify", "ignore");
        m_pTable->add(m_SkipCompared[i]);
    }

    m_pTable->arrange();

    *m_Scroll << m_pTable;
    *this << m_Scroll;

    for (int i = 0; i < MaxResultNum; i++)
    {
        m_Labels[i] = new VerifyButton("", nullptr, 30, 1);
        m_Labels[i]->disable(glv::Property::DrawBorder);
        m_Labels[i]->SetFunction(func);
    }
}

void Viewer::ShowVerifyResult() NN_NOEXCEPT
{
    for (int i = 0; i < m_ResultNum; i++)
    {
        m_Labels[i]->setValue(m_Result[i]);
        m_pTable->add(m_Labels[i]);
    }

    m_pTable->fit(false);
    m_pTable->arrange();

    m_Scroll->add(m_pTable);
    this->add(m_Scroll);
}

void Viewer::Settings() NN_NOEXCEPT
{
    for (int i = 0; i < PartitionNum; i++)
    {
        m_pTable->add(m_SkipCompared[i]);
    }

    m_pTable->fit(false);
    m_pTable->arrange();

    m_Scroll->add(m_pTable);
    this->add(m_Scroll);
}

void Viewer::Refresh() NN_NOEXCEPT
{
    for (int i = 0; i < MaxResultNum; i++)
    {
        m_Labels[i]->setValue("");
        m_Labels[i]->remove();
    }

    for (int i = 0; i < PartitionNum; i++)
    {
        m_SkipCompared[i]->remove();
    }

    m_pTable->remove();

    if (m_Scroll)
    {
        delete m_Scroll;
        m_Scroll = CreateScroll();
    }
}

void Viewer::AddVerifyResult(int i, std::string prefix, std::string hashValue, std::string fileName) NN_NOEXCEPT
{
    const int bufferSize = 200;
    char buffer[bufferSize];

    if (fileName != "")
    {
        nn::util::SNPrintf(buffer, bufferSize, "%s %s", prefix.c_str(), fileName.c_str());
    }
    else
    {
        nn::util::SNPrintf(buffer, bufferSize, "%s %s", prefix.c_str(), hashValue.c_str());
    }
    m_Result[m_ResultNum] = std::string(buffer);
    m_ResultNum++;
}

bool Viewer::Verify() NN_NOEXCEPT
{
    bool result = true;
    m_ResultNum = 0;

    const int bufferSize = 200;
    char buffer[bufferSize];

    for (int i = 0; i < PartitionNum; i++)
    {
        int answerPos = 0;
        int hashPos = 0;

        //パーティション名を記録
        m_Result[m_ResultNum] = m_ppHashList[i]->GetPartitionName();
        m_ResultNum++;

        if (!m_SkipCompared[i]->isActive())
        {
            nn::util::SNPrintf(buffer, bufferSize, "Skipped");
            AddVerifyResult(i, "", buffer, "");
        }
        else
        {
            std::sort(m_Answer[i].begin(), m_Answer[i].end());
            if (m_Answer[i].size() != m_ppHashList[i]->GetStoredSize())
            {
                result = false;

                nn::util::SNPrintf(buffer, bufferSize, "Expected: %d file(s), Results: %d file(s).", m_Answer[i].size(), m_ppHashList[i]->GetStoredSize());
                AddVerifyResult(i, "", buffer, "");
            }
            while (NN_STATIC_CONDITION(true))
            {
                if (answerPos == m_Answer[i].size() || hashPos == m_ppHashList[i]->GetStoredSize())
                {
                    break;
                }

                if (m_Answer[i][answerPos].hashValue != m_ppHashList[i]->GetHashValue(hashPos))
                {
                    result = false;
                    if (m_Answer[i][answerPos].hashValue < m_ppHashList[i]->GetHashValue(hashPos))
                    {
                        AddVerifyResult(i, "x", m_Answer[i][answerPos].hashValue, m_Answer[i][answerPos].fileName);
                        answerPos++;
                    }
                    else
                    {
                        hashPos++;
                    }
                }
                else
                {
                    answerPos++;
                    hashPos++;
                }
            }

            for (int j = answerPos; j < m_Answer[i].size(); j++)
            {
                result = false;
                AddVerifyResult(i, "x", m_Answer[i][j].hashValue, m_Answer[i][answerPos].fileName);
                ++answerPos;
            }

            if (result)
            {
                nn::util::SNPrintf(buffer, bufferSize, "Correct");
                AddVerifyResult(i, "", buffer, "");
            }
        }
    }

    return result;
}

nn::Result Viewer::ReadAnswerFile(const char* path)  NN_NOEXCEPT
{
    nn::fs::FileHandle handle;
    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));

    int64_t fileSize = 0;
    NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));

    int totalSize = static_cast<int>(fileSize);
    std::unique_ptr<char[]> storage(new char[totalSize]);

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_RESULT_DO(nn::fs::ReadFile(handle, 0, storage.get(), totalSize));
    std::stringstream ss(storage.get());
    int partitionId = 0;
    while (!ss.eof())
    {
        if (partitionId == PartitionNum)
        {
            break;
        }

        int num;
        std::string name;
        ss >> name >> num;
        m_Answer[partitionId].resize(num);

        for (int i = 0; i < num; i++)
        {
            std::string hashValue;
            std::string fileName;
            ss >> hashValue >> fileName;

            m_Answer[partitionId][i] = { hashValue, fileName };
        }

        partitionId++;
    }

    NN_RESULT_SUCCESS;
}

nn::Result Viewer::DumpVerifyResult(glv::Label* label)
{
    std::string merged;
    merged += label->getValue().c_str();
    merged += "\n";

    for (int i = 0; i < m_ResultNum; i++)
    {
        NN_LOG("%s\n", m_Result[i].c_str());
        merged += m_Result[i].c_str();
        merged += "\n";
    }

    const char* MountName = "SdCard";
    NN_RESULT_DO(nn::fs::MountSdCardForDebug(MountName));

    NN_UTIL_SCOPE_EXIT{ nn::fs::Unmount(MountName); };

    const int PathBufferSize = 200;
    char path[PathBufferSize];

    const int MaxIdValue = 100;
    nn::Result result = nn::fs::CreateDirectory("SdCard:/NandVerifier");
    if (result.IsFailure())
    {
        if (!nn::fs::ResultPathAlreadyExists::Includes(result))
        {
            NN_LOG("Failed.\n");
            return result;
        }
    }

    for (int i = 0; i < MaxIdValue; i++)
    {
        nn::util::SNPrintf(path, PathBufferSize, "%s:/NandVerifier/Dump%04d.result", MountName, i);
        result = nn::fs::CreateFile(path, 0);
        if (result.IsFailure())
        {
            if (!nn::fs::ResultPathAlreadyExists::Includes(result))
            {
                NN_LOG("Failed.\n");
                return result;
            }
        }
        else
        {
            break;
        }
    }

    nn::fs::FileHandle handle;
    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    NN_LOG("dumped = %s\n", path);
    NN_RESULT_DO(nn::fs::WriteFile(handle, 0, merged.c_str(), merged.size(), nn::fs::WriteOption()));
    NN_RESULT_DO(nn::fs::FlushFile(handle));

    NN_RESULT_SUCCESS;
}
nn::Result Viewer::ReadAnswerFileName(std::string fileName) NN_NOEXCEPT
{
    if (CheckExtension(fileName, ".ans"))
    {
        m_Result[m_ResultNum] = fileName;
        m_ResultNum++;
    }

    NN_RESULT_SUCCESS;
}

nn::Result Viewer::EnumerateAnswerFiles(std::string rootPath) NN_NOEXCEPT
{
    nn::fs::DirectoryHandle directoryHandle;
    NN_RESULT_DO(
        nn::fs::OpenDirectory(&directoryHandle, rootPath.c_str(), nn::fs::OpenDirectoryMode_All));
    NN_UTIL_SCOPE_EXIT{ nn::fs::CloseDirectory(directoryHandle); };

    int64_t entryCount;
    NN_RESULT_DO(
        nn::fs::GetDirectoryEntryCount(&entryCount, directoryHandle));

    std::vector<nn::fs::DirectoryEntry> directoryEntries;
    directoryEntries.resize(entryCount);

    int64_t readEntryCount;
    NN_RESULT_DO(
        nn::fs::ReadDirectory(&readEntryCount, &directoryEntries[0], directoryHandle, directoryEntries.size()));

    directoryEntries.resize(readEntryCount);

    if (rootPath[rootPath.length() - 1] != '/')
    {
        rootPath += "/";
    }

    for (auto directoryEntry : directoryEntries)
    {
        switch (directoryEntry.directoryEntryType)
        {
        case nn::fs::DirectoryEntryType_File:
        {
            NN_RESULT_DO(
                ReadAnswerFileName(rootPath + directoryEntry.name));
            break;
        }
        case nn::fs::DirectoryEntryType_Directory:
        {
            NN_RESULT_DO(
                EnumerateAnswerFiles(rootPath + directoryEntry.name));
            break;
        }
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result Viewer::ReadFileList(std::string rootPath) NN_NOEXCEPT
{
    NN_RESULT_DO(EnumerateAnswerFiles(rootPath));

    std::sort(m_Result, m_Result + m_ResultNum);

    NN_RESULT_SUCCESS;
}

void Viewer::ClearAnswerFileList() NN_NOEXCEPT
{
    m_ResultNum = 0;
}
