﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstdlib>
#include <functional>
#include <string>
#include <sstream>
#include <vector>
#include <algorithm>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/nn_Macro.h>
#include <nn/crypto/crypto_Sha1Generator.h>
#include <nn/fs.h>
#include <nn/fs_Base.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_Host.h>
#include <nn/fs/fs_IStorage.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/ns/ns_SystemUpdateSystemApi.h>
#include <nn/ns/ns_InitializationApi.h>

#include <glv.h>
#include <glv_binding.h>
#include <glv_resources.h>
#include <glv_ScissorBoxView.h>
#include <nv/nv_MemoryManagement.h>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <nn/util/util_FormatString.h>

#include "NandVerifier_Allocator.h"
#include "NandVerifier_PartitionDefinitions.h"
#include "NandVerifier_LoadScene.h"
#include "NandVerifier_VerifyScene.h"
#include "NandVerifier_HashList.h"

namespace {
    const size_t GraphicsSystemReservedMemorySize = 8 * 1024 * 1024;
    NN_OS_ALIGNAS_THREAD_STACK static uint8_t s_ThreadStack[0x50000];

    static void SetupPeripherals() NN_NOEXCEPT
    {
        nv::SetGraphicsAllocator(Allocate, Deallocate, Reallocate, nullptr);
        nv::InitializeGraphics(malloc(GraphicsSystemReservedMemorySize), GraphicsSystemReservedMemorySize);
        glslcSetAllocator(Allocate, Deallocate, Reallocate, nullptr);
    }

    void MainFunction(void* args) NN_NOEXCEPT
    {
        glv::Application::run();
    }

    void Main(glv::Window& window) NN_NOEXCEPT
    {
        nn::ns::Initialize();
        NN_UTIL_SCOPE_EXIT{ nn::ns::Finalize(); };
        //起動時に未適用の NUP データを削除する
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::ns::DestroySystemUpdateTask());

        auto hashList = new HashList*[PartitionNum];
        for (int i = 0; i < PartitionNum; i++)
        {
            hashList[i] = new HashList();
        }

        auto loadScene = new LoadScene(hashList);

        window.setGLV(*loadScene);

        static nn::os::ThreadType mainThread;
        auto result = nn::os::CreateThread(
            &mainThread, MainFunction, nullptr, s_ThreadStack,
            sizeof(s_ThreadStack), nn::os::DefaultThreadPriority);
        NN_ASSERT(result.IsSuccess(), "Failed to create thread\n");
        nn::os::StartThread(&mainThread);

        loadScene->CalculateHash();

        auto verifyScene = new VerifyScene(hashList);
        window.setGLV(*verifyScene);

        nn::os::WaitThread(&mainThread);
    }
}

extern "C" void nnMain()
{
    SetupPeripherals();
    nn::fs::SetAllocator(Allocate, Deallocate);

    glv::ApplicationFrameworkInitialize(glv::HidInitialConfiguration());

    const int width = glv::glutGet(GLUT_SCREEN_WIDTH);
    const int height = glv::glutGet(GLUT_SCREEN_HEIGHT);

    glv::Window* window = new glv::Window(width, height, "Main Window");
    NN_ABORT_UNLESS_NOT_NULL(window);

    NN_UTIL_SCOPE_EXIT
    {
        delete window;
    };

    Main(*window);
}
