﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "VibrationDemo_ControllerManager.h"

namespace VibrationDemo
{
    //!< BasicLockable な MutexType を表す構造体です。
    struct LockableMutexType final
    {
        ::nn::os::MutexType _mutex;

        void lock() NN_NOEXCEPT
        {
            ::nn::os::LockMutex(&_mutex);
        }

        void unlock() NN_NOEXCEPT
        {
            ::nn::os::UnlockMutex(&_mutex);
        }
    };

    //!< SpeedChangeableVibrationPlayer 内の排他処理に用いるミューテックス
    LockableMutexType s_Mutex = { NN_OS_MUTEX_INITIALIZER(false) };

    const nn::TimeSpan ControllerSequenceManager::DefaultWaitTime = nn::TimeSpan::FromSeconds(5);
    NN_OS_ALIGNAS_THREAD_STACK char                      ControllerSequenceManager::ThreadStack[ThreadStackSize];
    nn::os::ThreadType                                   ControllerSequenceManager::Thread;

    bool ControllerSequenceManager::IsShortControllerCount() NN_NOEXCEPT
    {
        int npadCount = 0;
        for (int i = 0; i < m_SupportedNpadIdCount; ++i)
        {
            if (
                m_NpadIds[i] != nn::hid::NpadId::Handheld &&
                nn::hid::GetNpadStyleSet(m_NpadIds[i]).IsAnyOn()
                )
            {
                ++npadCount;
            }
        }
        return m_MinPlayerCount > npadCount;
    }

    bool ControllerSequenceManager::IsInputByOtherController() NN_NOEXCEPT
    {
        for (int i = 0; i < m_SupportedNpadIdCount; ++i)
        {
            if (m_NpadIds[i] != m_MasterController)
            {
                if (IsInputController(m_NpadIds[i]))
                {
                    return true;
                }
            }
        }
        return false;
    }

    bool ControllerSequenceManager::IsConnectedControllerCountExcessive() NN_NOEXCEPT
    {
        if (m_Mode == CheckControllerMode_Single)
        {   // 1人プレイのみ確認を行う
            if (m_ConnectedControllerCount > 1)
            {   // 0本 → 1本 を判定しないようにする
                return (m_PrevConnectedControllerCount < m_ConnectedControllerCount);
            }
        }
        return false;
    }

    bool ControllerSequenceManager::IsLostMasterController() NN_NOEXCEPT
    {
        return nn::hid::GetNpadStyleSet(m_MasterController).IsAllOff();
    }

    bool ControllerSequenceManager::IsResumeFocus() NN_NOEXCEPT
    {
        nn::oe::Message message;
        if (nn::oe::TryPopNotificationMessage(&message))
        {
            if (message == nn::oe::MessageFocusStateChanged)
            {
                return
                    m_PrevFocusState == nn::oe::FocusState_Background &&
                    m_CurFocusState == nn::oe::FocusState_InFocus;
            }
        }
        return false;
    }

    void ControllerSequenceManager::ControllerCheckThread(void *arg) NN_NOEXCEPT
    {
        NN_UNUSED(arg);
        ControllerSequenceManager& manager = ControllerSequenceManager::GetInstance();

        nn::os::TimerEventType timer;
        nn::os::InitializeTimerEvent(&timer, nn::os::EventClearMode_AutoClear);
        while (timer._state != timer.State_Initialized)
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));
        }
        nn::os::StartPeriodicTimerEvent(&timer, nn::TimeSpan::FromMilliSeconds(10), nn::TimeSpan::FromMilliSeconds(10));

        while (NN_STATIC_CONDITION(true))
        {
            nn::os::WaitTimerEvent(&timer);
            manager.CalcCurrentTime();
            manager.UpdateConnectedControllers();
            manager.UpdateFocusState();

            if (manager.CheckNecessaryCallApplet())
            {
                manager.CallControllerSupportApplet();
            }
        }
    }

    void ControllerSequenceManager::Initialize() NN_NOEXCEPT
    {
        static bool s_IsInitialized = false;
        if (!s_IsInitialized)
        {
            m_MemoryPool.Initialize(new nn::Bit8[MemorySize], MemorySize);
            s_IsInitialized = true;
        }
    }

    bool ControllerSequenceManager::TryWait() NN_NOEXCEPT
    {
        return CheckNecessaryCallApplet();
    }

    void ControllerSequenceManager::UpdateConnectedControllers() NN_NOEXCEPT
    {
        bool isMasterControllerConnected = false;
        bool isSubMasterControllerConnected = false;
        m_PrevConnectedControllerCount = m_ConnectedControllerCount;
        m_ConnectedControllerCount = 0;
        // コントローラの接続状態を更新します
        for (int i = 0; i < m_SupportedNpadIdCount; ++i)
        {
            if (nn::hid::GetNpadStyleSet(m_NpadIds[i]).IsAnyOn())
            {   // 接続されているコントローラを発見した場合
                m_ConnectedNpadIds[m_ConnectedControllerCount++] = m_NpadIds[i];
                if (
                    m_NpadIds[i] == nn::hid::NpadId::No1 ||
                    m_NpadIds[i] == nn::hid::NpadId::Handheld
                    )
                {   // マスターコントローラ候補のコントローラが接続されている場合
                    if (m_IsDecidedMasterController)
                    {   // マスターコントローラが確定している場合
                        if (m_NpadIds[i] == m_MasterController)
                        {   // マスターコントローラだった場合
                            isMasterControllerConnected = true;
                        }
                        else
                        {   // マスターコントローラ候補のコントローラが接続されている場合
                            isSubMasterControllerConnected = true;
                        }
                    }
                    else
                    {   // マスターコントローラが確定していない場合
                        if (IsInputController(m_NpadIds[i]))
                        {   // 入力がなされている場合
                            m_IsDecidedMasterController = true;
                            m_MasterController = m_NpadIds[i];
                        }
                    }
                }
            }
        }

        if (!isMasterControllerConnected && isSubMasterControllerConnected)
        {   // マスターコントローラが接続されていない場合はマスターコントローラ候補のコントローラに権限を委譲します
            isMasterControllerConnected = true;
            m_MasterController = m_MasterController == nn::hid::NpadId::No1 ? nn::hid::NpadId::Handheld : nn::hid::NpadId::No1;
        }

        // コントローラの接続待ち状態 (1人プレイ) の際の処理
        if (m_IsAwaitConnection)
        {   // 接続待ち状態の場合
            if (isMasterControllerConnected)
            {   // マスターコントローラが再度/新規接続された場合
                m_IsAwaitConnection = false;
            }
            else
            {   // マスターコントローラが接続されていない場合
                if (IsSwitchOperationMode())
                {   // マスターコントローラが接続されていない状態で動作モードの切り替えが発生した場合
                    m_IsAddWaitTime = true;
                }
                if (m_CurrentTime - m_LostTime > m_WaitTime)
                {   // 切断から一定時間が経過した場合
                    if (m_IsAddWaitTime)
                    {   // 動作モードの切り替えが発生している場合は、待ち時間を延長する
                        m_WaitTime += m_AdditionalTime;
                    }
                }
            }
        }
        else
        {   // 接続待ち状態で無い場合
            if (isMasterControllerConnected)
            {   // マスターコントローラが消失している場合接続待ち状態に移行する
                m_LostTime = m_CurrentTime;
                m_WaitTime = DefaultWaitTime;
                m_IsAwaitConnection = true;
            }
        }
    }

    void ControllerSequenceManager::UpdateFocusState() NN_NOEXCEPT
    {
        m_PrevFocusState = m_CurFocusState;
        m_CurFocusState = nn::oe::GetCurrentFocusState();
    }

    bool ControllerSequenceManager::SwitchMasterController() NN_NOEXCEPT
    {
        if (
            m_MasterController == nn::hid::NpadId::No1 &&
            nn::hid::GetNpadStyleSet(nn::hid::NpadId::Handheld).IsAnyOn()
            )
        {
            m_MasterController = nn::hid::NpadId::Handheld;
            return true;
        }
        if (
            m_MasterController == nn::hid::NpadId::Handheld &&
            nn::hid::GetNpadStyleSet(nn::hid::NpadId::No1).IsAnyOn()
            )
        {
            m_MasterController = nn::hid::NpadId::No1;
            return true;
        }
        return false;
    }

    void ControllerSequenceManager::ChangeSingleMode() NN_NOEXCEPT
    {
        // 既に接続されているマスター以外のコントローラの接続は維持します。
        // ただし、一度切断して再接続した場合はアプレットの呼び出しを行います。
        m_Mode = CheckControllerMode_Single;
    }

    void ControllerSequenceManager::ChangeSingleMode(nn::hid::NpadIdType master) NN_NOEXCEPT
    {
        // 既に接続されているマスター以外のコントローラの接続は維持します。
        // ただし、一度切断して再接続した場合はアプレットの呼び出しを行います。
        m_Mode = CheckControllerMode_Single;
        m_MasterController = master;
        m_IsDecidedMasterController = true;
    }

    void ControllerSequenceManager::ChangeMultiMode() NN_NOEXCEPT
    {
        m_Mode = CheckControllerMode_Multi;
    }

    void ControllerSequenceManager::SetArgument(const ControllerSequenceManagerArg& arg) NN_NOEXCEPT
    {
        m_MinPlayerCount = arg.minPlayerCount;
        m_MaxPlayerCount = arg.maxPlayerCount;
        arg.isSingleMode ? ChangeSingleMode(m_MasterController) : ChangeMultiMode();
    }

    void ControllerSequenceManager::SetSupportedNpadIds(nn::hid::NpadIdType* ids, int count)
    {
        NN_ASSERT_NOT_NULL(ids);
        NN_ASSERT_LESS_EQUAL(count, 9);
        for (int i = 0; i < count; ++i)
        {
            m_NpadIds[i] = ids[i];
        }
        m_SupportedNpadIdCount = count;
    }

    void ControllerSequenceManager::StartControllerCheckThread() NN_NOEXCEPT
    {
        if (Thread._state != Thread.State_Started)
        {
            m_ConnectedControllerCount = CalcConnectedControllerCount();
            nn::os::StartThread(&Thread);
        }
    }

    void ControllerSequenceManager::StartControllerCheckThread(const ControllerSequenceManagerArg& arg) NN_NOEXCEPT
    {
        SetArgument(arg);
        StartControllerCheckThread();
    }

    void ControllerSequenceManager::StopControllerCheckThread() NN_NOEXCEPT
    {
        nn::os::StopTimerEvent(&m_CheckEvent);
    }

    bool ControllerSequenceManager::CheckNecessaryCallApplet() NN_NOEXCEPT
    {
        static bool prevResult = false;
        bool result = false;
        switch (m_Mode)
        {
        case VibrationDemo::ControllerSequenceManager::CheckControllerMode_Single:  // 1人プレイ時のコンサポ呼び出し確認
        {
            // マスターコントローラ以外のコントローラから入力が入った場合
            if (IsInputByOtherController())
            {
                if (!prevResult) { NN_LOG("Detected the input from the controller which was not a master controller.\n"); }
                result = true;
            }
            // 新しいコントローラが接続された場合
            if (IsConnectedControllerCountExcessive())
            {
                if (!prevResult) { NN_LOG("Connected New Controller.\n"); }
                result = true;
            }
            // マスターコントローラを消失した場合
            if (IsLostMasterController())
            {   // 一定時間待つ
                if (m_CurrentTime - m_LostTime > m_WaitTime)
                {   // 切断から一定時間が経過した場合
                    if (!prevResult) { NN_LOG("Lost a master controller.\n"); }
                    result = true;
                }
            }
            // 復帰直後の処理
            if (IsResumeFocus())
            {   // 復帰直後にマスターコントローラが消失していた場合は即時呼び出しを行う
                if (nn::hid::GetNpadStyleSet(m_MasterController).IsAllOff())
                {
                    if (!prevResult) { NN_LOG("Lost a master controller. (Resume)\n"); }
                    result = true;
                }
            }
            break;
        }
        case VibrationDemo::ControllerSequenceManager::CheckControllerMode_Multi:   // マルチプレイ時のコンサポ呼び出し確認
        {
            // コントローラが人数分接続されていない場合、即時呼び出し
            result |= IsShortControllerCount();
            break;
        }
        default: NN_UNEXPECTED_DEFAULT;
        }
        prevResult = result;
        return result;
    }

    nn::Result ControllerSequenceManager::CallControllerSupportApplet() NN_NOEXCEPT
    {
        std::lock_guard<decltype(s_Mutex)> lock(s_Mutex);

        nn::hid::ControllerSupportArg arg;
        nn::hid::ControllerSupportResultInfo resultInfo;

        arg.SetDefault();
        arg.playerCountMin = m_MinPlayerCount;
        arg.playerCountMax = m_MaxPlayerCount;
        arg.enableSingleMode = (m_Mode == CheckControllerMode_Single);
        nn::hid::SetNpadJoyHoldType(nn::hid::NpadJoyHoldType_Horizontal);

        auto result = nn::hid::ShowControllerSupport(&resultInfo, arg);

        if (result.IsSuccess())
        {
            if (m_Mode == CheckControllerMode_Single)
            {   // 1人プレイ時の場合
                m_IsDecidedMasterController = true;
                m_MasterController = resultInfo.selectedId;
                if (
                    m_MasterController == nn::hid::NpadId::No1 &&
                    nn::hid::GetNpadStyleSet(nn::hid::NpadId::Handheld).IsAnyOn()
                    )
                {   // 無線コンを選択した場合、本体装着コンを接続本数に反映する
                    m_ConnectedControllerCount = 2;
                }
                else
                {   // 本体装着コンを選択した場合、無線コンは切断される
                    m_ConnectedControllerCount = 1;
                }
            }

            UpdateConnectedControllers();
        }
        else if (nn::hid::ResultControllerSupportCanceled::Includes(result))
        {
            //ShowContinueDialog();
        }
        else if (nn::hid::ResultControllerSupportNotSupportedNpadStyle::Includes(result))
        {
            //ShowContinueDialog();
        }

        return result;
    }

    int ControllerSequenceManager::CalcConnectedControllerCount() NN_NOEXCEPT
    {
        int count = 0;
        for (int i = 0; i < m_SupportedNpadIdCount; ++i)
        {
            if (nn::hid::GetNpadStyleSet(m_NpadIds[i]).IsAnyOn())
            {
                ++count;
            }
        }
        return count;
    }

    bool ControllerSequenceManager::IsInputController(nn::hid::NpadIdType id) NN_NOEXCEPT
    {
        bool result = false;
        auto style = nn::hid::GetNpadStyleSet(id);
        if (style.Test<nn::hid::NpadStyleFullKey>()) {
            nn::hid::NpadFullKeyState state;
            nn::hid::GetNpadState(&state, id);
            result = state.buttons.IsAnyOn();
        }
        else if (style.Test<nn::hid::NpadStyleJoyDual>()) {
            nn::hid::NpadJoyDualState state;
            nn::hid::GetNpadState(&state, id);
            result = state.buttons.IsAnyOn();
        }
        else if (style.Test<nn::hid::NpadStyleJoyLeft>()) {
            nn::hid::NpadJoyLeftState state;
            nn::hid::GetNpadState(&state, id);
            result = state.buttons.IsAnyOn();
        }
        else if (style.Test<nn::hid::NpadStyleJoyRight>()) {
            nn::hid::NpadJoyRightState state;
            nn::hid::GetNpadState(&state, id);
            result = state.buttons.IsAnyOn();
        }
        else if (style.Test<nn::hid::NpadStyleHandheld>()) {
            nn::hid::NpadHandheldState state;
            nn::hid::GetNpadState(&state, id);
            result = state.buttons.IsAnyOn();
        }
        return result;
    }

    bool ControllerSequenceManager::ShowContinueDialog() NN_NOEXCEPT
    {
        nn::hid::NpadHandheldState states[nn::hid::NpadStateCountMax];
        while (true)
        {
            auto count = nn::hid::GetNpadStates(states, nn::hid::NpadStateCountMax, nn::hid::NpadId::Handheld);

            for (int i = 0; i < count; ++i)
            {
                if (states[i].buttons.Test(nn::hid::NpadButton::A::Index))
                {
                    return true;
                }
            }

            gDrawer.BeginDraw();
            {
                gDrawer.SetColor(nn::util::Color4u8::White());
                gDrawer.ScreenShot();
                gDrawer.Draw2DScreenShot(nn::util::MakeFloat2(0.f, 0.f), nn::util::MakeFloat2(0.f, 0.f));
            }
            gDrawer.EndDraw();
        }
        return false;
    }
}
