﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <map>
#include <unordered_map>
#include <vector>
#include <string>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_TimeSpan.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/mem.h>
#include <nn/audio.h>
#include <nns/audio/audio_HidUtilities.h>
#include <nns/audio/audio_WavFormat.h>

namespace VibrationDemo
{
    typedef int32_t SoundHandle;

    extern const SoundHandle ErrorSoundHandle;

    enum SoundFileStorageType
    {
        SoundFileStorageType_Memory,
        SoundFileStorageType_File,
        SoundFileStorageType_None
    };

    //=================================================
    // Audio 用メモリ管理クラス
    //=================================================
    template <const size_t HEAPSIZE>
    class AudioHeap
    {
        NN_DISALLOW_COPY(AudioHeap);
        NN_DISALLOW_MOVE(AudioHeap);
    public:
        AudioHeap() NN_NOEXCEPT : m_pMemoryHeap(nullptr)
        {
        }
        ~AudioHeap() NN_NOEXCEPT
        {
            if (m_pMemoryHeap != nullptr)
            {
                m_Allocator.Free(m_pMemoryHeap);
            }
            m_Allocator.Finalize();
        }
        void Initialize() NN_NOEXCEPT
        {
            m_pMemoryHeap = new nn::Bit8[HEAPSIZE];
            m_Allocator.Initialize(m_pMemoryHeap, HEAPSIZE);
        }
        void* Allocate(const size_t size) NN_NOEXCEPT
        {
            return this->Allocate(size, sizeof(void*));
        }
        void* Allocate(const size_t size, const size_t align) NN_NOEXCEPT
        {
            return m_Allocator.Allocate(size, align);
        }
        void Deallocate(void* ptr) NN_NOEXCEPT
        {
            if (ptr != nullptr)
            {
                m_Allocator.Free(ptr);
            }
        }
        size_t GetHeapSize() NN_NOEXCEPT { return HEAPSIZE; }

        void* GetHeap() NN_NOEXCEPT { return m_pMemoryHeap; }
    private:
        void*                       m_pMemoryHeap;
        nn::mem::StandardAllocator  m_Allocator;
    };

    typedef AudioHeap<64 * 1024 * 1024> WaveResourceHeap;
    typedef AudioHeap<8 * 1024 * 1024> RendererHeap;
    typedef AudioHeap<12 * 1024 * 1024> EffectHeap;

    class SoundSource
    {
    public:
        SoundSource() NN_NOEXCEPT;
        virtual ~SoundSource() NN_NOEXCEPT;

        virtual void Play() NN_NOEXCEPT = 0;
        virtual void Stop() NN_NOEXCEPT = 0;
    };

    class WaveSource : public SoundSource
    {
    public:
        virtual void Play() NN_NOEXCEPT {}
        virtual void Stop() NN_NOEXCEPT {}

        void* GetBuffer() NN_NOEXCEPT { return &m_pWaveHeap; }
        nn::audio::WaveBuffer* GetWavBuffer() NN_NOEXCEPT { return &m_WaveBuffer; }

        void ReadWave(uint8_t* pDate, size_t fileSize, WaveResourceHeap* pWaveHeap) NN_NOEXCEPT;
        void ClearWaveBuffer() NN_NOEXCEPT;

        void SetFileName(const char* name) NN_NOEXCEPT { m_FileName = name; }
        std::string GetFileName() NN_NOEXCEPT { return m_FileName; }
        size_t GetDataSize() NN_NOEXCEPT { return m_WaveDataSize; }
    protected:
        std::string             m_FileName;
        size_t                  m_WaveDataSize;
        void*                   m_pWaveHeap;;
        nn::audio::WaveBuffer   m_WaveBuffer;
    };

    struct AudioQueue
    {
        std::string     fileName;
        nn::TimeSpan    delay;
        float           balance;        //!< Left 0..f .. Center 0.5f .. Right 1.f
        float           volume;
    };

    class AudioManager
    {
        NN_DISALLOW_COPY(AudioManager);
        NN_DISALLOW_MOVE(AudioManager);
    public:
        static const nn::TimeSpan UpdateInterval;
        static const size_t AudioManagerThreadStackSize = 64 * 1024 * 1024;

        static const int ChannelCount = 2;
        static const int RenderRate = 32000;
        static const int RenderCount = (RenderRate / 200);

        static const int LimitWaveFileSize = 4 * 1024 * 1024;

        static const int EffectHeapSize = 12 * 1024 * 1024;
        static const int VoiceCountMax = 12;
        static const float LimitVolume;
    public:
        AudioManager() NN_NOEXCEPT;
        ~AudioManager() NN_NOEXCEPT;

        static AudioManager& GetInstance() NN_NOEXCEPT;

        size_t LoadSoundFilesFromResource() NN_NOEXCEPT;

        void Initialize() NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;
        static void UpdateThread(void *arg) NN_NOEXCEPT;
        void UpdateCurrentAudio() NN_NOEXCEPT;

        void StartUpdateThread() NN_NOEXCEPT;
        void StopUpdateThread() NN_NOEXCEPT;

        void RunAudioQueue() NN_NOEXCEPT;

        std::map<std::string, WaveSource>& GetWaveFileList() NN_NOEXCEPT { return m_WaveFileList; }

        void PlayWav(const char* fileName) NN_NOEXCEPT;
        void PlayWav(const char* fileName, float volume) NN_NOEXCEPT;
        void PlayWav(const char* fileName, float volume, float balance) NN_NOEXCEPT;
        void PlayWav(const char* fileName, const nn::TimeSpan& delay) NN_NOEXCEPT;
        void PlayWav(const char* fileName, const nn::TimeSpan& delay, float volume) NN_NOEXCEPT;
        void PlayWav(const char* fileName, const nn::TimeSpan& delay, float volume, float balance) NN_NOEXCEPT;
    public:
        static nn::os::TimerEventType UpdateEvemt;
        static nn::os::SystemEvent     RendererEvent;
        static NN_OS_ALIGNAS_THREAD_STACK char                      ThreadStack[AudioManagerThreadStackSize];
    private:
        nn::audio::AudioRendererHandle          m_RendererHandle;
        nn::audio::AudioRendererConfig          m_RendererConfig;
        nn::audio::DelayType                    m_EffectDelay;

        nn::os::ThreadType                      m_UpdateThread;
        std::map<std::string, WaveSource>       m_WaveFileList;

        RendererHeap                m_RendererBuffer;
        WaveResourceHeap            m_WaveResourceBuffer;
        EffectHeap                  m_EffectBuffer;

        nn::audio::FinalMixType     m_FinalMix;
        nn::audio::SubMixType       m_SubMix[ChannelCount];
        nn::audio::VoiceType        m_Voices[VoiceCountMax];

        std::vector<AudioQueue>     m_AudioQueue;

        int8_t                      m_MainBus[ChannelCount];
        int8_t                      m_AuxBusA[ChannelCount];
        int8_t                      m_AuxBusB[ChannelCount];
    };

#define gAudioManager (VibrationDemo::AudioManager::GetInstance())
}
