﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include "VibrationDemo_AudioManager.h"

namespace VibrationDemo
{
    const SoundHandle ErrorSoundHandle = -1;

    SoundSource::SoundSource() NN_NOEXCEPT {}

    SoundSource::~SoundSource() NN_NOEXCEPT {}

    void WaveSource::ReadWave(uint8_t* pDate, size_t fileSize, WaveResourceHeap* pWaveHeap) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pDate);
        NN_ASSERT_NOT_NULL(pWaveHeap);

        nns::audio::WavFormat format;

        // DATA チャンクを読む必要がありますが、ここではそれが 1024 バイト以内に見つかると仮定しています
        const std::size_t WavHeaderDataSize = fileSize < 1024 ? fileSize : 1024;
        nns::audio::WavResult wavResult = nns::audio::ParseWavFormat(&format, pDate, WavHeaderDataSize);
        NN_ABORT_UNLESS_EQUAL(wavResult, nns::audio::WavResult_Success);
        NN_ABORT_UNLESS_EQUAL(format.bitsPerSample, 16);  // このサンプルでは 16bit PCM を仮定しています

        m_WaveDataSize = static_cast<std::size_t>(format.dataSize);
        m_pWaveHeap = pWaveHeap->Allocate( m_WaveDataSize, nn::audio::BufferAlignSize);
        std::memcpy(m_pWaveHeap, &pDate[format.dataOffset], m_WaveDataSize);
    }

    void WaveSource::ClearWaveBuffer() NN_NOEXCEPT
    {
        m_WaveBuffer.endSampleOffset = m_WaveDataSize / sizeof(int16_t) / (AudioManager::ChannelCount > 0 ? AudioManager::ChannelCount : 1);
        m_WaveBuffer.pContext = nullptr;
        m_WaveBuffer.contextSize = 0;
        m_WaveBuffer.buffer = m_pWaveHeap;
        m_WaveBuffer.size = m_WaveDataSize;
        m_WaveBuffer.startSampleOffset = 0;
        m_WaveBuffer.loop = false;
        m_WaveBuffer.isEndOfStream = true;
    }

    const nn::TimeSpan AudioManager::UpdateInterval = nn::TimeSpan::FromMilliSeconds(5);
    nn::os::TimerEventType AudioManager::UpdateEvemt;
    nn::os::SystemEvent AudioManager::RendererEvent;
    NN_OS_ALIGNAS_THREAD_STACK char AudioManager::ThreadStack[AudioManagerThreadStackSize];
    const float AudioManager::LimitVolume = 0.02f;

    AudioManager::AudioManager() NN_NOEXCEPT
    {

    }

    AudioManager::~AudioManager() NN_NOEXCEPT
    {

    }

    AudioManager& AudioManager::GetInstance() NN_NOEXCEPT
    {
        static AudioManager instance;
        return instance;
    }

    size_t AudioManager::LoadSoundFilesFromResource() NN_NOEXCEPT
    {
        size_t resultFileCount = 0;
        const size_t cacheSize = 16 * 1024 * 1024;
        uint8_t* waveCache = new uint8_t[LimitWaveFileSize];
        char* cache = new char[cacheSize];

        nn::Result result;

        result = nn::fs::MountRom("AUDR", cache, cacheSize);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        nn::fs::DirectoryHandle handle;

        result = nn::fs::OpenDirectory(&handle, "AUDR:/se/", nn::fs::OpenDirectoryMode_File);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        nn::fs::FileHandle fileHandle;
        int64_t fileCount = 0;
        nn::fs::DirectoryEntry fileEntry[256];

        result = nn::fs::ReadDirectory(&fileCount, fileEntry, handle, NN_ARRAY_SIZE(fileEntry));
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        for (int64_t i = 0; i < fileCount; ++i)
        {
            std::string str = fileEntry[i].name;
            std::string ext;
            size_t pos = str.rfind('.');
            if (pos != std::string::npos) {
                ext = str.substr(pos + 1, str.size() - pos);
                std::string::iterator itr = ext.begin();
                while (itr != ext.end()) {
                    *itr = tolower(*itr);
                    itr++;
                }
                itr = ext.end() - 1;
                while (itr != ext.begin()) {    // パスの最後に\0やスペースがあったときの対策
                    if (*itr == 0 || *itr == 32) {
                        ext.erase(itr--);
                    }
                    else {
                        itr--;
                    }
                }
            }
            if (pos != std::string::npos && (ext == "wave" || ext == "wav"))
            {
                if (m_WaveFileList.find(fileEntry[i].name) == m_WaveFileList.end())
                {
                    result = nn::fs::OpenFile(&fileHandle, (std::string("AUDR:/se/") + fileEntry[i].name).c_str(), nn::fs::OpenMode_Read);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

                    WaveSource source;
                    bool isReadFile = false;
                    int64_t fileSize = 0;
                    std::string fileName = fileEntry[i].name;

                    nn::fs::GetFileSize(&fileSize, fileHandle);
                    if (fileSize < LimitWaveFileSize)
                    {
                        nn::fs::ReadFile(fileHandle, 0, waveCache, fileSize);
                        source.ReadWave(waveCache, fileSize, &m_WaveResourceBuffer);
                        isReadFile = true;
                    }
                    source.SetFileName(fileEntry[i].name);

                    m_WaveFileList.emplace(std::make_pair(fileName, source));

                    nn::fs::CloseFile(fileHandle);
                }
                ++resultFileCount;
            }
        }
        nn::fs::CloseDirectory(handle);

        nn::fs::Unmount("AUDR");
        delete[] cache;
        delete[] waveCache;
        return resultFileCount;
    }

    void AudioManager::Initialize() NN_NOEXCEPT
    {
        m_RendererBuffer.Initialize();
        m_WaveResourceBuffer.Initialize();
        m_EffectBuffer.Initialize();
        //------------------------------------------------------
        // レンダラの初期化
        //------------------------------------------------------
        // レンダラのパラメータを設定します
        nn::audio::AudioRendererParameter parameter;
        nn::audio::InitializeAudioRendererParameter(&parameter);
        parameter.sampleRate = RenderRate;
        parameter.sampleCount = RenderCount;
        parameter.mixBufferCount = 6 + 2; // FinalMix(6) + SubMix(2)
        parameter.subMixCount = 2;
        parameter.voiceCount = VoiceCountMax * ChannelCount;
        parameter.sinkCount = 1;
        parameter.effectCount = 3;
        parameter.performanceFrameCount = 0;

        // ミックスバッファとオーディオバスの関係を定義します。
        m_AuxBusA[nn::audio::ChannelMapping_FrontLeft] = 0;
        m_AuxBusA[nn::audio::ChannelMapping_FrontRight] = 1;
        m_AuxBusB[nn::audio::ChannelMapping_FrontLeft] = 2;
        m_AuxBusB[nn::audio::ChannelMapping_FrontRight] = 3;
        m_MainBus[nn::audio::ChannelMapping_FrontLeft] = 4;
        m_MainBus[nn::audio::ChannelMapping_FrontRight] = 5;

        // パラメータがシステムでサポートされているかどうかを確認します。
        NN_ABORT_UNLESS(
            nn::audio::IsValidAudioRendererParameter(parameter),
            "Invalid AudioRendererParameter specified."
        );

        // レンダラのワークバッファを準備します。
        size_t workBufferSize = nn::audio::GetAudioRendererWorkBufferSize(parameter);
        void* workBuffer = m_RendererBuffer.Allocate(workBufferSize, nn::os::MemoryPageSize);
        NN_ABORT_UNLESS_NOT_NULL(workBuffer);

        // レンダラを初期化します。
        NN_ABORT_UNLESS(
            nn::audio::OpenAudioRenderer(&m_RendererHandle, &RendererEvent, parameter, workBuffer, workBufferSize).IsSuccess(),
            "Failed to open AudioRenderer"
        );

        // AudioRendererConfig を初期化します。
        size_t configBufferSize = nn::audio::GetAudioRendererConfigWorkBufferSize(parameter);
        void* configBuffer = m_RendererBuffer.Allocate(configBufferSize, nn::os::MemoryPageSize);
        NN_ABORT_UNLESS_NOT_NULL(configBuffer);
        nn::audio::InitializeAudioRendererConfig(&m_RendererConfig, parameter, configBuffer, configBufferSize);

        nn::audio::AcquireFinalMix(&m_RendererConfig, &m_FinalMix, 6);

        for (size_t i = 0; i < NN_ARRAY_SIZE(m_SubMix); ++i)
        {
            nn::audio::AcquireSubMix(&m_RendererConfig, &m_SubMix[i], parameter.sampleRate, 1);
        }

        //------------------------------------------------------
        // レンダラの接続
        //------------------------------------------------------
        // レンダラの出力先を用意します。
        nn::audio::DeviceSinkType deviceSink;
        // オーディオ出力デバイスへの入力を設定します。
        // mainBus に指定したミックスバッファのインデックスに応じて出力チャンネルが決定されます。
        // mainBus[nn::audio::ChannelMapping_FrontLeft] が L チャンネルに、
        // mainBus[nn::audio::ChannelMapping_FrontRight] が R チャンネルにそれぞれ出力されます。
        nn::Result result = nn::audio::AddDeviceSink(&m_RendererConfig, &deviceSink, &m_FinalMix, m_MainBus, ChannelCount, "MainAudioOut");
        NN_ABORT_UNLESS(result.IsSuccess());

        // SubMix を FinalMix へ接続します
        nn::audio::SetSubMixDestination(&m_RendererConfig, &m_SubMix[0], &m_FinalMix);

        ///ボリューム 0.5f でミックスするように設定します
        nn::audio::SetSubMixMixVolume(&m_SubMix[0], &m_FinalMix, 0.5f, 0, m_MainBus[0]);
        nn::audio::SetSubMixMixVolume(&m_SubMix[0], &m_FinalMix, 0.5f, 0, m_MainBus[1]);

        for (size_t i = 1; i < NN_ARRAY_SIZE(m_SubMix); ++i)
        {
            // SubMix(1) を SubMix(0) へ接続します
            nn::audio::SetSubMixDestination(&m_RendererConfig, &m_SubMix[i], &m_SubMix[0]);

            // SubMix(1) の 0 番目のバッファを SubMix(0) の 0 番目のバッファへ
            // ボリューム 0.5f でミックスするように設定します
            nn::audio::SetSubMixMixVolume(&m_SubMix[i], &m_SubMix[0], 0.5f, 0, 0);
        }

        //------------------------------------------------------
        // エフェクトの初期化
        //------------------------------------------------------
        // エフェクトバッファ用のメモリプールを準備します。
        nn::audio::MemoryPoolType effectBufferPool;
        bool rt = AcquireMemoryPool(&m_RendererConfig, &effectBufferPool, m_EffectBuffer.GetHeap(), EffectHeapSize);
        NN_ABORT_UNLESS(rt);
        // メモリプールをアタッチします。設定は次の nn::audio::RequestUpdateAudioRenderer() が呼ばれた以降に反映されます。
        rt = RequestAttachMemoryPool(&effectBufferPool);
        NN_ABORT_UNLESS(rt);

        // Delay と BufferMixer を用意し、Delay, BufferMixer の順に登録します。

        const nn::TimeSpan delayTimeMax = nn::TimeSpan::FromSeconds(10); // delay エフェクトで利用しうる最大のディレイ時間を指定します。
        size_t delaySize = nn::audio::GetRequiredBufferSizeForDelay(delayTimeMax, parameter.sampleRate, ChannelCount);
        // エフェクトバッファを取得します。g_EffectBufferAllocator が管理する全領域は effectBufferPool メモリプールに登録してあります。
        void* delayBuffer = m_EffectBuffer.Allocate(delaySize, nn::audio::BufferAlignSize);
        NN_ASSERT_NOT_NULL(delayBuffer);

        nn::audio::BufferMixerType mixer0;
        nn::audio::BufferMixerType mixer1;

        result = nn::audio::AddDelay(&m_RendererConfig, &m_EffectDelay, delayBuffer, delaySize, &m_FinalMix, delayTimeMax, ChannelCount);
        NN_ABORT_UNLESS(result.IsSuccess());
        nn::audio::SetDelayInputOutput(&m_EffectDelay, m_AuxBusA, m_AuxBusA, ChannelCount);
        result = nn::audio::AddBufferMixer(&m_RendererConfig, &mixer0, &m_FinalMix);
        NN_ABORT_UNLESS(result.IsSuccess());
        nn::audio::SetBufferMixerInputOutput(&mixer0, m_AuxBusB, m_AuxBusA, ChannelCount);
        result = nn::audio::AddBufferMixer(&m_RendererConfig, &mixer1, &m_FinalMix);
        NN_ABORT_UNLESS(result.IsSuccess());
        nn::audio::SetBufferMixerInputOutput(&mixer1, m_AuxBusA, m_MainBus, ChannelCount);

        // Delay パラメータを設定します。
        nn::audio::SetDelayEnabled(&m_EffectDelay, true);
        nn::audio::SetDelayTime(&m_EffectDelay, delayTimeMax);
        nn::audio::SetDelayFeedbackGain(&m_EffectDelay, 0.7f);
        nn::audio::SetDelayChannelSpread(&m_EffectDelay, 0.256f);
        nn::audio::SetDelayLowPassAmount(&m_EffectDelay, 0.9f);
        nn::audio::SetDelayDryGain(&m_EffectDelay, 0.5f);
        nn::audio::SetDelayInGain(&m_EffectDelay, 0.4f);

        // BufferMixer パラメータを設定します。
        nn::audio::SetBufferMixerVolume(&mixer0, 0, 1.0f);
        nn::audio::SetBufferMixerVolume(&mixer0, 1, 1.0f);
        nn::audio::SetBufferMixerVolume(&mixer1, 0, 1.0f);
        nn::audio::SetBufferMixerVolume(&mixer1, 1, 1.0f);

        // 設定したパラメータをレンダラに反映させます。
        result = nn::audio::RequestUpdateAudioRenderer(m_RendererHandle, &m_RendererConfig);
        if (nn::audio::ResultNoMemoryPoolEntry::Includes(result))
        {
            NN_LOG("\n\n\nResultNoMemoryPoolEntry\n\n\n");
        }
        if (nn::audio::ResultInvalidUpdateInfo::Includes(result))
        {
            NN_LOG("\n\n\nResultInvalidUpdateInfo\n\n\n");
        }
        NN_ABORT_UNLESS(result.IsSuccess());

        // レンダリングを開始します。
        result = nn::audio::StartAudioRenderer(m_RendererHandle);
        NN_ABORT_UNLESS(result.IsSuccess());

        // WaveBuffer に追加するサンプルデータを保持するためのメモリプールを準備します。
        nn::audio::MemoryPoolType waveBufferMemoryPool;
        rt = AcquireMemoryPool(&m_RendererConfig, &waveBufferMemoryPool, m_WaveResourceBuffer.GetHeap(), m_WaveResourceBuffer.GetHeapSize());
        NN_ABORT_UNLESS(rt);
        rt = RequestAttachMemoryPool(&waveBufferMemoryPool);
        NN_ABORT_UNLESS(rt);
        for (auto& voice : m_Voices)
        {
            rt = nn::audio::AcquireVoiceSlot(
                &m_RendererConfig, &voice, RenderRate,
                ChannelCount, nn::audio::SampleFormat_PcmInt16, nn::audio::VoiceType::PriorityHighest, nullptr, 0);
            NN_ABORT_UNLESS(rt);

            nn::audio::SetVoicePlayState(&voice, nn::audio::VoiceType::PlayState_Play);
            nn::audio::SetVoiceDestination(&m_RendererConfig, &voice, &m_FinalMix);

            // ボイスの 0 チャンネルを mainBus[0] へ、1 チャンネルを mainBus[1] へ出力するミックスボリュームを設定します。
            nn::audio::SetVoiceMixVolume(&voice, &m_FinalMix, 1.000f, 0, m_MainBus[0]);
            nn::audio::SetVoiceMixVolume(&voice, &m_FinalMix, 1.000f, 0, m_MainBus[1]);

            // カットオフ 2048Hz のローパスフィルタを設定します
            nn::audio::BiquadFilterParameter firstFilter = { true,{ 720, 1439, 720 },{ 22684, -9350 } };
            nn::audio::SetVoiceBiquadFilterParameter(&voice, 0, firstFilter);
            // カットオフ 1024 Hz のハイパスフィルタを無効状態で設定します。
            nn::audio::BiquadFilterParameter secondFilter = { false,{ 14041, -28083, 14041 },{ 29547, -13563 } };
            nn::audio::SetVoiceBiquadFilterParameter(&voice, 1, secondFilter);
        }

        m_AudioQueue.clear();

        // タイマーイベントを初期化する
        nn::os::InitializeTimerEvent(&UpdateEvemt, nn::os::EventClearMode_AutoClear);
        // スレッドを生成する
        NN_ASSERT(nn::os::CreateThread(&m_UpdateThread, UpdateThread, &UpdateEvemt, ThreadStack, AudioManagerThreadStackSize, nn::os::DefaultThreadPriority).IsSuccess(), "Cannot create thread.");
        // スレッドの実行を開始する
        nn::os::StartThread(&m_UpdateThread);
    } // NOLINT(impl/function_size)

    void AudioManager::Finalize() NN_NOEXCEPT
    {
    }

    void AudioManager::UpdateThread(void *arg) NN_NOEXCEPT
    {
        NN_UNUSED(arg);
        while (true)
        {
            nn::os::WaitTimerEvent(&UpdateEvemt);
            AudioManager::GetInstance().UpdateCurrentAudio();
        }
    }

    void AudioManager::UpdateCurrentAudio() NN_NOEXCEPT
    {
        bool isNeedUpdate = nn::audio::GetAudioRendererState(m_RendererHandle) == nn::audio::AudioRendererState_Started;
        if (isNeedUpdate)
        {
            RunAudioQueue();
            auto result = nn::audio::RequestUpdateAudioRenderer(m_RendererHandle, &m_RendererConfig);
            NN_ABORT_UNLESS(result.IsSuccess());
        }
    }

    void AudioManager::StartUpdateThread() NN_NOEXCEPT
    {
        // レンダリングを開始します。
        nn::os::StartPeriodicTimerEvent(&UpdateEvemt, UpdateInterval, UpdateInterval);
    }

    void AudioManager::StopUpdateThread() NN_NOEXCEPT
    {
        // レンダリングを停止します
        nn::os::StopTimerEvent(&UpdateEvemt);
    }


    void AudioManager::RunAudioQueue() NN_NOEXCEPT
    {
        for (
            std::vector<AudioQueue>::iterator it = m_AudioQueue.begin();
            it != m_AudioQueue.end();
            )
        {
            auto& queue = (*it);
            auto fileItr = m_WaveFileList.find(queue.fileName.c_str());

            if (queue.delay < UpdateInterval)
            {
                if (fileItr != m_WaveFileList.end())
                {
                    for (auto& voice : m_Voices)
                    {
                        bool isReleasedBuffer = false;
                        while (nn::audio::GetReleasedWaveBuffer(&voice) != nullptr) { isReleasedBuffer = true; }
                        if (nn::audio::GetWaveBufferCount(&voice) == 0)
                        {
                            fileItr->second.ClearWaveBuffer();
                            auto buffer = fileItr->second.GetWavBuffer();
                            NN_ASSERT_NOT_NULL(buffer);

                            bool rt = nn::audio::AppendWaveBuffer(&voice, buffer);

                            auto volume = nn::audio::VoiceType::GetVolumeMin() + queue.volume * nn::audio::VoiceType::GetVolumeMax();

                            nn::audio::SetVoiceVolume(&voice, volume);

                            auto leftBalance = std::min(1.f, (1.f - queue.balance) * 2.f);
                            auto rightBalance = std::min(1.f, queue.balance * 2.f);

                            nn::audio::SetVoiceMixVolume(&voice, &m_FinalMix, volume * leftBalance, 0, m_MainBus[0]);
                            nn::audio::SetVoiceMixVolume(&voice, &m_FinalMix, volume * rightBalance, 0, m_MainBus[1]);

                            NN_ABORT_UNLESS(rt);
                            break;
                        }
                    }
                }
                it = m_AudioQueue.erase(it);
            }
            else
            {
                queue.delay -= UpdateInterval;
                ++it;
            }
        }
    }

    void AudioManager::PlayWav(const char* fileName) NN_NOEXCEPT
    {
        PlayWav(fileName, nn::TimeSpan::FromMilliSeconds(0), 1.f, 0.5f);
    }

    void AudioManager::PlayWav(const char* fileName, float volume) NN_NOEXCEPT
    {
        PlayWav(fileName, nn::TimeSpan::FromMilliSeconds(0), volume, 0.5f);
    }

    void AudioManager::PlayWav(const char* fileName, float volume, float balance) NN_NOEXCEPT
    {
        PlayWav(fileName, nn::TimeSpan::FromMilliSeconds(0), volume, balance);
    }

    void AudioManager::PlayWav(const char* fileName, const nn::TimeSpan& delay) NN_NOEXCEPT
    {
        PlayWav(fileName, delay, 1.f, 0.5f);
    }

    void AudioManager::PlayWav(const char* fileName, const nn::TimeSpan& delay, float volume) NN_NOEXCEPT
    {
        PlayWav(fileName, delay, volume, 0.5f);
    }

    void AudioManager::PlayWav(const char* fileName, const nn::TimeSpan& delay, float volume, float balance) NN_NOEXCEPT
    {
        auto itr = m_WaveFileList.find(fileName);
        if (itr != m_WaveFileList.end())
        {
            AudioQueue queue;
            queue.delay = delay;
            queue.fileName = fileName;
            queue.volume = std::max(0.f, std::min(LimitVolume, volume * LimitVolume));
            queue.balance = std::max(0.f, std::min(1.f, balance));

            m_AudioQueue.push_back(queue);
        }
    }
}
