﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "AssertDialog.h"

namespace VibrationCollection
{
    nns::hidfw::layout::SeekBar     g_SettingSeekBar[SettingButtonType_Num];
    nns::hidfw::layout::ButtonGroup g_DialogSeekBar[2];
    nns::hidfw::layout::ButtonGroup g_DialogButtons;
    nns::hidfw::layout::ButtonGroup g_CloseButton;
    nns::hidfw::layout::ButtonGroup g_LoopSettingButtons;

    nns::hidfw::layout::ButtonGroup g_PreviewVibrationViewer;

    bool g_IsLoopStartEdit = false;
    bool g_IsLoopEndEdit = false;

    void ShowSetting(void* pushButton, void* param) NN_NOEXCEPT
    {
        NN_UNUSED(pushButton);
        NN_UNUSED(param);

        if (gFileManager.GetBnvibFileCount() == 0)
        {
            if (ShowAssertDialog("error", "file is not loaded yet\n", AssertDialogType_Warning, AssertButtonType_OK))
            {
                return;
            }
        }
        else
        {
            nns::hidfw::layout::Dialog dialog;
            nns::hidfw::layout::Dialog::DialogInfo info;

            if (g_CloseButton.GetListSize() == 0)
            {
                nns::hidfw::layout::Button button;
                button.SetDefault();
                button.Setup(995, 72, 96, 40, nullptr, "(+) Close");
                button.SetMainColor(nn::util::Color4u8(232, 57, 54, 255));
                button.SetTextColor(nn::util::Color4u8::White());
                button.OnOffMode(false);
                button.SetEffectColor(nn::util::Color4u8(255, 255, 255, 0));
                button.SetFunc(CloseButtonFunc);
                g_CloseButton.Add(new nns::hidfw::layout::Button(button));
                g_CloseButton.Focus(false);
                g_CloseButton.SetOrientation(nns::hidfw::layout::ButtonGroup::eOrientation_Horizontal);
            }
            g_CloseButton.GetItemSet().at(0)->SetParam(&dialog);

            if (g_LoopSettingButtons.GetListSize() == 0)
            {
                InitializeVibrationLoopSettingButtons();
                g_LoopSettingButtons.Select(2);
            }
            if (g_DialogSeekBar[0].GetListSize() == 0)
            {
                InitializeVibrationSettingSeekBar(g_DialogSeekBar);
                g_DialogSeekBar[0].FocusItem(0);
                g_DialogSeekBar[1].FocusItem(0);
                g_DialogSeekBar[0].Focus(false);
                g_DialogSeekBar[1].Focus(false);
            }
            if (g_DialogButtons.GetListSize() == 0)
            {
                InitializeVibrationSettingButtons(&g_DialogButtons);
                g_DialogButtons.FocusItem(1);
                g_DialogButtons.Focus(true);
                g_DialogButtons.GetItemSet().at(0)->SetParam(&dialog);
            }
            if (g_PreviewVibrationViewer.GetListSize() == 0)
            {
                InitializePreviewVibrationValueViewer();
            }

            auto fileInfo = gFileManager.GetBnvibFile().at(g_SelectedBnvibButtonInedx).Loop;

            g_VibrationPlayer.SetLoopStartPosition(fileInfo.loopStartPosition);
            g_VibrationPlayer.SetLoopEndPosition(fileInfo.loopEndPosition);

            static_cast<nns::hidfw::layout::SeekBar*>(g_LoopSettingButtons.GetItemSet().at(0))->SetValue(fileInfo.loopInterval);
            static_cast<nns::hidfw::layout::SeekBar*>(g_LoopSettingButtons.GetItemSet().at(0))->CallFunc();
            g_LoopSettingButtons.SelectItem(fileInfo.isLoop != 0 ? 1 : 2, true);

            g_CloseButton.SetPrevItemSet(&g_DialogButtons);
            g_CloseButton.SetNextItemSet(&g_PreviewVibrationViewer);
            g_PreviewVibrationViewer.SetPrevItemSet(&g_CloseButton);
            g_PreviewVibrationViewer.SetNextItemSet(&g_LoopSettingButtons);
            g_LoopSettingButtons.SetPrevItemSet(&g_PreviewVibrationViewer);
            g_LoopSettingButtons.SetNextItemSet(&g_DialogSeekBar[0]);
            g_DialogSeekBar[0].SetPrevItemSet(&g_LoopSettingButtons);
            g_DialogSeekBar[0].SetNextItemSet(&g_DialogSeekBar[1]);
            g_DialogSeekBar[1].SetPrevItemSet(&g_DialogSeekBar[0]);
            g_DialogSeekBar[1].SetNextItemSet(&g_DialogButtons);
            g_DialogButtons.SetPrevItemSet(&g_DialogSeekBar[1]);
            g_DialogButtons.SetNextItemSet(&g_CloseButton);

            info.Size = nn::util::MakeFloat2(950.f, 600.f);
            info.Title = "Vibration Setting";
            info.UpdateFunc = UpdateVibrationSetting;
            info.DrawerFunc = DrawVibrationSetting;

            auto loopPos = std::max(0, std::min(g_VibrationPlayer.GetFileInfo().sampleLength, g_VibrationPlayer.GetLoopStartPosition()));
            g_VibrationPlayer.SetLoopStartPosition(loopPos);
            loopPos = std::max(0, std::min(g_VibrationPlayer.GetFileInfo().sampleLength, g_VibrationPlayer.GetLoopEndPosition()));
            g_VibrationPlayer.SetLoopEndPosition(loopPos);

            CalcPreviewVibrationValueViewer();
            dialog.ShowDialog(info);
        }
    }

    void InitializePreviewVibrationValueViewer() NN_NOEXCEPT
    {
        nns::hidfw::layout::LineChart lineChart;
        lineChart.SetDefault();
        lineChart.AddElement(nns::hidfw::layout::LineChart::CreateElement(nn::util::Color4u8( 16,  96, 242, 255), "Original Gain-Low", 200, 1.f, 0.f, 2.f));
        lineChart.AddElement(nns::hidfw::layout::LineChart::CreateElement(nn::util::Color4u8(255,   0,   2, 255), "Original Gain-High", 200, 1.f, 0.f, 2.f));
        lineChart.AddElement(nns::hidfw::layout::LineChart::CreateElement(nn::util::Color4u8( 16, 255,  96, 255), "Edit Gain-Low", 200, 1.f, 0.f, 2.f));
        lineChart.AddElement(nns::hidfw::layout::LineChart::CreateElement(nn::util::Color4u8(255, 128,   0, 255), "Edit Gain-High", 200, 1.f, 0.f, 2.f));
        lineChart.SetChooseMode(nns::hidfw::layout::BaseItem::ChoiceMode::Controller::Mask);
        lineChart.SetPos(190, 140);
        lineChart.SetSize(900, 188);
        lineChart.SetMainColor(nn::util::Color4u8(42, 66, 77, 255));
        lineChart.SetBorderColor(nn::util::Color4u8(0, 32, 0, 255));
        lineChart.ThroughCancel(true);
        g_PreviewVibrationViewer.Add(new nns::hidfw::layout::LineChart(lineChart));
    }

    void CalcPreviewVibrationValueViewer() NN_NOEXCEPT
    {
        if (gFileManager.GetBnvibFileCount() == 0)
        {
            return;
        }

        auto chart = static_cast<nns::hidfw::layout::LineChart*>(g_PreviewVibrationViewer.GetItemSet().at(0));

        nn::hid::VibrationFileInfo outInfo;
        nn::hid::VibrationFileParserContext outContext;
        nn::hid::ParseVibrationFile(&outInfo, &outContext, gFileManager.GetBnvibFile().at(g_SelectedBnvibButtonInedx).FileData, gFileManager.GetBnvibFile().at(g_SelectedBnvibButtonInedx).FileSize);
        nn::hid::VibrationValue value;

        static uint8_t vibrationBuffer[BnvibFileData::FileSizeMax];
        size_t outSize = 0;
        ExportVibrationValue(vibrationBuffer, BnvibFileData::FileSizeMax, &outSize);

        nn::hid::VibrationFileInfo editOutInfo;
        nn::hid::VibrationFileParserContext editOutContext;
        nn::hid::ParseVibrationFile(&editOutInfo, &editOutContext, vibrationBuffer, outSize);

        for (int i = 0; i < 200; ++i)
        {
            value = nn::hid::VibrationValue::Make();
            if (i < outInfo.sampleLength)
            {
                nn::hid::RetrieveVibrationValue(&value, i, &outContext);
            }
            chart->AddValue(0, value.amplitudeLow);
            chart->AddValue(1, value.amplitudeHigh);
            value = nn::hid::VibrationValue::Make();
            if (i < editOutInfo.sampleLength)
            {
                nn::hid::RetrieveVibrationValue(&value, i, &editOutContext);
            }
            chart->AddValue(2, value.amplitudeLow);
            chart->AddValue(3, value.amplitudeHigh);
        }
    }

    void ResetVibrationSetting(void* pushButton, void* param) NN_NOEXCEPT
    {
        NN_UNUSED(pushButton);
        NN_ASSERT_NOT_NULL(param);

        nns::hidfw::layout::ButtonGroup* pSeekBarSet = (nns::hidfw::layout::ButtonGroup*)param;
        NN_ASSERT_EQUAL(pSeekBarSet[0].GetListSize(), SettingButtonType_Num / 2);
        NN_ASSERT_EQUAL(pSeekBarSet[1].GetListSize(), SettingButtonType_Num / 2);

        for (size_t i = 0; i < SettingButtonType_Num; ++i)
        {
            reinterpret_cast<nns::hidfw::layout::SeekBar*>(pSeekBarSet[i / 3].GetItemSet().at(i % 3))->SetValue((SettingButtonType)i == SettingButtonType_Offset ? 0 : 100);
            reinterpret_cast<nns::hidfw::layout::SeekBar*>(pSeekBarSet[i / 3].GetItemSet().at(i % 3))->CallFunc();
        }

        auto fileInfo = g_VibrationPlayer.GetFileInfo();

        g_VibrationPlayer.SetLoopStartPosition(fileInfo.loopStartPosition);
        g_VibrationPlayer.SetLoopEndPosition(fileInfo.loopEndPosition);

        static_cast<nns::hidfw::layout::SeekBar*>(g_LoopSettingButtons.GetItemSet().at(0))->SetValue(fileInfo.loopInterval);
        static_cast<nns::hidfw::layout::SeekBar*>(g_LoopSettingButtons.GetItemSet().at(0))->CallFunc();
        g_LoopSettingButtons.SelectItem(fileInfo.isLoop != 0 ? 1 : 2, true);

        CalcPreviewVibrationValueViewer();
    }

    void ChangeHighPitch(void* pushButton, void* param) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pushButton);
        NN_ASSERT_NOT_NULL(param);

        nns::hidfw::layout::SeekBar* pSeekBar = (nns::hidfw::layout::SeekBar*) pushButton;
        VibrationTarget* pVibrationPlayer = (VibrationTarget*)param;
        nn::hid::VibrationModulation modulation;
        float value = static_cast<float>(pSeekBar->GetValue()) / 100.f;
        for (size_t i = 0; i < VibrationTarget::MaxTargetCount; ++i)
        {
            modulation = pVibrationPlayer->Connection[i].GetModulation();
            modulation.pitchHigh = value;
            pVibrationPlayer->Connection[i].SetModulation(modulation);
        }
        CalcPreviewVibrationValueViewer();
    }

    void ChangeHighGain(void* pushButton, void* param) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pushButton);
        NN_ASSERT_NOT_NULL(param);

        nns::hidfw::layout::SeekBar* pSeekBar = (nns::hidfw::layout::SeekBar*) pushButton;
        VibrationTarget* pVibrationPlayer = (VibrationTarget*)param;
        nn::hid::VibrationModulation modulation;
        float value = static_cast<float>(pSeekBar->GetValue()) / 100.f;
        for (size_t i = 0; i < VibrationTarget::MaxTargetCount; ++i)
        {
            modulation = pVibrationPlayer->Connection[i].GetModulation();
            modulation.gainHigh = value;
            pVibrationPlayer->Connection[i].SetModulation(modulation);
        }
        CalcPreviewVibrationValueViewer();
    }

    void ChangeLowPitch(void* pushButton, void* param) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pushButton);
        NN_ASSERT_NOT_NULL(param);

        nns::hidfw::layout::SeekBar* pSeekBar = (nns::hidfw::layout::SeekBar*) pushButton;
        VibrationTarget* pVibrationPlayer = (VibrationTarget*)param;
        nn::hid::VibrationModulation modulation;
        float value = static_cast<float>(pSeekBar->GetValue()) / 100.f;
        for (size_t i = 0; i < VibrationTarget::MaxTargetCount; ++i)
        {
            modulation = pVibrationPlayer->Connection[i].GetModulation();
            modulation.pitchLow = value;
            pVibrationPlayer->Connection[i].SetModulation(modulation);
        }
        CalcPreviewVibrationValueViewer();
    }

    void ChangeLowGain(void* pushButton, void* param) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pushButton);
        NN_ASSERT_NOT_NULL(param);

        nns::hidfw::layout::SeekBar* pSeekBar = (nns::hidfw::layout::SeekBar*) pushButton;
        VibrationTarget* pVibrationPlayer = (VibrationTarget*)param;
        nn::hid::VibrationModulation modulation;
        float value = static_cast<float>(pSeekBar->GetValue()) / 100.f;
        for (size_t i = 0; i < VibrationTarget::MaxTargetCount; ++i)
        {
            modulation = pVibrationPlayer->Connection[i].GetModulation();
            modulation.gainLow = value;
            pVibrationPlayer->Connection[i].SetModulation(modulation);
        }
        CalcPreviewVibrationValueViewer();
    }

    void ChangeSpeed(void* pushButton, void* param) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pushButton);
        NN_UNUSED(param);

        nns::hidfw::layout::SeekBar* pSeekBar = (nns::hidfw::layout::SeekBar*) pushButton;
        if (pSeekBar->GetValue() == 0)
        {
            pSeekBar->AddValue(1);
        }
        float value = static_cast<float>(pSeekBar->GetValue()) / 100.f;
        for (size_t i = 0; i < VibrationTarget::MaxTargetCount; ++i)
        {
            g_VibrationPlayer.SetPlaySpeed(value);
        }
        CalcPreviewVibrationValueViewer();
    }

    void ChangeOffset(void* pushButton, void* param) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pushButton);
        NN_UNUSED(param);

        nns::hidfw::layout::SeekBar* pSeekBar = (nns::hidfw::layout::SeekBar*) pushButton;
        auto value = static_cast<float>(pSeekBar->GetValue()) * 5;
        for (size_t i = 0; i < VibrationTarget::MaxTargetCount; ++i)
        {
            g_VibrationPlayer.SetOffset(nn::TimeSpan::FromMilliSeconds(value));
        }
        CalcPreviewVibrationValueViewer();
    }

    void ChangeLoopFlag(void* pushButton, void* param)
    {
        NN_ASSERT_NOT_NULL(pushButton);
        NN_UNUSED(param);

        auto& loopSetting = gFileManager.GetBnvibFile().at(g_SelectedBnvibButtonInedx).Loop;
        loopSetting.isLoop = (uint64_t)pushButton == (uint64_t)g_LoopSettingButtons.GetItemSet().at(1) ? 1 : 0;

        g_VibrationPlayer.SetLoop(loopSetting.isLoop != 0);
    }

    void ChangeLoopInterval(void* pushButton, void* param)
    {
        NN_ASSERT_NOT_NULL(pushButton);
        NN_UNUSED(param);

        auto seekBar = (nns::hidfw::layout::SeekBar*)pushButton;
        gFileManager.GetBnvibFile().at(g_SelectedBnvibButtonInedx).Loop.loopInterval = seekBar->GetValue();

        g_VibrationPlayer.SetLoopInterval(seekBar->GetValue());
    }

    void InitializeVibrationSettingSeekBar(nns::hidfw::layout::ButtonGroup* pButtonGroup) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pButtonGroup);

        nn::util::Float2 basePos = NN_UTIL_FLOAT_2_INITIALIZER(215, 440);
        nn::util::Float2 margin = NN_UTIL_FLOAT_2_INITIALIZER(300, 96);

        for (int i = 0; i < SettingButtonType_Num; ++i)
        {
            g_SettingSeekBar[i].SetDefault();
            g_SettingSeekBar[i].SetSize(250, 20);
            g_SettingSeekBar[i].SetValue(100);
            g_SettingSeekBar[i].SetMaxValue(500);
            g_SettingSeekBar[i].EnableAutoShowValue(false);
            g_SettingSeekBar[i].SetMainColor(nn::util::Color4u8(245, 226, 169));
            g_SettingSeekBar[i].SetSubColor(nn::util::Color4u8(245, 226, 169));
            g_SettingSeekBar[i].SetParam(&g_VibrationTarget);

            g_SettingSeekBar[i].SetPos(
                basePos.x + margin.x * (i % 3),
                basePos.y + margin.y * (i / 3)
            );
        }

        g_SettingSeekBar[SettingButtonType_GainLow].SetFunc(ChangeLowGain);
        g_SettingSeekBar[SettingButtonType_PitchLow].SetFunc(ChangeLowPitch);
        g_SettingSeekBar[SettingButtonType_Speed].SetFunc(ChangeSpeed);
        g_SettingSeekBar[SettingButtonType_GainHigh].SetFunc(ChangeHighGain);
        g_SettingSeekBar[SettingButtonType_PitchHigh].SetFunc(ChangeHighPitch);
        g_SettingSeekBar[SettingButtonType_Offset].SetFunc(ChangeOffset);
        g_SettingSeekBar[SettingButtonType_Offset].SetValue(0);

        for (int i = 0; i < SettingButtonType_Num; ++i)
        {
            pButtonGroup[i / 3].Add(new nns::hidfw::layout::SeekBar(g_SettingSeekBar[i]));
            pButtonGroup[i / 3].SetOrientation(nns::hidfw::layout::ButtonGroup::eOrientation_Horizontal);
        }
    }

    void InitializeVibrationSettingButtons(nns::hidfw::layout::ButtonGroup* pButtonGroup) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pButtonGroup);

        const char* buttonText[] =
        {
            "Export",
            "(X) Play Vibration",
            "(Y) Reset"
        };

        nns::hidfw::layout::LayoutFunction pFunc[] =
        {
            ShowExportBnvibDialog,
            PlayVibration,
            ResetVibrationSetting
        };

        nn::util::Color4u8 buttonColor[] =
        {
            nn::util::Color4u8(192, 188, 64, 255),
            nn::util::Color4u8(49, 119, 125, 255),
            nn::util::Color4u8(232, 57, 54, 255),
        };

        nn::util::Float2 pos[] =
        {
            nn::util::MakeFloat2(178, 600),
            nn::util::MakeFloat2(491, 600),
            nn::util::MakeFloat2(804, 600)
        };

        nn::util::Float2 size[] =
        {
            nn::util::MakeFloat2(300, 48),
            nn::util::MakeFloat2(300, 48),
            nn::util::MakeFloat2(300, 48)
        };

        // 標準的なボタンの設定で作成します
        for (size_t i = 0; i < NN_ARRAY_SIZE(buttonText); ++i)
        {
            nns::hidfw::layout::Button button;
            button.SetPos(pos[i].x, pos[i].y);
            button.SetSize(size[i].x, size[i].y);
            button.SetText(buttonText[i]);
            button.SetFunc(pFunc[i]);
            button.SetTextColor(nn::util::Color4u8::White());
            button.ThroughCancel(true);                             // キャンセルを無効化します
            button.SetMainColor(buttonColor[i]);
            button.OnOffMode(false);
            button.SetParam(g_DialogSeekBar);
            pButtonGroup->Add(new nns::hidfw::layout::Button(button));
        }
        pButtonGroup->SetOrientation(nns::hidfw::layout::ButtonGroup::eOrientation_Horizontal);
    }

    void InitializeVibrationLoopSettingButtons() NN_NOEXCEPT
    {
        const char* buttonText[] =
        {
            "Enable",
            "Disable"
        };

        nns::hidfw::layout::LayoutFunction pFunc[] =
        {
            ChangeLoopFlag,
            ChangeLoopFlag
        };

        nn::util::Color4u8 buttonColor[] =
        {
            nn::util::Color4u8(49, 119, 125, 255),
            nn::util::Color4u8(232, 57, 54, 255),
        };

        nn::util::Float2 pos[] =
        {
            nn::util::MakeFloat2(1104 - 228, 340),
            nn::util::MakeFloat2(1104 - 108, 340)
        };

        nn::util::Float2 size[] =
        {
            nn::util::MakeFloat2(96, 32),
            nn::util::MakeFloat2(96, 32)
        };

        nns::hidfw::layout::SeekBar seekBar;
        {
            seekBar.SetDefault();
            seekBar.SetSize(200, 16);
            seekBar.SetValue(0);
            seekBar.SetMaxValue(200);
            seekBar.EnableAutoShowValue(false);
            seekBar.SetMainColor(nn::util::Color4u8(245, 226, 169));
            seekBar.SetSubColor(nn::util::Color4u8(245, 226, 169));
            seekBar.SetParam(&gFileManager.GetBnvibFile().at(g_SelectedBnvibButtonInedx).Loop);
            seekBar.SetFunc(ChangeLoopInterval);
            seekBar.SetPos(pos[0].x - 560, pos[0].y + 8.f);
            seekBar.MultiMode(true);
        }
        g_LoopSettingButtons.Add(new nns::hidfw::layout::SeekBar(seekBar));

        // 標準的なボタンの設定で作成します
        for (size_t i = 0; i < NN_ARRAY_SIZE(buttonText); ++i)
        {
            nns::hidfw::layout::Button button;
            button.SetPos(pos[i].x, pos[i].y);
            button.SetSize(size[i].x, size[i].y);
            button.SetText(buttonText[i]);
            button.SetFunc(pFunc[i]);
            button.SetTextColor(nn::util::Color4u8::White());
            button.ThroughCancel(true);                             // キャンセルを無効化します
            button.SetMainColor(buttonColor[i]);
            button.OnOffMode(true);
            button.SetParam(nullptr);
            button.MultiMode(false);
            g_LoopSettingButtons.Add(new nns::hidfw::layout::Button(button));
        }
        g_LoopSettingButtons.SetOrientation(nns::hidfw::layout::ButtonGroup::eOrientation_Horizontal);
    }

    void UpdateVibrationSetting(void* executant, void* param) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(executant);
        NN_UNUSED(param);

        static const char* SeekBarNames[SettingButtonType_Num] =
        {
            "Gain-Low", "Pitch-Low", "Speed",
            "Gain-High", "Pitch-High", "Offset"
        };

        nns::hidfw::layout::Dialog* pDialog = (nns::hidfw::layout::Dialog*)executant;
        const nn::util::Float2 dialogSize = pDialog->GetDialogInfo().Size;

        g_CloseButton.Update();
        g_LoopSettingButtons.Update();
        g_DialogSeekBar[0].Update();
        g_DialogSeekBar[1].Update();

        g_DialogButtons.Update();

        g_PreviewVibrationViewer.Update();

        for (
            std::vector<nns::hidfw::hid::Controller*>::iterator it = gController.GetConnectedControllerList().begin();
            it != gController.GetConnectedControllerList().end();
            ++it
            )
        {
            if ((*it)->IsTrigger(nn::hid::NpadButton::X::Mask))
            {
                PlayVibration(nullptr, nullptr);
                g_DialogButtons.SelectItem(1, false);
                break;
            }
            else if ((*it)->IsTrigger(nn::hid::NpadButton::Y::Mask))
            {
                g_DialogButtons.SelectItem(2, true);
                break;
            }
        }
        gTextWriter.SetTextColor(nn::util::Color4u8::Blue());
        gTextWriter.SetScale(1.4f, 1.4f);
        gTextWriter.SetCursorX((1280.f - gTextWriter.CalculateStringWidth(pDialog->GetDialogInfo().Title.c_str())) / 2.f);
        gTextWriter.SetCursorY((720.f - dialogSize.y) / 2.f + 16.f);
        gTextWriter.Print(pDialog->GetDialogInfo().Title.c_str());
        gTextWriter.SetScale(1.f, 1.f);

        for (size_t i = 0; i < NN_ARRAY_SIZE(SeekBarNames); ++i)
        {
            size_t index = i;
            nns::hidfw::layout::SeekBar* pSeekBar = reinterpret_cast<nns::hidfw::layout::SeekBar*>(g_DialogSeekBar[i / 3].GetItemSet().at(i % 3));
            gTextWriter.SetTextColor((
                g_DialogSeekBar[i / 3].IsFocus() && pSeekBar->IsSelected() && g_DialogSeekBar[i / 3].GetFocusItemIndex() == (i % 3)) ?
                nn::util::Color4u8::White(): nn::util::Color4u8::Blue());
            gTextWriter.SetCursorX(pSeekBar->GetPos().x + 16.f);
            gTextWriter.SetCursorY(pSeekBar->GetPos().y - (gTextWriter.GetLineHeight()));
            gTextWriter.Print("%s", SeekBarNames[index]);
            gTextWriter.SetCursorX(pSeekBar->GetPos().x + pSeekBar->GetSize().x - 96.f);
            if (i == NN_ARRAY_SIZE(SeekBarNames) - 1)
            {
                gTextWriter.Print("| %04.3f s", static_cast<float>(pSeekBar->GetValue()) / 200.f);
            }
            else
            {
                gTextWriter.Print("|  %03.2f", static_cast<float>(pSeekBar->GetValue()) / 100.f);
            }
        }

        auto chart = static_cast<nns::hidfw::layout::LineChart*>(g_PreviewVibrationViewer.GetItemSet().at(0));
        gTextWriter.SetCursor(chart->GetPos().x + chart->GetSize().x - 640.f, chart->GetPos().y);
        gTextWriter.SetTextColor(chart->GetElements().at(0).Color);
        gTextWriter.Print("- %s", chart->GetElements().at(0).Name.c_str());
        gTextWriter.SetCursorX(gTextWriter.GetCursorX() + 16.f);
        gTextWriter.SetTextColor(chart->GetElements().at(1).Color);
        gTextWriter.Print("- %s", chart->GetElements().at(1).Name.c_str());
        gTextWriter.SetCursorX(gTextWriter.GetCursorX() + 16.f);
        gTextWriter.SetTextColor(chart->GetElements().at(2).Color);
        gTextWriter.Print("- %s", chart->GetElements().at(2).Name.c_str());
        gTextWriter.SetCursorX(gTextWriter.GetCursorX() + 16.f);
        gTextWriter.SetTextColor(chart->GetElements().at(3).Color);
        gTextWriter.Print("- %s", chart->GetElements().at(3).Name.c_str());

        // ループ再生用
        gTextWriter.SetTextColor(nn::util::Color4u8::Blue());
        gTextWriter.SetCursor(chart->GetPos().x, chart->GetPos().y + chart->GetSize().y + 16.f);
        gTextWriter.Print("LoopInterval");
        gTextWriter.SetCursorX(g_LoopSettingButtons.GetItemSet().at(0)->GetPos().x + g_LoopSettingButtons.GetItemSet().at(0)->GetSize().x + 12.f);
        gTextWriter.Print("%llu Sample",static_cast<nns::hidfw::layout::SeekBar*>(g_LoopSettingButtons.GetItemSet().at(0))->GetValue());
        gTextWriter.SetCursorX(g_LoopSettingButtons.GetItemSet().at(1)->GetPos().x - 112.f);
        gTextWriter.Print("| LoopFlag");

        auto pos = g_PreviewVibrationViewer.GetItemSet().at(0)->GetPos();
        auto size = g_PreviewVibrationViewer.GetItemSet().at(0)->GetSize();
        auto offset = 4.5f * static_cast<float>((float)g_VibrationPlayer.GetOffset().GetMilliSeconds() / nn::hid::VibrationNode::DefaultVibrationSampleInterval.GetMilliSeconds());
        auto startLineX = pos.x + std::min(static_cast<float>(size.x), offset + 4.5f * (g_VibrationPlayer.GetLoopStartPosition() / g_VibrationPlayer.GetPlaySpeed()));
        auto endLineX = pos.x + std::min(static_cast<float> (size.x), offset + 4.5f * (g_VibrationPlayer.GetLoopEndPosition() / g_VibrationPlayer.GetPlaySpeed()));

        bool isStartEdit = false;
        bool isTouch = false;

        // 画面全体の何処かをタッチしていた場合、ループ地点の編集フラグを立てます
        if (gTouch.IsEnable(nn::util::MakeFloat2(0, 0), nn::util::MakeFloat2(1280, 720)) && g_VibrationPlayer.IsLoop())
        {
            // ステータスを取得します。
            nn::hid::TouchState startState, endState;
            isTouch = gTouch.GetTouchState(&startState, nn::util::MakeFloat2(startLineX, pos.y + size.y / 2.f));
            isTouch = gTouch.GetTouchState(&endState, nn::util::MakeFloat2(endLineX, pos.y + size.y / 2.f));
            auto startDir = std::sqrt(std::pow(startLineX - startState.x, 2) + std::pow((pos.y + size.y / 2.f) - startState.y, 2));
            auto endDir = std::sqrt(std::pow(endLineX - endState.x, 2) + std::pow((pos.y + size.y / 2.f) - endState.y, 2));

            isTouch = isTouch && ((startDir < size.y / 2.f) || (endDir < size.y / 2.f));

            if ((g_IsLoopStartEdit || g_IsLoopEndEdit) && !isTouch && gTouch.IsEnable(nn::util::MakeFloat2(0, 0), nn::util::MakeFloat2(1280, 720)))
            {
                isTouch = gTouch.GetTouchState(&startState, nn::util::MakeFloat2(640, 360));
                endState = startState;
            }

            if (isTouch)
            {
                if (g_IsLoopStartEdit || g_IsLoopEndEdit)
                {
                    // 既に編集を開始している場合
                    isStartEdit = g_IsLoopStartEdit;
                }
                else
                {
                    // まだ編集を開始していない場合
                    // 2点がタッチ状態の場合、もっとも近いタッチ点をもつ方を優先します
                    isStartEdit = startDir < endDir;
                }
                g_IsLoopStartEdit = isStartEdit;
                g_IsLoopEndEdit = !isStartEdit;

                // 編集に採用するステータスを決定
                nn::hid::TouchState& state = isStartEdit ? startState : endState;
                auto loopPos = std::min(g_VibrationPlayer.GetFileInfo().sampleLength, static_cast<int>(std::max(0.f, (static_cast<float>(state.x - (pos.x + offset)) / 4.5f) * g_VibrationPlayer.GetPlaySpeed())));
                if (isStartEdit)
                {
                    gFileManager.GetBnvibFile().at(g_SelectedBnvibButtonInedx).Loop.loopStartPosition = loopPos;
                    g_VibrationPlayer.SetLoopStartPosition(loopPos);
                }
                else
                {
                    gFileManager.GetBnvibFile().at(g_SelectedBnvibButtonInedx).Loop.loopEndPosition = loopPos;
                    g_VibrationPlayer.SetLoopEndPosition(loopPos);
                }
            }
        }
        else
        {
            if (g_IsLoopStartEdit || g_IsLoopEndEdit)
            {
                // 画面のタッチが開放された場合編集を終了します
                g_IsLoopStartEdit = g_IsLoopEndEdit = false;
                auto startPos = g_VibrationPlayer.GetLoopStartPosition();
                // ループのスタート地点と終了地点がクロスしていた場合は入れ替えます
                if (startPos > g_VibrationPlayer.GetLoopEndPosition())
                {
                    g_VibrationPlayer.SetLoopStartPosition(g_VibrationPlayer.GetLoopEndPosition());
                    g_VibrationPlayer.SetLoopEndPosition(startPos);
                    gFileManager.GetBnvibFile().at(g_SelectedBnvibButtonInedx).Loop.loopStartPosition = g_VibrationPlayer.GetLoopStartPosition();
                    gFileManager.GetBnvibFile().at(g_SelectedBnvibButtonInedx).Loop.loopEndPosition = g_VibrationPlayer.GetLoopEndPosition();
                }
            }
            else
            {
                // LineChart にフォーカスがあっている場合はスティック操作でループ範囲を編集できるようにします
                if (g_PreviewVibrationViewer.IsFocus())
                {
                    auto& loopInfo = gFileManager.GetBnvibFile().at(g_SelectedBnvibButtonInedx).Loop;

                    for (auto itr : gController.GetConnectedControllerList())
                    {
                        if (itr->IsRepeat(nn::hid::NpadButton::StickLLeft::Mask) && (g_VibrationPlayer.GetLoopStartPosition() > 0))
                        {
                            loopInfo.loopStartPosition = g_VibrationPlayer.GetLoopStartPosition() - 1.f;
                            g_VibrationPlayer.SetLoopStartPosition(loopInfo.loopStartPosition);
                        }
                        else if (itr->IsRepeat(nn::hid::NpadButton::StickLRight::Mask) && (g_VibrationPlayer.GetLoopStartPosition() < g_VibrationPlayer.GetLoopEndPosition()))
                        {
                            loopInfo.loopStartPosition = g_VibrationPlayer.GetLoopStartPosition() + 1.f;
                            g_VibrationPlayer.SetLoopStartPosition(loopInfo.loopStartPosition);
                        }
                        if (itr->IsRepeat(nn::hid::NpadButton::StickRLeft::Mask) && g_VibrationPlayer.GetLoopEndPosition() > g_VibrationPlayer.GetLoopStartPosition())
                        {
                            loopInfo.loopEndPosition = g_VibrationPlayer.GetLoopEndPosition() > 0 ? g_VibrationPlayer.GetLoopEndPosition() - 1 : 0;
                            g_VibrationPlayer.SetLoopEndPosition(loopInfo.loopEndPosition);
                        }
                        else if (itr->IsRepeat(nn::hid::NpadButton::StickRRight::Mask) && g_VibrationPlayer.GetLoopEndPosition() < g_VibrationPlayer.GetFileInfo().sampleLength)
                        {
                            loopInfo.loopEndPosition = g_VibrationPlayer.GetLoopEndPosition() + 1;
                            g_VibrationPlayer.SetLoopEndPosition(loopInfo.loopEndPosition);
                        }

                        if (
                            itr->IsHold(nn::hid::NpadButton::StickLLeft::Mask) ||
                            itr->IsHold(nn::hid::NpadButton::StickLRight::Mask) ||
                            itr->IsHold(nn::hid::NpadButton::StickRLeft::Mask) ||
                            itr->IsHold(nn::hid::NpadButton::StickRRight::Mask)
                            )
                        {
                            break;
                        }
                    }
                }
            }
        }
    } // NOLINT(impl/function_size)

    void DrawVibrationSetting(void* executant, void* param) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(executant);
        NN_UNUSED(param);

        nn::util::Color4u8 frameColor = nn::util::Color4u8(24, 224, 80, 168);
        nn::util::Color4u8 hoverColor = nn::util::Color4u8(224, 224, 80, 168);
        nn::util::Color4u8 selectedColor = nn::util::Color4u8(4, 3, 2, 242);

        auto pos = g_PreviewVibrationViewer.GetItemSet().at(0)->GetPos();
        auto size = g_PreviewVibrationViewer.GetItemSet().at(0)->GetSize();
        auto offset = g_VibrationPlayer.GetOffset().GetMilliSeconds() / nn::hid::VibrationNode::DefaultVibrationSampleInterval.GetMilliSeconds();

        g_PreviewVibrationViewer.Draw();

        if (g_VibrationPlayer.IsPlaying())
        {
            float lineX = pos.x + 4.5 * g_VibrationPlayer.GetCurrentPosition() / g_VibrationPlayer.GetPlaySpeed();
            lineX += 4.5f * offset;
            lineX = std::max(pos.x, std::min(lineX, pos.x + g_PreviewVibrationViewer.GetItemSet().at(0)->GetSize().x));

            NN_ASSERT_GREATER(g_VibrationPlayer.GetPlaySpeed(), 0.001f);
            gDrawer.SetColor(hoverColor);
            gDrawer.Draw2DLine(
                nn::util::MakeFloat2(lineX, pos.y),
                nn::util::MakeFloat2(lineX, pos.y + g_PreviewVibrationViewer.GetItemSet().at(0)->GetSize().y));
        }
        if (g_VibrationPlayer.IsLoop())
        {
            nn::util::Float2 loopLinePos[] =
            {
                nn::util::MakeFloat2(pos.x + 4.5f * (g_VibrationPlayer.GetLoopStartPosition() / g_VibrationPlayer.GetPlaySpeed() + offset), pos.y),
                nn::util::MakeFloat2(pos.x + 4.5f * (g_VibrationPlayer.GetLoopStartPosition() / g_VibrationPlayer.GetPlaySpeed() + offset), pos.y + size.y),
                nn::util::MakeFloat2(pos.x + 4.5f * (g_VibrationPlayer.GetLoopEndPosition() / g_VibrationPlayer.GetPlaySpeed() + offset), pos.y),
                nn::util::MakeFloat2(pos.x + 4.5f * (g_VibrationPlayer.GetLoopEndPosition() / g_VibrationPlayer.GetPlaySpeed() + offset), pos.y + size.y)
            };

            for (auto& linePos : loopLinePos)
            {
                linePos.x = std::min(pos.x + size.x, linePos.x);
            }

            nn::util::Float2 loopRange[] =
            {
                nn::util::MakeFloat2(loopLinePos[0].x, pos.y),
                nn::util::MakeFloat2(loopLinePos[2].x - loopLinePos[0].x, size.y),
            };

            gDrawer.SetColor(nn::util::Color4u8(255, 255, 255, 48));
            gDrawer.Draw2DRect(loopRange[0], loopRange[1]);
            gDrawer.SetColor(g_IsLoopStartEdit ? nn::util::Color4u8::Yellow() : nn::util::Color4u8::White());
            gDrawer.Draw2DLine(loopLinePos[0], loopLinePos[1]);
            gDrawer.Draw2DTriangle(loopLinePos[0], 8.f, 180.f);
            gDrawer.Draw2DTriangle(loopLinePos[1], 8.f, 0.f);
            if (g_PreviewVibrationViewer.IsFocus())
            {
                auto drawPos = nn::util::MakeFloat2(loopLinePos[0].x, (loopLinePos[0].y + loopLinePos[1].y) / 2.f);
                gDrawer.Draw2DCircle(drawPos, 12.f, 16, 0);
                gDrawer.SetColor(nn::util::Color4u8(49, 119, 125, 255));
                DrawNpadButton(
                    nn::util::MakeFloat2(drawPos.x - 16.f, drawPos.y - 16),
                    nn::util::MakeFloat2(0.5f, 0.5f),
                    nn::hid::NpadButton::StickL::Mask,
                    ButtonIconType_Fill
                );
            }
            gDrawer.SetColor(g_IsLoopEndEdit ? nn::util::Color4u8::Yellow() : nn::util::Color4u8::White());
            gDrawer.Draw2DLine(loopLinePos[2], loopLinePos[3]);
            gDrawer.Draw2DTriangle(loopLinePos[2], 8.f, 180.f);
            gDrawer.Draw2DTriangle(loopLinePos[3], 8.f, 0.f);
            if (g_PreviewVibrationViewer.IsFocus())
            {
                auto drawPos = nn::util::MakeFloat2(loopLinePos[2].x, (loopLinePos[2].y + loopLinePos[3].y) / 2.f);
                gDrawer.Draw2DCircle(drawPos, 12.f, 16, 0);
                gDrawer.SetColor(nn::util::Color4u8(232, 57, 54, 255));
                DrawNpadButton(
                    nn::util::MakeFloat2(drawPos.x - 16.f, drawPos.y - 16),
                    nn::util::MakeFloat2(0.5f, 0.5f),
                    nn::hid::NpadButton::StickR::Mask,
                    ButtonIconType_Fill
                );
            }
        }

        // 閉じるボタンの描画
        g_CloseButton.Draw();

        // ループの設定ボタンの描画
        if (
            g_LoopSettingButtons.IsFocus() &&
            g_LoopSettingButtons.GetFocusItemIndex() == 0
            )
        {
            auto itemPos = g_LoopSettingButtons.GetItemSet().at(0)->GetPos();
            auto itemSize = g_LoopSettingButtons.GetItemSet().at(0)->GetSize();

            gDrawer.SetColor(g_LoopSettingButtons.GetItemSet().at(0)->IsSelected() ? selectedColor : hoverColor);
            gDrawer.Draw2DRoundedRect(
                nn::util::MakeFloat2(itemPos.x - 4.f, itemPos.y + 12.f),
                nn::util::MakeFloat2(itemSize.x + 8.f, itemSize.y - 24.f),
                0.25f, 32.f);

            gDrawer.SetColor(frameColor);
            gDrawer.Draw2DRoundedFrame(
                nn::util::MakeFloat2(itemPos.x - 4.f, itemPos.y + 12.f),
                nn::util::MakeFloat2(itemSize.x + 8.f, itemSize.y - 24.f),
                0.25f, 32.f, 3.f);
        }
        g_LoopSettingButtons.Draw();
        // シークバーの描画
        for (auto itr : g_DialogSeekBar)
        {
            if (itr.IsFocus())
            {
                nn::util::Float2 itemPos = itr.GetItemSet().at(itr.GetFocusItemIndex())->GetPos();
                nn::util::Float2 itemSize = itr.GetItemSet().at(itr.GetFocusItemIndex())->GetSize();
                gDrawer.SetColor(itr.GetItemSet().at(itr.GetFocusItemIndex())->IsSelected() ? selectedColor : hoverColor);
                gDrawer.Draw2DRoundedRect(
                    nn::util::MakeFloat2(itemPos.x - 8.f, itemPos.y - (gTextWriter.GetLineHeight() + 8.f)),
                    nn::util::MakeFloat2(itemSize.x + 16.f, itemSize.y + (gTextWriter.GetLineHeight() + 8.f)),
                    0.25f, 32.f);

                gDrawer.SetColor(frameColor);
                gDrawer.Draw2DRoundedFrame(
                    nn::util::MakeFloat2(itemPos.x - 8.f, itemPos.y - (gTextWriter.GetLineHeight() + 8.f)),
                    nn::util::MakeFloat2(itemSize.x + 16.f, itemSize.y + (gTextWriter.GetLineHeight() + 8.f)),
                    0.25f, 32.f, 3.f);
            }
            itr.Draw();
        }
        // 出力/再生/リセットボタンの描画
        g_DialogButtons.Draw();
    } // NOLINT(impl/function_size)
}
