﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "ExportDialog.h"

namespace VibrationCollection
{
    nns::hidfw::layout::ButtonGroup ExportDialogButton::closeButton;
    nns::hidfw::layout::ButtonGroup ExportDialogButton::closeExportDialogButton;
    nns::hidfw::layout::ButtonGroup ExportDialogButton::renameButtons[2];
    nns::hidfw::layout::ButtonGroup ExportDialogButton::mainButton;

    nns::hidfw::layout::ButtonGroup ExportDialogButton::inputPathButtons;
    nns::hidfw::layout::ButtonGroup ExportDialogButton::changeMountTargetButtons;
    nns::hidfw::layout::ButtonGroup ExportDialogButton::changePathButtons;

    std::string ExportBnvibFile::path = "";
    std::string ExportBnvibFile::fileName = "output.bnvib";
    std::string ExportBnvibFile::openPath = "";

    void ShowSwkbd(void* pushButton, void* param) NN_NOEXCEPT
    {
        NN_UNUSED(pushButton);
        NN_ASSERT_NOT_NULL(param);

        std::string* pPath = (std::string*)param;

        nn::swkbd::ShowKeyboardArg showKeyboardArg;
        nn::swkbd::MakePreset(&(showKeyboardArg.keyboardConfig), nn::swkbd::Preset_Default);
        nn::swkbd::SetGuideTextUtf8(&(showKeyboardArg.keyboardConfig), u8"入力");
        ApplicationHeap applicationHeap;
        applicationHeap.Initialize(2 * 1024 * 1024);
        void* swkbdWorkBuffer = applicationHeap.Allocate(nn::os::MemoryPageSize, nn::os::MemoryPageSize);

        showKeyboardArg.keyboardConfig.isUseUtf8 = true;
        showKeyboardArg.keyboardConfig.keyboardMode = nn::swkbd::KeyboardMode_ASCII;
        showKeyboardArg.keyboardConfig.inputFormMode = nn::swkbd::InputFormMode_OneLine;
        showKeyboardArg.keyboardConfig.isUseNewLine = false;
        showKeyboardArg.keyboardConfig.isUseBlurBackground = true;
        showKeyboardArg.keyboardConfig.initialCursorPos = nn::swkbd::InitialCursorPos_Last;
        showKeyboardArg.keyboardConfig.textMaxLength = 100;
        showKeyboardArg.workBuf = swkbdWorkBuffer;
        showKeyboardArg.workBufSize = nn::os::MemoryPageSize;

        nn::swkbd::SetInitialTextUtf8(&showKeyboardArg, pPath->c_str());

        nn::swkbd::String outputString;
        outputString.ptr = reinterpret_cast<char *>(applicationHeap.Allocate(nn::os::MemoryPageSize, nn::os::MemoryPageSize));
        outputString.bufSize = nn::swkbd::GetRequiredStringBufferSize();

        nn::Result result = nn::swkbd::ShowKeyboard(&outputString, showKeyboardArg);

        if (result.IsSuccess())
        {
            *pPath = static_cast<char*>(outputString.ptr);
        }

        applicationHeap.Deallocate(outputString.ptr);
        applicationHeap.Deallocate(swkbdWorkBuffer);
        applicationHeap.Finalize();
    }

    void CallExportVibrationValue(void* pushButton, void* param) NN_NOEXCEPT
    {
        NN_UNUSED(pushButton);
        NN_UNUSED(param);

        static uint8_t vibrationBuffer[BnvibFileData::FileSizeMax];
        size_t outSize = 0;

        auto mountType = gFileManager.GetMountType();
        std::string rootPath = gFileManager.GetRootPath();
        gFileManager.SetRootPath((ExportBnvibFile::path).c_str());
        gFileManager.ChangePath();
        ExportVibrationValue(vibrationBuffer, BnvibFileData::FileSizeMax, &outSize);

        auto result = gFileManager.ExportFile((ExportBnvibFile::fileName).c_str(), vibrationBuffer, outSize);

        gFileManager.SetMountType(mountType);
        gFileManager.SetRootPath(rootPath.c_str());
        gFileManager.ChangePath();

        if (result.IsSuccess())
        {
            Reload(nullptr, nullptr);
            for (int i = 0; i < gFileManager.GetBnvibFileCount(); ++i)
            {
                if (gFileManager.GetBnvibFile().at(i).FileName == ExportBnvibFile::fileName)
                {
                    auto adValue = std::max(gFileManager.GetBnvibFileCount(), 10) - 10;
                    g_SelectedBnvibButtonInedx = i;
                    g_BnvibButtonStartIndex = std::min(i, adValue);
                    auto index = g_SelectedBnvibButtonInedx > adValue ? g_SelectedBnvibButtonInedx - adValue : 0;
                    if (g_ButtonGroup[ApplicationButtonGroup_Vibration].GetSelectedItemIndex() > 0)
                    {
                        g_ButtonGroup[ApplicationButtonGroup_Vibration].SelectItem(index, false);
                    }
                }
            }
            PlayVibration(nullptr, nullptr);
        }
    }

    void CallChangeDirectory(void* pushButton, void* param) NN_NOEXCEPT
    {
        ReloadButton(pushButton, param);
        CloseButtonFunc(pushButton, param);
    }

    void CallChangeMountTarget(void* pushButton, void* param) NN_NOEXCEPT
    {
        NN_UNUSED(pushButton);
        MountType type = (MountType)(uint64_t)param;
        NN_ASSERT_MINMAX(type, MountType_Sd, MountType_Host);

        switch (type)
        {
        case VibrationCollection::MountType_Sd:
            ExportBnvibFile::openPath = g_SdRootPath;
            gFileManager.SetRootPath(g_SdRootPath.c_str());
            break;
        case VibrationCollection::MountType_Host:
            ExportBnvibFile::openPath = g_HostRootPath;
            gFileManager.SetRootPath(g_HostRootPath.c_str());
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }
        gFileManager.SetMountType(type);
    }

    void ShowExportBnvibDialog(void* pushButton, void* param) NN_NOEXCEPT
    {
        NN_UNUSED(pushButton);
        NN_UNUSED(param);

        nns::hidfw::layout::Dialog dialog;
        nns::hidfw::layout::Dialog::DialogInfo info;
        info.Size = nn::util::MakeFloat2(600.f, 480.f);

        // 閉じるボタン
        if (ExportDialogButton::closeExportDialogButton.GetListSize() == 0)
        {
            InitializeCloseButton(&ExportDialogButton::closeExportDialogButton);
        }
        NN_ASSERT_EQUAL(ExportDialogButton::closeExportDialogButton.GetListSize(), 1);
        ExportDialogButton::closeExportDialogButton.GetItemSet().at(0)->SetParam(&dialog);

        // 出力先の変更ボタン
        if (ExportDialogButton::renameButtons[0].GetListSize() == 0)
        {
            InitializeRenameButtons(ExportDialogButton::renameButtons);
        }
        NN_ASSERT_EQUAL(ExportDialogButton::renameButtons[0].GetListSize(), 1);
        NN_ASSERT_EQUAL(ExportDialogButton::renameButtons[1].GetListSize(), 1);

        // 出力・キャンセルボタン
        if (ExportDialogButton::mainButton.GetListSize() == 0)
        {
            InitializeMainButtons(&ExportDialogButton::mainButton);
        }
        NN_ASSERT_EQUAL(ExportDialogButton::mainButton.GetListSize(), 2);
        ExportDialogButton::mainButton.GetItemSet().at(1)->SetParam(&dialog);

        if (
            !ExportDialogButton::closeExportDialogButton.IsFocus() && !ExportDialogButton::renameButtons[0].IsFocus() &&
            !ExportDialogButton::renameButtons[1].IsFocus() && !ExportDialogButton::mainButton.IsFocus()
            )
        {
            ExportDialogButton::closeExportDialogButton.Focus(true);
        }

        // ボタンの前後関係を設定します
        ExportDialogButton::closeExportDialogButton.SetPrevItemSet(&ExportDialogButton::mainButton);
        ExportDialogButton::closeExportDialogButton.SetNextItemSet(&ExportDialogButton::renameButtons[0]);
        ExportDialogButton::renameButtons[0].SetPrevItemSet(&ExportDialogButton::closeExportDialogButton);
        ExportDialogButton::renameButtons[0].SetNextItemSet(&ExportDialogButton::renameButtons[1]);
        ExportDialogButton::renameButtons[1].SetPrevItemSet(&ExportDialogButton::renameButtons[0]);
        ExportDialogButton::renameButtons[1].SetNextItemSet(&ExportDialogButton::mainButton);
        ExportDialogButton::mainButton.SetPrevItemSet(&ExportDialogButton::renameButtons[1]);
        ExportDialogButton::mainButton.SetNextItemSet(&ExportDialogButton::closeExportDialogButton);

        // アニメーションのリセット
        ExportDialogButton::closeExportDialogButton.Reset();
        ExportDialogButton::renameButtons[0].Reset();
        ExportDialogButton::renameButtons[1].Reset();
        ExportDialogButton::mainButton.Reset();

        ExportBnvibFile::path = gFileManager.GetTempRootPath();
        ExportBnvibFile::path += "/Bnvib/";

        info.Title = "Export";
        info.UpdateFunc = UpdateExportBnvibDialog;
        info.DrawerFunc = DrawExportBnvibDialog;

        dialog.ShowDialog(info);

        CalcPreviewVibrationValueViewer();
    }

    void ShowOpenDirectoryDialog(void* pushButton, void* param) NN_NOEXCEPT
    {
        NN_UNUSED(pushButton);
        NN_UNUSED(param);

        nns::hidfw::layout::Dialog dialog;
        nns::hidfw::layout::Dialog::DialogInfo info;
        info.Size = nn::util::MakeFloat2(600.f, 320.f);

        // 閉じるボタン
        if (ExportDialogButton::closeButton.GetListSize() == 0)
        {
            InitializeCloseButton(&ExportDialogButton::closeButton);
        }
        NN_ASSERT_EQUAL(ExportDialogButton::closeButton.GetListSize(), 1);
        ExportDialogButton::closeButton.GetItemSet().at(0)->SetParam(&dialog);

        // パスの入力ボタン
        if (ExportDialogButton::inputPathButtons.GetListSize() == 0)
        {
            InitializeInputPathButton(&ExportDialogButton::inputPathButtons);
        }
        NN_ASSERT_EQUAL(ExportDialogButton::inputPathButtons.GetListSize(), 1);

        // マウント切り替えボタン
        if (ExportDialogButton::changeMountTargetButtons.GetListSize() == 0)
        {
            InitializeChangeMountTargetButton(&ExportDialogButton::changeMountTargetButtons);
        }
        NN_ASSERT_EQUAL(ExportDialogButton::changeMountTargetButtons.GetListSize(), 2);

        // 適用・キャンセルボタン
        if (ExportDialogButton::changePathButtons.GetListSize() == 0)
        {
            InitializeChangePathButton(&ExportDialogButton::changePathButtons);
        }
        NN_ASSERT_EQUAL(ExportDialogButton::changePathButtons.GetListSize(), 1);
        ExportDialogButton::changePathButtons.GetItemSet().at(0)->SetParam(&dialog);

        if (
            !ExportDialogButton::closeButton.IsFocus() && !ExportDialogButton::changeMountTargetButtons.IsFocus() &&
            !ExportDialogButton::inputPathButtons.IsFocus() && !ExportDialogButton::changePathButtons.IsFocus()
            )
        {
            ExportDialogButton::closeButton.Focus(true);
        }

        // ボタンの前後関係を設定します
        ExportDialogButton::closeButton.SetPrevItemSet(&ExportDialogButton::changePathButtons);
        ExportDialogButton::closeButton.SetNextItemSet(&ExportDialogButton::changeMountTargetButtons);
        ExportDialogButton::changeMountTargetButtons.SetPrevItemSet(&ExportDialogButton::closeButton);
        ExportDialogButton::changeMountTargetButtons.SetNextItemSet(&ExportDialogButton::inputPathButtons);
        ExportDialogButton::inputPathButtons.SetPrevItemSet(&ExportDialogButton::changeMountTargetButtons);
        ExportDialogButton::inputPathButtons.SetNextItemSet(&ExportDialogButton::changePathButtons);
        ExportDialogButton::changePathButtons.SetPrevItemSet(&ExportDialogButton::inputPathButtons);
        ExportDialogButton::changePathButtons.SetNextItemSet(&ExportDialogButton::closeButton);

        // アニメーションのリセット
        ExportDialogButton::closeButton.Reset();
        ExportDialogButton::inputPathButtons.Reset();
        //ExportDialogButton::changeMountTargetButtons.Reset();
        ExportDialogButton::changePathButtons.Reset();

        info.Title = "Open Directory";
        info.UpdateFunc = UpdateOpenDirectoryDialog;
        info.DrawerFunc = DrawOpenDirectoryDialog;

        ExportBnvibFile::openPath = gFileManager.GetTempRootPath();

        dialog.ShowDialog(info);
    }

    void InitializeMainButtons(nns::hidfw::layout::ButtonGroup* pButtonGroup) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pButtonGroup);

        static const nn::util::Float2 basePos = NN_UTIL_FLOAT_2_INITIALIZER(360.f, 540.f);
        static const nn::util::Float2 margin = NN_UTIL_FLOAT_2_INITIALIZER(376.f, 0.f);
        static const nn::util::Float2 size = NN_UTIL_FLOAT_2_INITIALIZER(360.f, 48.f);

        const char* buttonText[] =
        {
            "Export Bnvib File",
            "Cancel"
        };

        nns::hidfw::layout::LayoutFunction pFunc[] =
        {
            CallExportVibrationValue,
            CloseButtonFunc
        };

        nn::util::Color4u8 buttonColor[] =
        {
            nn::util::Color4u8(49, 119, 125, 255),
            nn::util::Color4u8(232, 57, 54, 255),
        };

        void* params[] =
        {
            (void*)ExportBnvibFile::fileName.c_str(),
            nullptr
        };

        nn::util::Float2 pos = basePos;

        for (size_t i = 0; i < NN_ARRAY_SIZE(buttonText); ++i)
        {
            nns::hidfw::layout::Button button;
            button.SetPos(pos.x, pos.y);
            button.SetSize((i == 0) ? size.x : size.x / 2.f, size.y);
            button.SetText(buttonText[i]);
            button.SetFunc(pFunc[i]);
            button.SetTextColor(nn::util::Color4u8::White());
            button.ThroughCancel(true);                             // キャンセルを無効化します
            button.SetMainColor(buttonColor[i]);
            button.OnOffMode(false);
            button.SetParam(params[i]);
            pButtonGroup->Add(new nns::hidfw::layout::Button(button));

            pos.x += margin.x;
            pos.y += margin.y;
        }
        pButtonGroup->SetOrientation(nns::hidfw::layout::ButtonGroup::eOrientation_Horizontal);
        pButtonGroup->Focus(false);
    }

    void InitializeRenameButtons(nns::hidfw::layout::ButtonGroup* pButtonGroup) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pButtonGroup);

        static const nn::util::Float2 basePos = NN_UTIL_FLOAT_2_INITIALIZER(848.f, 366.f);
        static const nn::util::Float2 margin = NN_UTIL_FLOAT_2_INITIALIZER(0.f, 90.f);
        static const nn::util::Float2 size = NN_UTIL_FLOAT_2_INITIALIZER(72.f, 48.f);

        const char* buttonText[] =
        {
            "...",
            "..."
        };

        nns::hidfw::layout::LayoutFunction pFunc[] =
        {
            ShowSwkbd,
            ShowSwkbd
        };

        nn::util::Color4u8 buttonColor[] =
        {
            nn::util::Color4u8(252, 252, 252, 255),
            nn::util::Color4u8(252, 252, 252, 255),
        };

        void* params[] =
        {
            &ExportBnvibFile::path,
            &ExportBnvibFile::fileName
        };

        nn::util::Float2 pos = basePos;

        for (size_t i = 0; i < NN_ARRAY_SIZE(buttonText); ++i)
        {
            nns::hidfw::layout::Button button;
            button.SetPos(pos.x, pos.y);
            button.SetSize(size.x, size.y);
            button.SetText(buttonText[i]);
            button.SetFunc(pFunc[i]);
            button.SetTextColor(nn::util::Color4u8::Black());
            button.ThroughCancel(true);                             // キャンセルを無効化します
            button.SetMainColor(buttonColor[i]);
            button.OnOffMode(false);
            button.SetParam(params[i]);

            pButtonGroup[i].Add(new nns::hidfw::layout::Button(button));
            pButtonGroup[i].SetOrientation(nns::hidfw::layout::ButtonGroup::eOrientation_Horizontal);
            pButtonGroup[i].Focus(false);
            pos.x += margin.x;
            pos.y += margin.y;
        }
    }

    void InitializeCloseButton(nns::hidfw::layout::ButtonGroup* pButtonGroup) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pButtonGroup);

        nns::hidfw::layout::Button button;
        button.SetDefault();
        button.SetSize(96.f, 40.f);
        button.SetPos(924.f - button.GetSize().x, 132.f);
        button.SetText("(+) Close");
        button.SetMainColor(nn::util::Color4u8(232, 57, 54, 255));
        button.SetTextColor(nn::util::Color4u8::White());
        button.SetChooseMode(nns::hidfw::layout::BaseItem::ChoiceMode::Touch::Mask);
        button.OnOffMode(false);
        button.SetEffectColor(nn::util::Color4u8(255, 255, 255, 0));
        button.SetFunc(CloseButtonFunc);

        pButtonGroup->Add(new nns::hidfw::layout::Button(button));
        pButtonGroup->Focus(true);
    }

    void InitializeInputPathButton(nns::hidfw::layout::ButtonGroup* pButtonGroup) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pButtonGroup);

        static const nn::util::Float2 basePos = NN_UTIL_FLOAT_2_INITIALIZER(848.f, 392.f);
        static const nn::util::Float2 margin = NN_UTIL_FLOAT_2_INITIALIZER(0.f, 90.f);
        static const nn::util::Float2 size = NN_UTIL_FLOAT_2_INITIALIZER(72.f, 48.f);

        const char* buttonText[] =
        {
            "..."
        };

        nns::hidfw::layout::LayoutFunction pFunc[] =
        {
            ShowSwkbd
        };

        nn::util::Color4u8 buttonColor[] =
        {
            nn::util::Color4u8(252, 252, 252, 255),
        };

        void* params[] =
        {
            &ExportBnvibFile::openPath
        };

        nn::util::Float2 pos = basePos;

        for (size_t i = 0; i < NN_ARRAY_SIZE(buttonText); ++i)
        {
            nns::hidfw::layout::Button button;
            button.SetPos(pos.x, pos.y);
            button.SetSize(size.x, size.y);
            button.SetText(buttonText[i]);
            button.SetFunc(pFunc[i]);
            button.SetTextColor(nn::util::Color4u8::Black());
            button.ThroughCancel(true);                             // キャンセルを無効化します
            button.SetMainColor(buttonColor[i]);
            button.OnOffMode(false);
            button.SetParam(params[i]);

            pButtonGroup[i].Add(new nns::hidfw::layout::Button(button));
            pButtonGroup[i].SetOrientation(nns::hidfw::layout::ButtonGroup::eOrientation_Horizontal);
            pButtonGroup[i].Focus(false);
            pos.x += margin.x;
            pos.y += margin.y;
        }
    }

    void InitializeChangeMountTargetButton(nns::hidfw::layout::ButtonGroup* pButtonGroup) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pButtonGroup);

        static const nn::util::Float2 basePos = NN_UTIL_FLOAT_2_INITIALIZER(372.f, 308.f);
        static const nn::util::Float2 margin = NN_UTIL_FLOAT_2_INITIALIZER(280.f, 0.f);
        static const nn::util::Float2 size = NN_UTIL_FLOAT_2_INITIALIZER(268.f, 48.f);

        const char* buttonText[] =
        {
            "SD card",
            "Host PC"
        };

        nn::util::Float2 pos = basePos;

        for (size_t i = 0; i < NN_ARRAY_SIZE(buttonText); ++i)
        {
            nns::hidfw::layout::Button button;
            button.SetPos(pos.x, pos.y);
            button.SetSize(size.x, size.y);
            button.SetText(buttonText[i]);
            button.ThroughCancel(true);
            button.SetMainColor(nn::util::Color4u8(49, 119, 125, 255));
            button.SetTextColor(nn::util::Color4u8::White());
            button.OnOffMode(true);
            button.SetFunc(CallChangeMountTarget);
            button.SetParam((uint64_t*)((MountType)i));
            pButtonGroup->Add(new nns::hidfw::layout::Button(button));

            pos.x += margin.x;
            pos.y += margin.y;
        }
        pButtonGroup->Select(gFileManager.GetTempMountType() == MountType_Sd ? 0 : 1);
    }

    void InitializeChangePathButton(nns::hidfw::layout::ButtonGroup* pButtonGroup) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pButtonGroup);

        static const nn::util::Float2 basePos = NN_UTIL_FLOAT_2_INITIALIZER(372.f, 456.f);
        static const nn::util::Float2 size = NN_UTIL_FLOAT_2_INITIALIZER(548.f, 48.f);

        nn::util::Float2 pos = basePos;

        nns::hidfw::layout::Button button;
        button.SetPos(pos.x, pos.y);
        button.SetSize(size.x, size.y);
        button.SetText("Reload and close the file");
        button.SetFunc(CallChangeDirectory);
        button.SetTextColor(nn::util::Color4u8::White());
        button.ThroughCancel(true);                             // キャンセルを無効化します
        button.SetMainColor(nn::util::Color4u8(232, 57, 54, 255));
        button.OnOffMode(false);
        pButtonGroup->Add(new nns::hidfw::layout::Button(button));

        pButtonGroup->SetOrientation(nns::hidfw::layout::ButtonGroup::eOrientation_Horizontal);
        pButtonGroup->Focus(false);
    }

    void UpdateExportBnvibDialog(void* executant, void* param) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(executant);
        NN_UNUSED(param);

        nns::hidfw::layout::Dialog* pDialog = (nns::hidfw::layout::Dialog*)executant;
        const nn::util::Float2 dialogSize = pDialog->GetDialogInfo().Size;

        ExportDialogButton::closeExportDialogButton.GetItemSet().at(0)->SetPos(
            (640.f + dialogSize.x / 2.f) - (ExportDialogButton::closeExportDialogButton.GetItemSet().at(0)->GetSize().x + 16.f),
            (360.f - dialogSize.y / 2.f) + (12.f)
        );

        gTextWriter.SetTextColor(nn::util::Color4u8::Blue());
        gTextWriter.SetScale(1.4f, 1.4f);
        gTextWriter.SetCursorX((1280.f - gTextWriter.CalculateStringWidth(pDialog->GetDialogInfo().Title.c_str())) / 2.f);
        gTextWriter.SetCursorY((720.f - dialogSize.y) / 2.f + 16.f);
        gTextWriter.Print(pDialog->GetDialogInfo().Title.c_str());
        gTextWriter.SetScale(1.f, 1.f);

        gTextWriter.SetTextColor(nn::util::Color4u8(16, 16, 16));
        gTextWriter.SetCursor(372, 204);
        gTextWriter.Print("BaseFile");

        gTextWriter.SetTextColor(nn::util::Color4u8(2,3,4));
        gTextWriter.SetCursor(388, 248);
        gTextWriter.SetFixedWidth(11.f);
        gTextWriter.SetFixedWidthEnabled(true);
        gTextWriter.Print("%.41s", gFileManager.GetBnvibFile().at(g_SelectedBnvibButtonInedx).FileName.c_str());
        gTextWriter.SetFixedWidthEnabled(false);

        gTextWriter.SetTextColor(nn::util::Color4u8(16, 16, 16));
        gTextWriter.SetCursor(372, 338);
        gTextWriter.Print("Output Dir");

        gTextWriter.SetTextColor(nn::util::Color4u8(16, 16, 16));
        gTextWriter.SetCursor(388, 382);
        gTextWriter.SetFixedWidth(11.f);
        gTextWriter.SetFixedWidthEnabled(true);
        gTextWriter.Print("%.39s", ExportBnvibFile::path.c_str());
        gTextWriter.SetFixedWidthEnabled(false);

        gTextWriter.SetTextColor(nn::util::Color4u8(16, 16, 16));
        gTextWriter.SetCursor(372, 428);
        gTextWriter.Print("FileName");

        gTextWriter.SetTextColor(nn::util::Color4u8(16, 16, 16));
        gTextWriter.SetCursor(388, 472);
        gTextWriter.SetFixedWidthEnabled(true);
        gTextWriter.Print("%.39s", ExportBnvibFile::fileName.c_str());
        gTextWriter.SetFixedWidthEnabled(false);

        ExportDialogButton::closeExportDialogButton.Update();
        ExportDialogButton::renameButtons[0].Update();
        ExportDialogButton::renameButtons[1].Update();
        ExportDialogButton::mainButton.Update();

        for (auto itr : gController.GetConnectedControllerList())
        {
            if (itr->IsTrigger(nn::hid::NpadButton::X::Mask))
            {
                PlayVibration(nullptr, nullptr);
                break;
            }
        }
    }

    void DrawExportBnvibDialog(void* executant, void* param) NN_NOEXCEPT
    {
        NN_UNUSED(executant);
        NN_UNUSED(param);

        ExportDialogButton::renameButtons[0].Draw();
        ExportDialogButton::renameButtons[1].Draw();

        ExportDialogButton::closeExportDialogButton.Draw();
        ExportDialogButton::mainButton.Draw();

        gDrawer.SetColor(gFileManager.GetTempMountType() == MountType_Sd ? nn::util::Color4u8::Blue() : nn::util::Color4u8(64, 64, 64, 128));
        DrawToolIcon(
            nn::util::MakeFloat2(480, 330.f),
            nn::util::MakeFloat2(0.5f, 0.5f),
            ToolIcon::Sd::Mask);
        gDrawer.SetColor(gFileManager.GetTempMountType() == MountType_Host ? nn::util::Color4u8::Blue() : nn::util::Color4u8(64, 64, 64, 128));
        DrawToolIcon(
            nn::util::MakeFloat2(528.f, 330.f),
            nn::util::MakeFloat2(0.5f, 0.5f),
            ToolIcon::Pc::Mask);

        gDrawer.SetColor(nn::util::Color4u8::Gray());
        gDrawer.Draw2DRect(nn::util::MakeFloat2(372, 232), nn::util::MakeFloat2(536, 48));
        gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Down, nn::util::Color4u8( 80, 80, 80, 255), nn::util::Color4u8::Gray());
        gDrawer.Draw2DRect(nn::util::MakeFloat2(372, 232), nn::util::MakeFloat2(536, 10));
        gDrawer.SetColor(nn::util::Color4u8::Black());
        gDrawer.Draw2DFrame(nn::util::MakeFloat2(372, 232), nn::util::MakeFloat2(536, 48), 1);

        gDrawer.SetColor(nn::util::Color4u8::Gray());
        gDrawer.Draw2DLine(nn::util::MakeFloat2(372, 310), nn::util::MakeFloat2(908, 310));
        gDrawer.Draw2DLine(nn::util::MakeFloat2(372, 310), nn::util::MakeFloat2(908, 310));

        gDrawer.SetColor(nn::util::Color4u8::White());
        gDrawer.Draw2DRect(nn::util::MakeFloat2(372, 366), nn::util::MakeFloat2(456, 48));
        gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Down, nn::util::Color4u8::Gray(), nn::util::Color4u8::White());
        gDrawer.Draw2DRect(nn::util::MakeFloat2(372, 366), nn::util::MakeFloat2(456, 10));
        gDrawer.SetColor(nn::util::Color4u8::Gray());
        gDrawer.Draw2DFrame(nn::util::MakeFloat2(372, 366), nn::util::MakeFloat2(456, 48), 1);

        gDrawer.SetColor(nn::util::Color4u8::White());
        gDrawer.Draw2DRect(nn::util::MakeFloat2(372, 456), nn::util::MakeFloat2(456, 48));
        gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Down, nn::util::Color4u8::Gray(), nn::util::Color4u8::White());
        gDrawer.Draw2DRect(nn::util::MakeFloat2(372, 456), nn::util::MakeFloat2(456, 10));
        gDrawer.SetColor(nn::util::Color4u8::Gray());
        gDrawer.Draw2DFrame(nn::util::MakeFloat2(372, 456), nn::util::MakeFloat2(456, 48), 1);
    }

    void UpdateOpenDirectoryDialog(void* executant, void* param) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(executant);
        NN_UNUSED(param);

        nns::hidfw::layout::Dialog* pDialog = (nns::hidfw::layout::Dialog*)executant;
        const nn::util::Float2 dialogSize = pDialog->GetDialogInfo().Size;

        ExportDialogButton::closeButton.GetItemSet().at(0)->SetPos(
            (640.f + dialogSize.x / 2.f) - (ExportDialogButton::closeButton.GetItemSet().at(0)->GetSize().x + 16.f),
            (360.f - dialogSize.y / 2.f) + (12.f)
        );

        ExportDialogButton::closeButton.Update();
        ExportDialogButton::inputPathButtons.Update();
        ExportDialogButton::changeMountTargetButtons.Update();
        ExportDialogButton::changePathButtons.Update();

        gTextWriter.SetTextColor(nn::util::Color4u8::Blue());
        gTextWriter.SetScale(1.4f, 1.4f);
        gTextWriter.SetCursorX((1280.f - gTextWriter.CalculateStringWidth(pDialog->GetDialogInfo().Title.c_str())) / 2.f);
        gTextWriter.SetCursorY((720.f - dialogSize.y) / 2.f + 16.f);
        gTextWriter.Print(pDialog->GetDialogInfo().Title.c_str());
        gTextWriter.SetScale(1.f, 1.f);

        if (ExportBnvibFile::openPath != gFileManager.GetTempRootPath())
        {
            gFileManager.SetRootPath(ExportBnvibFile::openPath.c_str());
        }

        gTextWriter.SetTextColor(nn::util::Color4u8::Blue());
        gTextWriter.SetCursor(372, ExportDialogButton::changeMountTargetButtons.GetItemSet().at(0)->GetPos().y - 32.f);
        gTextWriter.Print("Mount Type");

        gTextWriter.SetTextColor(nn::util::Color4u8(16, 16, 16));
        gTextWriter.SetCursor(388, 408);
        gTextWriter.SetFixedWidthEnabled(true);
        gTextWriter.Print("%.39s", gFileManager.GetTempRootPath());
        gTextWriter.SetFixedWidthEnabled(false);
    }

    void DrawOpenDirectoryDialog(void* executant, void* param) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(executant);
        NN_UNUSED(param);

        nns::hidfw::layout::Dialog* pDialog = (nns::hidfw::layout::Dialog*)executant;

        ExportDialogButton::closeButton.Draw();
        ExportDialogButton::inputPathButtons.Draw();
        ExportDialogButton::changeMountTargetButtons.Draw();
        ExportDialogButton::changePathButtons.Draw();

        gDrawer.SetColor(nn::util::Color4u8::White());
        gDrawer.Draw2DRect(nn::util::MakeFloat2(372, 392), nn::util::MakeFloat2(456, 48));
        gDrawer.SetColor(nns::hidfw::gfx::GraphicsDrawer::GradationDirection_Down, nn::util::Color4u8::Gray(), nn::util::Color4u8::White());
        gDrawer.Draw2DRect(nn::util::MakeFloat2(372, 392), nn::util::MakeFloat2(456, 10));
        gDrawer.SetColor(nn::util::Color4u8::Gray());
        gDrawer.Draw2DFrame(nn::util::MakeFloat2(372, 392), nn::util::MakeFloat2(456, 48), 1);

        gDrawer.SetColor(nn::util::Color4u8(192,192,192,255));
        auto lineX = gGraphics.GetGraphicsSystem().GetScreenWidth() / 2.f -  pDialog->GetDialogInfo().Size.x / 2.f;
        gDrawer.Draw2DLine(nn::util::MakeFloat2(lineX, 372), nn::util::MakeFloat2(lineX + pDialog->GetDialogInfo().Size.x, 372));
    }
}
