﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <map>

#include <nn/oe.h>
#include <nn/hid/hid_ControllerSupport.h>

namespace VibrationCollection
{
    typedef int IconIndex;

    struct NpadIconButton
    {
        typedef ::nn::util::BitFlagSet<64, nn::hid::NpadButton>::Flag<48> UpDown;
        typedef ::nn::util::BitFlagSet<64, nn::hid::NpadButton>::Flag<49> LeftRight;
        typedef ::nn::util::BitFlagSet<64, nn::hid::NpadButton>::Flag<50> Cross;             // 選択方向が無い十字キー
        typedef ::nn::util::BitFlagSet<64, nn::hid::NpadButton>::Flag<51> AllDirections;     // 全ての方向ボタン
        typedef ::nn::util::BitFlagSet<64, nn::hid::NpadButton>::Flag<52> Home;
        typedef ::nn::util::BitFlagSet<64, nn::hid::NpadButton>::Flag<53> Capture;
    };

    struct ToolIcon
    {
        typedef ::nn::util::BitFlagSet<8, ToolIcon>::Flag<0> Pc;    //!< PC のアイコン
        typedef ::nn::util::BitFlagSet<8, ToolIcon>::Flag<1> Sd;    //!< microSD のアイコン
    };

    typedef ::nn::util::BitFlagSet<8, ToolIcon> ToolIconSet;

    enum ButtonIconType
    {
        ButtonIconType_Border = 0,              // 白線
        ButtonIconType_Fill = 1,                // 塗りつぶし
        ButtonIconType_Border_2 = 2,            // 白線 (別デザイン 方向キーとスティックと+-キーが対象)
        ButtonIconType_Fill_2 = 3,              // 塗りつぶし (別デザイン 方向キーとスティックと+-キーが対象)
        ButtonIconType_Border_3 = 4,            // 白線 (別デザイン・方向キーが対象)
        ButtonIconType_Fill_3 = 5,              // 塗りつぶし (該当なし)
    };

    enum StyleIconType
    {
        StyleIconType_Default = 0,              // 基本的なスタイルアイコン
        StyleIconType_Kuina = 1,                // KUINA
        StyleIconType_Single = 1,               // 一本横持ち・共用スタイル ( NpadStyleJoyLeft と NpadStyleJoyRight が対象)
        StyleIconType_Horizontal = 2,           // 一本横持ち ( NpadStyleJoyLeft と NpadStyleJoyRight が対象)
    };

    struct TextureKey
    {
        int index;
        int type;
        static TextureKey Make(const int i, const int t) NN_NOEXCEPT
        {
            TextureKey out;
            out.index = i;
            out.type = t;
            return out;
        }
        bool operator < (const TextureKey& rhs) const NN_NOEXCEPT
        {
            if (index == rhs.index)
            {
                return type < rhs.type;
            }
            return index < rhs.index;
        }
    };

    struct TextureInfo
    {
        int32_t textureIndex;
        nn::util::Float2 size;
        nn::util::Float2 divSize;
        std::map<TextureKey, IconIndex> iconSet;
    };

    extern nn::gfx::ResTextureFile*     g_pResTextureFile;
    extern int32_t                      g_TextureIndex[4];
    extern nn::gfx::DescriptorSlot      g_TextureDescriptor[4];
    extern nn::gfx::DescriptorSlot      g_SamplerDescriptor;
    extern nn::gfx::Sampler             g_Sampler;

    extern nn::util::Float2             g_DefaultIconSize;

    typedef std::pair<TextureKey, IconIndex> TexturePair;
    extern TextureInfo g_NpadButtonIconSet;
    extern TextureInfo g_NpadStyleIconSet;
    extern TextureInfo g_ToolIconSet;

    void SetupNpadStyleIconSet() NN_NOEXCEPT;

    void SetupNpadButtonIconset() NN_NOEXCEPT;

    void SetupToolIconSet() NN_NOEXCEPT;

    void DrawBackGround() NN_NOEXCEPT;

    void DrawMainMenu() NN_NOEXCEPT;

    void DrawStatusBar() NN_NOEXCEPT;

    void DrawTitle() NN_NOEXCEPT;

    void InitializeResource(nn::mem::StandardAllocator* memory) NN_NOEXCEPT;

    void DrawTitleLogo(const nn::util::Float2& pos, const nn::util::Float2& size) NN_NOEXCEPT;

    void DrawIconSet(
        const nn::util::Float2& pos,
        const nn::util::Float2& scale,
        const TextureInfo& icon,
        const TextureKey& key
    ) NN_NOEXCEPT;

    // 64 x 64 のアイコン画像を描画します
    void DrawNpadStyle(
        const nn::util::Float2& pos,
        const nn::util::Float2& scale,
        const nn::hid::NpadStyleSet& style,
        const StyleIconType type) NN_NOEXCEPT;

    // 64 x 64 のアイコン画像を描画します
    void DrawNpadButton(
        const nn::util::Float2& pos,
        const nn::util::Float2& scale,
        const nn::hid::NpadButtonSet& button,
        const ButtonIconType type) NN_NOEXCEPT;

    // 64 x 64 のアイコン画像を描画します
    void DrawToolIcon(
        const nn::util::Float2& pos,
        const nn::util::Float2& scale,
        const ToolIconSet& icon) NN_NOEXCEPT;
}
