﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once
/*
 * 効果音の再生処理の処理
 */

#include <nn/nn_TimeSpan.h>
#include <nn/audio.h>
#include <nns/audio/audio_HidUtilities.h>
#include <nns/audio/audio_WavFormat.h>

#include "Base.h"
#include "File.h"

namespace VibrationCollection
{
    typedef int32_t VoiceHandle;

    struct VoiceData
    {
        nn::audio::VoiceType Voice;
        nn::audio::WaveBuffer Buffer;
        nn::audio::AdpcmHeaderInfo* Header;
        void* Data;
    };

    struct AudioQueue
    {
        bool        IsWave;
        VoiceHandle Handle;
        nn::TimeSpan Delay;
    };

    class Audio
    {
    public:
        static const int MaxPlayCount = 8;

        static const std::string MountName;
        static const std::string SDFilesRootPath;

        static std::string HostRootPath;
        static const std::string DefaultHostRootPath;
        static std::string HostFilesRootPath;

        static const int ChannelCount = 2;
        static const int RenderRate = 32000;
        static const int RenderCount = (RenderRate / 200);

        static const int MemoryPoolAlignSize    = 4096;
        static const int WorkBufferSize         = 8 * 1024 * 1024;
        static const int WaveBufferSize         = 14 * 1024 * 1024;
        static const int EffectBufferSize       = 12 * 1024 * 1024;
        static const int UpdateThreadStackSize = 1024 * 1024 * 64;
    private:
        nn::os::TimerEventType                  m_TimerEvent;
        nn::os::ThreadType                      m_UpdateThread;

        std::vector<VoiceData>                  m_Voice;
        int8_t                                  m_MainBus[ChannelCount];
        int8_t                                  m_AuxBusA[ChannelCount];
        int8_t                                  m_AuxBusB[ChannelCount];

        std::vector<AudioQueue>                 m_Queue;
        nn::audio::DelayType                    m_Delay;
        nn::audio::AudioRendererHandle          m_Handle;
        nn::audio::AudioRendererConfig          m_Config;
        nn::audio::FinalMixType                 m_FinalMix;
        nn::audio::SubMixType                   m_SubMix[2];

        ApplicationHeap                         m_Allocator;
        ApplicationHeap                         m_WaveBufferAllocator;
        ApplicationHeap                         m_EffectBufferAllocator;

        VoiceData                               m_SeVoice;
        bool                                    m_IsEnable;

        static NN_OS_ALIGNAS_THREAD_STACK char  ThreadStack[UpdateThreadStackSize];
    public:
        static Audio& GetInstance() NN_NOEXCEPT;

        static void UpdateAudioThreadFunction(void *arg) NN_NOEXCEPT;

        void Initialize() NN_NOEXCEPT;
        void Finalize() NN_NOEXCEPT;
        void Enable() NN_NOEXCEPT;
        void Disable() NN_NOEXCEPT;
        bool IsEnable() NN_NOEXCEPT;
        nn::Result Update() NN_NOEXCEPT;

        std::size_t ReadAudioFile(nn::audio::AdpcmHeaderInfo* header, nns::audio::WavFormat* format, AudioHandle handle, void** data) NN_NOEXCEPT;

        VoiceHandle LoadWaveFile(AudioHandle handle) NN_NOEXCEPT;
        VoiceHandle LoadAdpcmFile(AudioHandle handle) NN_NOEXCEPT;

        void Play(const VoiceHandle& handle) NN_NOEXCEPT;
        void Play(const VoiceHandle& handle, const nn::TimeSpan& delay) NN_NOEXCEPT;

        void PlayWav(const AudioHandle& handle) NN_NOEXCEPT;
        void PlayWav(const VoiceHandle& handle, const nn::TimeSpan& delay) NN_NOEXCEPT;

        void ReleaceVoice(const VoiceHandle& _handle) NN_NOEXCEPT;
        void ReLoadVoice(const VoiceHandle& _handle) NN_NOEXCEPT;

        nn::audio::AudioRendererState GetRendererState() NN_NOEXCEPT;
        nn::audio::VoiceType::PlayState GetSeVoiceState() NN_NOEXCEPT;
    };
}

#ifndef gAudio
#define gAudio (VibrationCollection::Audio::GetInstance())
#endif
