﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "AssertDialog.h"

namespace VibrationCollection
{
    nns::hidfw::layout::Dialog::DialogResult ShowAssertDialog(std::string title, std::string message, AssertDialogType dialogType, AssertButtonType buttonType) NN_NOEXCEPT
    {
        nns::hidfw::layout::Dialog dialog;
        nns::hidfw::layout::Dialog::DialogInfo info;

        // アサートの種類に応じて表示するボタンの種類を設定します
        switch (buttonType)
        {
        case VibrationCollection::AssertButtonType_OK:
        case VibrationCollection::AssertButtonType_Success:
            info.Buttons.Set<nns::hidfw::layout::Dialog::DialogButton::OK>();
            break;
        case VibrationCollection::AssertButtonType_YesNo:
            info.Buttons.Set<nns::hidfw::layout::Dialog::DialogButton::Yes>();
            info.Buttons.Set<nns::hidfw::layout::Dialog::DialogButton::NO>();
            break;
        default: NN_UNEXPECTED_DEFAULT;
        }

        info.Title = title;
        info.UpdateFuncParam = (void*)&dialogType;
        info.UpdateFunc = UpdateAssertDialog;
        info.DrawFuncParam = (void*)&dialogType;
        info.DrawerFunc = DrawAssertDialog;
        info.Message = message;

        dialog.SetDialogInfo(info);

        // ボタンを初期化します
        InitializeAssertDialogButtons(&dialog);

        return dialog.ShowDialog();
    }

    void InitializeAssertDialogButtons(nns::hidfw::layout::Dialog* pDialog) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pDialog);

        nns::hidfw::layout::Dialog& dialog = *pDialog;
        nns::hidfw::layout::Dialog::DialogInfo buttonInfo = dialog.GetDialogInfo();
        const nns::hidfw::layout::Dialog::DialogButtonSet& buttonSet = buttonInfo.Buttons;

        // 文字列に合わせてダイアログのサイズを変更します
        gTextWriter.SetScale(1.4f, 1.4f);
        const float width = std::max(gTextWriter.CalculateStringWidth(buttonInfo.Title.c_str()), gTextWriter.CalculateStringWidth(buttonInfo.Message.c_str())) + 64.f;
        const float height = gTextWriter.CalculateStringHeight(buttonInfo.Message.c_str()) + 172.f;
        gTextWriter.SetScale(1.f, 1.f);
        buttonInfo.Size = nn::util::MakeFloat2(width, height);

        const nn::util::Float2 basePos = NN_UTIL_FLOAT_2_INITIALIZER(640.f - buttonInfo.Size.x / 2.f + 12.f, 360.f + buttonInfo.Size.y / 2.f - 60.f);
        const nn::util::Float2 baseSize = NN_UTIL_FLOAT_2_INITIALIZER(buttonInfo.Size.x - 24.f, 48.f);
        const nn::util::Float2 margin = NN_UTIL_FLOAT_2_INITIALIZER(width / 2.f - 6, 0.f);

        nn::util::Color4u8 buttonColor[] =
        {
            nn::util::Color4u8(49, 119, 125, 255),
            nn::util::Color4u8(232, 57, 54, 255),
        };

        if (buttonSet.Test<nns::hidfw::layout::Dialog::DialogButton::OK>())
        {
            nns::hidfw::layout::Button okButton = nns::hidfw::layout::Dialog::DialogButton[nns::hidfw::layout::Dialog::DialogButton::OK::Index];
            okButton.SetPos(basePos.x, basePos.y);
            okButton.SetSize(baseSize.x, baseSize.y);
            okButton.SetMainColor(buttonColor[0]);
            //dialog.SetDialogButton(nns::hidfw::layout::Dialog::DialogButton::OK::Index, okButton);
        }
        if (buttonSet.Test<nns::hidfw::layout::Dialog::DialogButton::Yes>())
        {
            nns::hidfw::layout::Button yesButton = nns::hidfw::layout::Dialog::DialogButton[nns::hidfw::layout::Dialog::DialogButton::Yes::Index];
            yesButton.SetPos(basePos.x, basePos.y);
            yesButton.SetSize(baseSize.x / 2.f - 6, baseSize.y);
            yesButton.SetMainColor(buttonColor[0]);
            yesButton.SetTextColor(nn::util::Color4u8::White());
            dialog.SetDialogButton(nns::hidfw::layout::Dialog::DialogButton::Yes::Index, yesButton);
        }
        if (buttonSet.Test<nns::hidfw::layout::Dialog::DialogButton::NO>())
        {
            nns::hidfw::layout::Button noButton = nns::hidfw::layout::Dialog::DialogButton[nns::hidfw::layout::Dialog::DialogButton::NO::Index];
            noButton.SetPos(basePos.x + margin.x, basePos.y + margin.y);
            noButton.SetSize(baseSize.x / 2.f - 6, baseSize.y);
            noButton.SetMainColor(buttonColor[1]);
            noButton.SetTextColor(nn::util::Color4u8::White());
            dialog.SetDialogButton(nns::hidfw::layout::Dialog::DialogButton::NO::Index, noButton);
        }

        dialog.SetDialogInfo(buttonInfo);
    }

    void UpdateAssertDialog(void* executant, void* param) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(executant);
        NN_ASSERT_NOT_NULL(param);

        AssertDialogType type = *(AssertDialogType*)param;

        nns::hidfw::layout::Dialog* pDialog = (nns::hidfw::layout::Dialog*)executant;

        gTextWriter.SetTextColor(
            type == AssertDialogType_Error ? nn::util::Color4u8::Red() :
            type == AssertDialogType_Warning ? nn::util::Color4u8(64, 48, 8) :
            nn::util::Color4u8::Blue()
        );
        gTextWriter.SetScale(1.4f, 1.4f);
        gTextWriter.SetCursorX((1280.f - gTextWriter.CalculateStringWidth(pDialog->GetDialogInfo().Title.c_str())) / 2.f);
        gTextWriter.SetCursorY((720.f - pDialog->GetDialogInfo().Size.y) / 2.f + 16.f);
        gTextWriter.Print(pDialog->GetDialogInfo().Title.c_str());
        gTextWriter.SetTextColor(nn::util::Color4u8(16, 16, 16));
        gTextWriter.SetCursorX((1280.f - gTextWriter.CalculateStringWidth(pDialog->GetDialogInfo().Message.c_str())) / 2.f);
        gTextWriter.Print(("\n\n\n" + pDialog->GetDialogInfo().Message).c_str());
        gTextWriter.SetScale(1.f, 1.f);
    }

    void DrawAssertDialog(void* executant, void* param) NN_NOEXCEPT
    {
        NN_UNUSED(executant);
        NN_UNUSED(param);
    }
}
