﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>

#include "Json.h"

namespace nns { namespace hid { namespace util {

JsonDocument::JsonDocument() NN_NOEXCEPT
    : m_Mutex(true)
    , m_FileHandle()
    , m_FileSize(0)
    , m_ReadBuffer(nullptr)
    , m_JsonDocument()
{
}

JsonDocument::~JsonDocument() NN_NOEXCEPT
{
    Close();
}

JsonResult JsonDocument::Open(const char* jsonfile) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(jsonfile);

    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (OpenImpl(jsonfile).IsFailure())
    {
        return JsonResult::ReadFailed;
    }

    if (!Parse())
    {
        Close();
        return JsonResult::ReadFailed;
    }

    return JsonResult::Success;
}

void JsonDocument::Close() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lock(m_Mutex);

    if (IsOpened())
    {
        delete[] m_ReadBuffer;
        m_ReadBuffer = nullptr;
    }
}

nn::Result JsonDocument::OpenImpl(const char* jsonfile) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(jsonfile);

    Close();

    bool needsRollback = true;

    NN_RESULT_DO(nn::fs::OpenFile(&m_FileHandle, jsonfile, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(m_FileHandle);
    };

    NN_RESULT_DO(nn::fs::GetFileSize(&m_FileSize, m_FileHandle));

    m_ReadBuffer = new char[m_FileSize + 1];
    NN_UTIL_SCOPE_EXIT
    {
        if (needsRollback)
        {
            delete[] m_ReadBuffer;
            m_ReadBuffer = nullptr;
        }
    };

    NN_RESULT_DO(nn::fs::ReadFile(m_FileHandle, 0, m_ReadBuffer, m_FileSize));
    m_ReadBuffer[m_FileSize] = '\0';

    needsRollback = false;

    NN_RESULT_SUCCESS;
}

bool JsonDocument::Parse() NN_NOEXCEPT
{
    if (!IsOpened())
    {
        return false;
    }

    m_JsonDocument.ParseInsitu(m_ReadBuffer);
    if (m_JsonDocument.HasParseError())
    {
        // 解析失敗
        NN_LOG("[JsonDocument] ParseError: %d\n", m_JsonDocument.GetParseError());
        return false;
    }
    else if (!m_JsonDocument.IsObject())
    {
        // オブジェクトではない
        NN_LOG("[JsonDocument] Not an object\n");
        return false;
    }

    return true;
}

}}}  // nns::hid::util
