﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_StringUtil.h>

#include "../SimpleGfx.h"
#include "SimpleGfx_GuiTitle.h"

namespace nns { namespace sgx { namespace gui {

namespace {

const Point2D DrawOffset = { { 60.0f, 24.0f } };

const Point2D LineOffset = { { 28.0f, 80.0f } };

const float TitleTextSize = 36.0f;

}  // anonymous

Title::Title() NN_NOEXCEPT
    : Title("")
{
}

Title::Title(const char* text) NN_NOEXCEPT
    : m_pIconImage(nullptr)
    , m_TextColor()
    , m_Text()
{
    SetZ(TitleDefaultZ);
    SetText(text);
}

void Title::SetIconImage(ImageData* pImage) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pImage);

    NNS_SGX_GUI_SCOPED_LOCK;

    m_pIconImage = pImage;
}

void Title::SetText(const char* text) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(text);

    NNS_SGX_GUI_SCOPED_LOCK;

    nn::util::Strlcpy(m_Text, text, sizeof(m_Text));
}

void Title::Update() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    if (!IsVisible())
    {
        return;
    }
}

void Title::Render() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    if (!IsVisible())
    {
        return;
    }

    nns::sgx::Size canvasSize;
    nns::sgx::GetCanvasSize(&canvasSize);

    // ライン
    nns::sgx::DrawLine(
        LineOffset.x,
        LineOffset.y,
        canvasSize.width  - LineOffset.x,
        LineOffset.y,
        nns::sgx::Colors::Shadow(),
        1.0f);

    auto offset = DrawOffset;

    // アイコン
    if (m_pIconImage != nullptr && m_pIconImage->IsValid())
    {
        float adjustY = (TitleTextSize - m_pIconImage->size.height) / 2.0f;
        nns::sgx::DrawImage(*m_pIconImage, offset.x, offset.y + adjustY);
        offset.x += m_pIconImage->size.width + 8;
    }

    nns::sgx::ScopedFontContextSaver saver;

    // テキスト
    nns::sgx::SetTextColor(nns::sgx::Colors::Shadow());
    nns::sgx::SetTextSize(TitleTextSize);
    nns::sgx::DrawText(offset.x, offset.y, m_Text);
}

}}}  // nns::sgx::gui
