﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>

#include "../SimpleGfx_Types.h"
#include "SimpleGfx_GuiCommonDefs.h"
#include "SimpleGfx_GuiAttributes.h"
#include "SimpleGfx_GuiObject.h"
#include "SimpleGfx_GuiCursor.h"
#include "SimpleGfx_GuiCursorLink.h"
#include "SimpleGfx_GuiScrollBar.h"

namespace nns { namespace sgx { namespace gui {

// 前方宣言
class UiContainer;

/**
 * @brief   コンテナの ForEach で使用する関数宣言を簡略化するためのマクロです。
 */
#define NNS_SGX_GUI_CONTAINER_FOREACH(pObject, index, pContainer) \
    [](::nns::sgx::gui::DisplayObject* (pObject), \
       int (index), \
       ::nns::sgx::gui::UiContainer* (pContainer)) NN_NOEXCEPT

/**
 * @brief   UI コンテナの最大要素数です。
 */
const int UiContainerCapacity = 50;

/**
 * @brief   コンテナの ForEach で各要素に対して実行される関数の型です。
 */
typedef void(*ContainerForEachFunction)(DisplayObject* pObject, int index, UiContainer* pContainer);

/**
 * @brief   複数の UI オブジェクトを格納するためのコンテナクラスです。
 */
class UiContainer :
    public DisplayObject
{
    NN_DISALLOW_COPY(UiContainer);
    NN_DISALLOW_MOVE(UiContainer);

public:
    UiContainer() NN_NOEXCEPT;

    /**
     * @brief   コンテナかどうか判定します。
     */
    virtual bool IsContainer() const NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

    /**
     * @brief   フォーカスを解除します。
     */
    virtual void UnsetFocus() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   描画基準位置の X 座標を取得します。
     */
    virtual float GetRenderX() const NN_NOEXCEPT
    {
        return GetAbsoluteX();
    }

    /**
     * @brief   描画基準位置の Y 座標を取得します。
     */
    virtual float GetRenderY() const NN_NOEXCEPT
    {
        return GetAbsoluteY();
    }

    /**
     * @brief   クライアント領域の位置を取得します。
     */
    virtual Point2D GetClientAreaPosition() const NN_NOEXCEPT;

    /**
     * @brief   クライアント領域のサイズを取得します。
     */
    virtual Size GetClientAreaSize() const NN_NOEXCEPT;

    /**
     * @brief   クライアント領域の矩形を取得します。
     */
    virtual Rectangle GetClientArea() const NN_NOEXCEPT;

    /**
     * @brief   クライアント領域のスクロール量を取得します。
     */
    virtual Point2D GetScrollAmount() const NN_NOEXCEPT;

    /**
     * @brief   垂直スクロールの有効状態を設定します。
     */
    void SetVerticalLoopEnabled(bool isEnabled) NN_NOEXCEPT;

    /**
     * @brief   水平スクロールの有効状態を設定します。
     */
    void SetHorizontalLoopEnabled(bool isEnabled) NN_NOEXCEPT;

    /**
     * @brief   コンテナに子要素を追加します。
     */
    virtual void AddChild(DisplayObject* pChild) NN_NOEXCEPT;

    /**
     * @brief   コンテナの子要素から指定した要素を削除します。
     */
    virtual void RemoveChild(const DisplayObject* pChild) NN_NOEXCEPT;

    /**
     * @brief   子要素をすべて削除します。
     */
    void ClearChildren() NN_NOEXCEPT;

    /**
     * @brief   フォーカス状態の子要素を設定します。
     */
    void SetFocusedChild(DisplayObject* pChild) NN_NOEXCEPT;

    /**
     * @brief   コンテナの各子要素を引数として、指定した関数を実行します。
     */
    void ForEachChild(ContainerForEachFunction function) NN_NOEXCEPT;

    /**
     * @brief   子要素を Z オーダーの昇順 (画面の奥から手前順) にソートします。
     */
    void SortChildren() NN_NOEXCEPT;

    /**
     * @brief   フォーカス状態の子要素を取得します。
     */
    DisplayObject* GetFocusedChild() NN_NOEXCEPT
    {
        return m_pFocusedChild;
    }

    /**
     * @brief   カーソルのアニメーションを中断します。
     */
    void CancelCursorAnimation() NN_NOEXCEPT;

    /**
     * @brief   クライアント領域の描画時に呼び出されるハンドラを登録します。ハンドラは RenderClient の直後に呼び出されます。
     */
    void SetClientRenderHandler(GuiEventHandler handler, uintptr_t argument) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        m_HandlerForRenderClient.handler  = handler;
        m_HandlerForRenderClient.argument = argument;
    }

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE;

    virtual bool UpdateKeyInput() NN_NOEXCEPT NN_OVERRIDE;

    virtual bool UpdateTouchInput() NN_NOEXCEPT NN_OVERRIDE;

    virtual void CancelTouchInput() NN_NOEXCEPT NN_OVERRIDE;

    virtual void Render() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief   クライアント領域の描画処理です。継承先でクライアント領域の描画処理を行うために使用します。
     */
    virtual void RenderClient() NN_NOEXCEPT {}

protected:
    /**
     * @brief   カーソル移動可能な子要素を再構築します。
     */
    void RebuildKeyFocusableChildren() NN_NOEXCEPT;

    /**
     * @brief   カーソル位置を更新します。
     */
    void UpdateCursorPosition() NN_NOEXCEPT;

    /**
     * @brief   クライアント領域の描画を開始します。
     */
    virtual void BeginRenderClient() NN_NOEXCEPT;

    /**
     * @brief   クライアント領域の描画を終了します。
     */
    virtual void EndRenderClient() NN_NOEXCEPT;

protected:
    std::vector<DisplayObject*> m_Children;                 //!< コンテナの子要素
    std::vector<DisplayObject*> m_KeyFocusableChildren;     //!< キーフォーカス可能なコンテナの子要素
    ScrollBar                   m_VerticalScrollBar;        //!< 垂直スクロールバー
    ScrollBar                   m_HorizontalScrollBar;      //!< 水平スクロールバー
    Point2D                     m_Scroll;                   //!< スクロール位置
    Point2D                     m_MaxScroll;                //!< 最大スクロール量

private:
    /**
     * @brief   タッチ入力の状態です。
     */
    enum class TouchState
    {
        Idle,           //!< 通常
        TouchStart,     //!< タッチ開始
        Dragging,       //!< ドラッグ操作中
    };

    struct DestinationCandidate
    {
        DisplayObject* pObject;
        float          distance;
    };

private:
    /**
     * @brief   キーの入力方向を取得します。
     */
    static Direction GetKeyDirection() NN_NOEXCEPT;

    /**
     * @brief   スクロールに関する設定を調整します。
     */
    void UpdateScrollConfig() NN_NOEXCEPT;

    /**
     * @brief   フォーカス対象に応じてスクロール位置を調整します。
     */
    void UpdateScrollAmount() NN_NOEXCEPT;

    /**
     * @brief   キー入力によるカーソル移動処理を行います。
     */
    void UpdateCursorMoveByKeyInput() NN_NOEXCEPT;

    /**
     * @brief   カーソル移動先候補を探します。
     */
    DestinationCandidate FindCursorMoveCandidate(
        const Point2D& basePosition,
        Direction destDirection,
        const Point2D& vec) NN_NOEXCEPT;

private:
    bool                m_NeedsSortChildren;        //!< 子要素のソートが必要か
    bool                m_IsVerticalLoopEnabled;    //!< 垂直カーソルループが有効か
    bool                m_IsHorizontalLoopEnabled;  //!< 水平カーソルループが有効か
    Cursor              m_Cursor;                   //!< コンテナ内要素の選択用カーソル
    DisplayObject*      m_pFocusedChild;            //!< フォーカス状態の子要素
    TouchState          m_TouchState;               //!< タッチ状態
    Point2D             m_TouchStartPosition;       //!< タッチ開始位置
    Point2D             m_TouchStartScroll;         //!< タッチ開始時のスクロール位置

    GuiEventHandlerType m_HandlerForRenderClient;
};

}}}  // nns::sgx::gui
