﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/util/util_ScopeExit.h>

#include "../SimpleGfx.h"
#include "SimpleGfx_Gui.h"
#include "../../input/Input.h"

namespace nns { namespace sgx { namespace gui {

namespace
{

// スクロールバーの幅
const float ScrollBarWidth = 12.0f;

// ドラッグ開始と判定する移動距離
const float DragStartDistance = 20.0f;

}  // anonymous

UiContainer::UiContainer() NN_NOEXCEPT
    : m_Children()
    , m_KeyFocusableChildren()
    , m_VerticalScrollBar()
    , m_HorizontalScrollBar()
    , m_Scroll()
    , m_MaxScroll()
    , m_NeedsSortChildren(false)
    , m_IsVerticalLoopEnabled(false)
    , m_IsHorizontalLoopEnabled(false)
    , m_Cursor()
    , m_pFocusedChild(nullptr)
    , m_TouchState(TouchState::Idle)
    , m_TouchStartPosition()
    , m_TouchStartScroll()
    , m_HandlerForRenderClient()
{
    m_Children.reserve(UiContainerCapacity);
    m_VerticalScrollBar.SetVisible(false);
    m_VerticalScrollBar.SetMinValue(0);
    m_VerticalScrollBar.SetDirection(ScrollBar::ScrollDirection::Vertical);
    m_HorizontalScrollBar.SetVisible(false);
    m_HorizontalScrollBar.SetMinValue(0);
    m_HorizontalScrollBar.SetDirection(ScrollBar::ScrollDirection::Horizontal);
}

void UiContainer::UnsetFocus() NN_NOEXCEPT
{
    CancelTouchInput();
    DisplayObject::UnsetFocus();
}

Point2D UiContainer::GetClientAreaPosition() const NN_NOEXCEPT
{
    Point2D position;
    GetRenderPosition(&position);

    return position;
}

Size UiContainer::GetClientAreaSize() const NN_NOEXCEPT
{
    Size size;
    GetSize(&size);

    return size;
}

Rectangle UiContainer::GetClientArea() const NN_NOEXCEPT
{
    Rectangle rect;
    rect.position = GetClientAreaPosition();
    rect.size     = GetClientAreaSize();

    return rect;
}

Point2D UiContainer::GetScrollAmount() const NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    return m_Scroll;
}

void UiContainer::SetVerticalLoopEnabled(bool isEnabled) NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    m_IsVerticalLoopEnabled = isEnabled;
}

void UiContainer::SetHorizontalLoopEnabled(bool isEnabled) NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    m_IsHorizontalLoopEnabled = isEnabled;
}

void UiContainer::AddChild(DisplayObject* pChild) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pChild);
    NNS_SGX_GUI_SCOPED_LOCK;

    pChild->SetOwner(this);
    m_Children.push_back(pChild);
    m_NeedsSortChildren = true;

    UpdateScrollConfig();

    if (!pChild->IsKeyFocusable())
    {
        return;
    }

    m_KeyFocusableChildren.push_back(pChild);
    if (pChild->IsFocused())
    {
        if (m_pFocusedChild == nullptr)
        {
            // フォーカス状態の要素がなければ、追加された子をフォーカス状態にする
            SetFocusedChild(pChild);
        }
        else
        {
            // 既にフォーカス状態の要素があれば、そのフォーカス状態を維持
            pChild->UnsetFocus();
        }
    }
}

void UiContainer::RemoveChild(const DisplayObject* pChild) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pChild);
    NNS_SGX_GUI_SCOPED_LOCK;

    // 指定したコンテナから子要素を削除
    auto remove = [](decltype(m_Children) * pContainer, const DisplayObject* pEraseItem)
    {
        auto current = pContainer->begin();
        while (current != pContainer->end())
        {
            if (*current == pEraseItem)
            {
                (*current)->SetOwner(nullptr);
                current = pContainer->erase(current);
            }
            else
            {
                current++;
            }
        }
    };

    remove(&m_Children, pChild);
    remove(&m_KeyFocusableChildren, pChild);

    UpdateScrollConfig();

    // フォーカス状態の子要素が削除されたら、先頭の子要素にフォーカスを移す (仮)
    if (m_pFocusedChild == pChild)
    {
        if (m_KeyFocusableChildren.size() > 0)
        {
            SetFocusedChild(m_KeyFocusableChildren[0]);
        }
        else
        {
            SetFocusedChild(nullptr);
        }
    }
}

void UiContainer::ClearChildren() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    m_Children.clear();
    m_KeyFocusableChildren.clear();
    UpdateScrollConfig();
    SetFocusedChild(nullptr);
}

void UiContainer::SetFocusedChild(DisplayObject* pChild) NN_NOEXCEPT
{
    // 何か選択されたらカーソルを表示する
    if (m_pFocusedChild != nullptr)
    {
        m_Cursor.SetVisible(true);
    }

    if (m_pFocusedChild == pChild)
    {
        return;
    }

    if (m_pFocusedChild != nullptr)
    {
        m_pFocusedChild->UnsetFocus();
    }

    m_pFocusedChild = pChild;
    if (m_pFocusedChild != nullptr)
    {
        m_pFocusedChild->SetFocus();
    }

    UpdateCursorPosition();
    UpdateScrollAmount();
}

void UiContainer::ForEachChild(ContainerForEachFunction function) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(function);
    NNS_SGX_GUI_SCOPED_LOCK;

    int index = 0;
    for (auto child = m_Children.begin(); child != m_Children.end(); child++)
    {
        if (*child != nullptr)
        {
            function(*child, index, this);
        }
        index++;
    }
}

void UiContainer::SortChildren() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    auto predicator = [](DisplayObject* obj1, DisplayObject* obj2) NN_NOEXCEPT
    {
        return obj1->GetZ() < obj2->GetZ();
    };

    std::stable_sort(m_Children.begin(), m_Children.end(), predicator);
    std::stable_sort(m_KeyFocusableChildren.begin(), m_KeyFocusableChildren.end(), predicator);

    m_NeedsSortChildren = false;
}

void UiContainer::CancelCursorAnimation() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    m_Cursor.StopDecideEffect();
}

void UiContainer::Update() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    DisplayObject::Update();

    m_Cursor.SetEnabled(IsEnabled() && IsFocused());
    m_Cursor.Update();
    ForEachChild(NNS_SGX_GUI_CONTAINER_FOREACH(pChild, index, pContainer)
    {
        NN_ASSERT_NOT_NULL(pChild);
        NN_UNUSED(index);
        NN_UNUSED(pContainer);
        pChild->Update();
    });

    UpdateScrollConfig();

    if (m_TouchState == TouchState::Dragging)
    {
        return;
    }

    // Z 座標が大きい順に走査するので逆順
    for (auto iter = m_Children.rbegin(); iter != m_Children.rend(); iter++)
    {
        auto pChild = *iter;
        if (pChild == nullptr)
        {
            continue;
        }

        // XXX: タッチ位置にスクロール量を反映させるために、一時的にずらす
        Point2D originPosition;
        pChild->GetRenderPosition(&originPosition);
        pChild->SetPosition(originPosition - m_Scroll);
        bool isTouched = pChild->UpdateTouchInput();
        pChild->SetPosition(originPosition);

        if (isTouched)
        {
            const auto displayOffset = GetClientAreaPosition() - m_Scroll;
            if (pChild->IsKeyFocusable())
            {
                // タッチしたオブジェクトにフォーカスを移動
                SetFocusedChild(pChild);
                if (pChild->IsDecideEffectEnabled())
                {
                    m_Cursor.StartDecideEffect(displayOffset);
                }
            }
            else
            {
                // エフェクトだけ表示
                if (pChild->IsDecideEffectEnabled())
                {
                    m_Cursor.StartDecideEffect(*pChild, displayOffset);
                }
            }
            break;
        }
    }
}

bool UiContainer::UpdateKeyInput() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    if (!IsFocused())
    {
        return false;
    }

    if (m_pFocusedChild != nullptr)
    {
        if (m_pFocusedChild->UpdateKeyInput() &&
            m_pFocusedChild->IsDecideEffectEnabled())
        {
            m_Cursor.StartDecideEffect(GetClientAreaPosition() - m_Scroll);
            return true;
        }
    }

    UpdateCursorMoveByKeyInput();

    // フォーカスが当たっていれば必ず true
    return true;
}

bool UiContainer::UpdateTouchInput() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    if (!IsFocused() || m_MaxScroll.CalcSqrMagnitude() <= 0)
    {
        return false;
    }

    auto* tp = input::TouchPanel::GetInstance();

    Rectangle rect = { { 0, 0, GetWidth(), GetHeight() } };
    GetAbsolutePosition(&rect.position);
    auto rectF = rect.ToFloat4();
    if (tp->IsTouchTriggered(rectF))
    {
        if (m_TouchState == TouchState::Idle)
        {
            // タッチ開始
            m_TouchState       = TouchState::TouchStart;
            m_TouchStartScroll = m_Scroll;
            tp->GetTouchPoints(&m_TouchStartPosition.x, &m_TouchStartPosition.y, 1);
        }
    }
    else if (tp->IsTouched(rectF))
    {
        Point2D touchPoint;
        tp->GetTouchPoints(&touchPoint.x, &touchPoint.y, 1);
        auto diff = m_TouchStartPosition - touchPoint;

        if (m_TouchState == TouchState::TouchStart)
        {
            // 一定距離移動したらドラッグ開始と見なす
            if (diff.CalcSqrMagnitude() >= DragStartDistance * DragStartDistance)
            {
                m_TouchState = TouchState::Dragging;
                m_Cursor.SetVisible(false);

                // ドラッグを開始したら、子のタッチ操作をキャンセル
                ForEachChild(NNS_SGX_GUI_CONTAINER_FOREACH(pChild, index, pContainer)
                {
                    NN_ASSERT_NOT_NULL(pChild);
                    NN_UNUSED(index);
                    NN_UNUSED(pContainer);
                    pChild->CancelTouchInput();
                });
                return true;
            }
        }
        else if (m_TouchState == TouchState::Dragging)
        {
            // 移動距離に応じてスクロール
            m_Scroll.x = std::min(std::max(m_TouchStartScroll.x + diff.x, 0.0f), m_MaxScroll.x);
            m_Scroll.y = std::min(std::max(m_TouchStartScroll.y + diff.y, 0.0f), m_MaxScroll.y);

            m_HorizontalScrollBar.SetValue(m_Scroll.x);
            m_VerticalScrollBar.SetValue(m_Scroll.y);

            return true;
        }
    }
    else if (tp->IsTouchReleased())
    {
        if (m_TouchState == TouchState::Dragging)
        {
        }

        CancelTouchInput();
    }

    return false;
}

void UiContainer::CancelTouchInput() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    ForEachChild(NNS_SGX_GUI_CONTAINER_FOREACH(pChild, index, pContainer)
    {
        NN_ASSERT_NOT_NULL(pChild);
        NN_UNUSED(index);
        NN_UNUSED(pContainer);
        pChild->CancelTouchInput();
    });

    m_TouchState = TouchState::Idle;
}

void UiContainer::Render() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    if (m_NeedsSortChildren)
    {
        SortChildren();
    }

    if (!IsVisible())
    {
        return;
    }

    BeginRenderClient();

    ForEachChild(NNS_SGX_GUI_CONTAINER_FOREACH(pChild, index, pContainer)
    {
        NN_ASSERT_NOT_NULL(pChild);
        NN_UNUSED(index);
        pChild->RenderWithOffset(-pContainer->m_Scroll);
    });

    {
        // カーソルはクライアント領域外に描画する
        EndRenderClient();

        const auto renderPosition = GetClientAreaPosition() - m_Scroll;
        if (m_Cursor.IsVisible())
        {
            m_Cursor.RenderWithOffset(renderPosition);
        }
        m_Cursor.RenderDecideEffect(renderPosition);

        BeginRenderClient();
    }

    RenderClient();
    m_HandlerForRenderClient.Invoke(this);

    EndRenderClient();

    const Point2D pos = { { GetX(), GetY() } };
    m_VerticalScrollBar.RenderWithOffset(pos);
    m_HorizontalScrollBar.RenderWithOffset(pos);
}

void UiContainer::RebuildKeyFocusableChildren() NN_NOEXCEPT
{
    m_KeyFocusableChildren.clear();

    for (auto pChild : m_Children)
    {
        if (pChild->IsKeyFocusable())
        {
            m_KeyFocusableChildren.push_back(pChild);
        }
    }
}

void UiContainer::UpdateCursorPosition() NN_NOEXCEPT
{
    if (m_pFocusedChild != nullptr &&
        m_pFocusedChild->IsKeyFocusable() &&
        m_pFocusedChild->IsVisible())
    {
        Point2D position;
        Size size;
        m_pFocusedChild->GetRenderPosition(&position);
        m_pFocusedChild->GetSize(&size);
        //ApplyScroll(&position, m_Scroll);

        m_Cursor.SetPosition(position);
        m_Cursor.SetSize(size);
        m_Cursor.SetVisible(true);
    }
    else
    {
        m_Cursor.SetVisible(false);
    }
}

void UiContainer::BeginRenderClient() NN_NOEXCEPT
{
    // 描画可能範囲をクライアント領域に制限する
    auto rect = GetClientArea();
    NNS_SGX_GUI_DRAW_CLIENT_AREA_DEBUG(rect);
    ApplyRenderArea(rect);

#if 0
    // スクロールされている場合は描画位置をずらす
    if (m_Scroll.CalcSqrMagnitude() > 0.0f)
    {
        rect.x = rect.y = 0;
        ApplyScroll(&rect.position, m_Scroll);
        rect.width  += m_Scroll.x;
        rect.height += m_Scroll.y;
        ApplyRenderArea(rect);
    }
#endif
}

void UiContainer::EndRenderClient() NN_NOEXCEPT
{
#if 0
    // スクロールされている場合は描画位置を戻す
    if (m_Scroll.CalcSqrMagnitude() > 0.0f)
    {
        RestoreRenderArea();
    }
#endif

    // 描画範囲の制限を解除
    RestoreRenderArea();
}

Direction UiContainer::GetKeyDirection() NN_NOEXCEPT
{
    auto controllers = input::GetNpadControllers();
    controllers.push_back(input::NpadManager::GetInstance()->GetDummyController());

    for (auto iter = controllers.begin(); iter != controllers.end(); iter++)
    {
        auto pController = *iter;
        if (pController->IsRepeated<nn::hid::NpadButton::Up>() ||
            pController->IsRepeated<nn::hid::NpadButton::StickLUp>() ||
            pController->IsRepeated<nn::hid::NpadButton::StickRUp>())
        {
            return Direction::Up;
        }
        else if (pController->IsRepeated<nn::hid::NpadButton::Down>() ||
            pController->IsRepeated<nn::hid::NpadButton::StickLDown>() ||
            pController->IsRepeated<nn::hid::NpadButton::StickRDown>())
        {
            return Direction::Down;
        }
        else if (pController->IsRepeated<nn::hid::NpadButton::Left>() ||
            pController->IsRepeated<nn::hid::NpadButton::StickLLeft>() ||
            pController->IsRepeated<nn::hid::NpadButton::StickRLeft>())
        {
            return Direction::Left;
        }
        else if (pController->IsRepeated<nn::hid::NpadButton::Right>() ||
            pController->IsRepeated<nn::hid::NpadButton::StickLRight>() ||
            pController->IsRepeated<nn::hid::NpadButton::StickRRight>())
        {
            return Direction::Right;
        }
    }

    return Direction::None;
}

void UiContainer::UpdateScrollConfig() NN_NOEXCEPT
{
    m_MaxScroll.Set(0, 0);

    ForEachChild(NNS_SGX_GUI_CONTAINER_FOREACH(pChild, index, pContainer)
    {
        NN_ASSERT_NOT_NULL(pChild);
        NN_UNUSED(index);

        if (!pChild->IsVisible())
        {
            return;
        }

        auto& maxScroll = pContainer->m_MaxScroll;

        const auto clientSize = pContainer->GetClientAreaSize();
        float overX = (pChild->GetX() + pChild->GetWidth())  - clientSize.width;
        float overY = (pChild->GetY() + pChild->GetHeight()) - clientSize.height;
        maxScroll.x = std::max(maxScroll.x, overX);
        maxScroll.y = std::max(maxScroll.y, overY);
    });

    auto width  = GetWidth();
    auto height = GetHeight();
    m_HorizontalScrollBar.SetVisible(m_MaxScroll.x > 0.0f);
    m_HorizontalScrollBar.SetPosition(0, height - m_HorizontalScrollBar.GetHeight());
    m_HorizontalScrollBar.SetSize(width, ScrollBarWidth);
    m_HorizontalScrollBar.SetMaxValue(m_MaxScroll.x);
    m_HorizontalScrollBar.SetStepValue((width + m_MaxScroll.x) / width);

    m_VerticalScrollBar.SetVisible(m_MaxScroll.y > 0.0f);
    m_VerticalScrollBar.SetPosition(width - m_VerticalScrollBar.GetWidth(), 0);
    m_VerticalScrollBar.SetSize(ScrollBarWidth, height);
    m_VerticalScrollBar.SetMaxValue(m_MaxScroll.y);
    m_VerticalScrollBar.SetStepValue((height + m_MaxScroll.y) / height);
}

void UiContainer::UpdateScrollAmount() NN_NOEXCEPT
{
    if (m_pFocusedChild == nullptr || !m_pFocusedChild->IsVisible())
    {
        return;
    }

    const auto size = GetClientAreaSize();

    float diffX     = m_pFocusedChild->GetX() - m_Scroll.x;
    float diffRight = size.width - (diffX + m_pFocusedChild->GetWidth());
    if (diffX < 0.0f)
    {
        // 左にはみ出ているので左にスクロール
        m_Scroll.x += diffX;
        //NN_LOG(" >> Scroll left: (%.f, %.f)\n", m_Scroll.x, m_Scroll.y);
    }
    else if (diffRight < 0.0f)
    {
        // 右にはみ出ているので右にスクロール
        m_Scroll.x -= diffRight;
        //NN_LOG(" >> Scroll right: (%.f, %.f)\n", m_Scroll.x, m_Scroll.y);
    }

    float diffY      = m_pFocusedChild->GetY() - m_Scroll.y;
    float diffBottom = size.height - (diffY + m_pFocusedChild->GetHeight());
    if (diffY < 0.0f)
    {
        // 上にはみ出ているので上にスクロール
        m_Scroll.y += diffY;
        //NN_LOG(" >> Scroll up: (%.f, %.f)\n", m_Scroll.x, m_Scroll.y);
    }
    else if (diffBottom < 0.0f)
    {
        // 下にはみ出ているので下にスクロール
        m_Scroll.y -= diffBottom;
        //NN_LOG(" >> Scroll down: (%.f, %.f)\n", m_Scroll.x, m_Scroll.y);
    }

    m_HorizontalScrollBar.SetValue(m_Scroll.x);
    m_VerticalScrollBar.SetValue(m_Scroll.y);
}

void UiContainer::UpdateCursorMoveByKeyInput() NN_NOEXCEPT
{
    // キーフォーカス不可なオブジェクトにフォーカスが当たっている場合は移動しない
    if (m_pFocusedChild != nullptr &&
        !m_pFocusedChild->IsKeyFocusable())
    {
        return;
    }

    Direction keyDirection = GetKeyDirection();
    Direction destDirection;

    // キー入力方向に対応した移動ベクトルと移動先基準点を設定
    Point2D vec;
    switch (keyDirection)
    {
    case Direction::Up:
        destDirection = Direction::Down;
        vec = { {  0, -1 } };
        break;
    case Direction::Down:
        destDirection = Direction::Up;
        vec = { {  0,  1 } };
        break;
    case Direction::Left:
        destDirection = Direction::Right;
        vec = { { -1,  0 } };
        break;
    case Direction::Right:
        destDirection = Direction::Left;
        vec = { {  1,  0 } };
        break;
    default:
        return;
    }

    // 移動元基準点
    auto basePosition = m_Cursor.GetAnchorPoint(keyDirection);

    auto candidate = FindCursorMoveCandidate(basePosition, destDirection, vec);
    if (candidate.pObject == nullptr)
    {
        // 見つからなければカーソルループ判定
        if (m_IsVerticalLoopEnabled)
        {
            if (keyDirection == Direction::Up)
            {
                basePosition.y = 9999.0f;
                candidate = FindCursorMoveCandidate(basePosition, destDirection, vec);
            }
            else if (keyDirection == Direction::Down)
            {
                basePosition.y = 0.0f;
                candidate = FindCursorMoveCandidate(basePosition, destDirection, vec);
            }
        }
        else if (m_IsHorizontalLoopEnabled)
        {
            if (keyDirection == Direction::Left)
            {
                basePosition.x = 9999.0f;
                candidate = FindCursorMoveCandidate(basePosition, destDirection, vec);
            }
            else if (keyDirection == Direction::Right)
            {
                basePosition.x = 0.0f;
                candidate = FindCursorMoveCandidate(basePosition, destDirection, vec);
            }
        }
    }

    // 移動先候補があればカーソルを移動
    if (candidate.pObject != nullptr)
    {
        SetFocusedChild(candidate.pObject);
        //CancelCursorAnimation();
        PlaySystemSe(SystemSe::Cursor);
    }
    else
    {
        PlaySystemSe(SystemSe::Buzzer);
    }
}

UiContainer::DestinationCandidate UiContainer::FindCursorMoveCandidate(
    const Point2D& basePosition,
    Direction destDirection,
    const Point2D& vec) NN_NOEXCEPT
{
    DestinationCandidate candidate = { nullptr, static_cast<float>(0xFFFFFFFF) };

    // カーソル移動先候補の選定
    for (auto pChild : m_KeyFocusableChildren)
    {
        // 非表示やフォーカス済みの要素は除外
        if (!pChild->IsVisible() || pChild == m_pFocusedChild)
        {
            continue;
        }

        auto dstPosition = pChild->GetAnchorPoint(destDirection);
        float diffX = dstPosition.x - basePosition.x;
        float diffY = dstPosition.y - basePosition.y;

        // キー入力方向ではない (= 内積が負) なら除外
        // 直交はとりあえず許可
        if (diffX * vec.x + diffY * vec.y < 0.0f)
        {
            continue;
        }

        // 距離が候補より遠ければ除外
        float distance = diffX * diffX + diffY * diffY;
        if (candidate.distance < distance)
        {
            continue;
        }

        // 候補を記録
        candidate.pObject  = pChild;
        candidate.distance = distance;
    }

    return candidate;
}

}}}  // nns::sgx::gui
