﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/hid/hid_NpadCommonTypes.h>
#include <nn/util/util_BitFlagSet.h>

#include "../SimpleGfx_Types.h"
#include "SimpleGfx_GuiCommon.h"

namespace nns { namespace sgx { namespace gui {

/**
 * @brief   チェックボックステキストの最大長
 */
const int CheckBoxTextLengthMax = 64;

/**
 * @brief   チェックボックスの見た目です。
 */
struct CheckBoxAppearance
{
    typedef nn::util::BitFlagSet<32, CheckBoxAppearance>::Flag<0> Checked;
    typedef nn::util::BitFlagSet<32, CheckBoxAppearance>::Flag<1> Enabled;
    typedef nn::util::BitFlagSet<32, CheckBoxAppearance>::Flag<2> Focused;
    typedef nn::util::BitFlagSet<32, CheckBoxAppearance>::Flag<3> Pressed;
};

typedef ::nn::util::BitFlagSet<32, CheckBoxAppearance> CheckBoxAppearanceSet;

/**
 * @brief   チェックボックスを扱うクラスです。
 */
class CheckBox :
    public DisplayObject
{
    NN_DISALLOW_COPY(CheckBox);
    NN_DISALLOW_MOVE(CheckBox);

public:
    CheckBox() NN_NOEXCEPT;

    /**
     * @brief   キー操作によってフォーカス可能かどうか取得します。
     */
    virtual bool IsKeyFocusable() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_IsKeyFocusable;
    }

    /**
     * @brief   キー操作によってフォーカス可能か設定します。
     */
    void SetKeyFocusable(bool isFocusable) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        m_IsKeyFocusable = isFocusable;
    }

    /**
     * @brief   ボタン押下として扱う Npad のボタンを設定します。
     */
    void SetButtonMaskForPress(nn::hid::NpadButtonSet mask) NN_NOEXCEPT
    {
        m_ButtonMaskForPress = mask;
    }

    /**
     * @brief   ボタン押下時の処理を設定します。
     *
     * @details ハンドラのユーザー引数には常に 0 が渡されます。
     */
    void SetPushEventHandler(GuiEventHandler handler) NN_NOEXCEPT
    {
        SetPushEventHandler(handler, 0);
    }

    /**
     * @brief   ボタン押下時の処理を設定します。
     */
    void SetPushEventHandler(GuiEventHandler handler, uintptr_t userArg) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        m_HandlerForPushEvent.handler  = handler;
        m_HandlerForPushEvent.argument = userArg;
    }

    /**
     * @brief   押下されているか判定します。
     */
    bool IsPressed() const NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        auto state = GetState();
        return state == CheckBoxState::PressByKey || state == CheckBoxState::PressByTouch;
    }

    /**
     * @brief   チェックされているか返します。
     */
    bool IsChecked() const NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        return m_IsChecked;
    }

    /**
     * @brief   チェック状態を設定します。
     */
    void SetChecked(bool isChecked) NN_NOEXCEPT
    {
        NNS_SGX_GUI_SCOPED_LOCK;

        m_IsChecked = isChecked;
    }

    /**
     * @brief   ボタンの見た目を取得します。
     */
    CheckBoxAppearanceSet GetAppearance() const NN_NOEXCEPT;

    /**
     * @brief   ボタンのテキストを UTF-8 で設定します。
     */
    void SetText(const char* text) NN_NOEXCEPT;

    /**
     * @brief   ボタンのテキストを UTF-32 で設定します。
     */
    void SetText(const uint32_t* text) NN_NOEXCEPT;

    /**
     * @brief   テキストの描画サイズを設定します。
     */
    void SetTextSize(float size) NN_NOEXCEPT
    {
        m_TextSize = size;
        UpdateAutoSize();
    }

    /**
     * @brief   押下時の処理を実行します。
     */
    void PerformPress() NN_NOEXCEPT;

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE;

    virtual bool UpdateKeyInput() NN_NOEXCEPT NN_OVERRIDE;

    virtual bool UpdateTouchInput() NN_NOEXCEPT NN_OVERRIDE;

    virtual void CancelTouchInput() NN_NOEXCEPT NN_OVERRIDE;

    virtual void Render() NN_NOEXCEPT NN_OVERRIDE;

protected:
    /**
     * @brief   ボタンの状態です。
     */
    enum class CheckBoxState
    {
        Init,           //!< 初期状態
        Idle,           //!< 通常
        PressByTouch,   //!< タッチによる押下状態
        PressByKey,     //!< キー操作による押下状態
    };

protected:
    /**
     * @brief   チェックボックスの状態を取得します。
     */
    CheckBoxState GetState() const NN_NOEXCEPT
    {
        return m_State;
    }

    /**
     * @brief   チェックボックスの状態を設定します。
     */
    void SetState(CheckBoxState state) NN_NOEXCEPT
    {
        m_State = state;
    }

    /**
     * @brief   チェックボックスのサイズを自動調整します。
     */
    void UpdateAutoSize() NN_NOEXCEPT;

private:
    bool                        m_IsKeyFocusable;
    bool                        m_IsChecked;
    nn::hid::NpadButtonSet      m_ButtonMaskForPress;
    uint32_t                    m_Text[CheckBoxTextLengthMax];
    float                       m_TextSize;         //!< 文字の描画サイズ
    CheckBoxState               m_State;
    GuiEventHandlerType         m_HandlerForPushEvent;
};

}}}  // nns::sgx::gui
