﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <mutex>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_StringUtil.h>

#include "../SimpleGfx.h"
#include "SimpleGfx_GuiButtonBase.h"
#include "SimpleGfx_GuiUtil.h"
#include "../../input/Input.h"

namespace nns { namespace sgx { namespace gui {

ButtonBase::ButtonBase() NN_NOEXCEPT
    : m_State(ButtonState_Init)
    , m_ButtonMaskForPress(nn::hid::NpadButton::A::Mask)
    , m_HandlerForPushEvent()
{
    SetDecideEffectEnabled(true);
}

ButtonAppearance ButtonBase::GetAppearance() const NN_NOEXCEPT
{
    if (!IsEnabled())
    {
        return ButtonAppearance::Disabled;
    }
    else if (IsPressed())
    {
        return ButtonAppearance::Pressed;
    }
    else if (IsFocused())
    {
        return ButtonAppearance::Focused;
    }
    else
    {
        return ButtonAppearance::Normal;
    }
}

void ButtonBase::PerformPress() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    if (!IsVisible() || !IsEnabled())
    {
        // 押せない
        return;
    }

    SetState(ButtonState_Idle);
    m_HandlerForPushEvent.Invoke(this);
}

void ButtonBase::Update() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    DisplayObject::Update();

    if (!IsVisible() || !IsEnabled())
    {
        SetState(ButtonState_Idle);
        return;
    }
    else if (!IsFocused() && GetState() == ButtonState_PressByKey)
    {
        SetState(ButtonState_Idle);
        return;
    }
}

bool ButtonBase::UpdateKeyInput() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    if (!IsFocused())
    {
        return false;
    }

    auto onPush = [](ButtonBase* pSelf)
    {
        // 押下状態からリリースされたらボタン押し判定
        PlaySystemSe(SystemSe::Decide);
        //pSelf->m_HandlerForPushEvent.Invoke(pSelf);
        pSelf->ReservePostUpdateEvent(pSelf->m_HandlerForPushEvent);
    };

    // パッド操作による押下判定
    auto controllers = input::GetNpadControllers();
    controllers.push_back(input::NpadManager::GetInstance()->GetDummyController());

    auto state = GetState();
    for (auto pController : controllers)
    {
        if (pController->IsAnyTriggered(m_ButtonMaskForPress))
        {
            if (IsEnabled())
            {
                // ボタン押下開始
                SetState(ButtonState_PressByKey);
#if defined(NNS_SGX_GUI_PUSH_ON_RELEASE)
                PlaySystemSe(SystemSe::Press);
#else
                onPush(this);
                return true;
#endif  // if defined(NNS_SGX_GUI_PUSH_ON_RELEASE)
            }
            else
            {
                // 禁止状態
                PlaySystemSe(SystemSe::Buzzer);
            }
            return false;
        }
        else if (pController->IsAnyPressed(m_ButtonMaskForPress))
        {
            // 押下継続
            // カーソル移動によるキャンセルを受け付けるため、入力は処理されていないものとして振る舞う
            return false;
        }
        else if (!pController->IsAnyPressed(m_ButtonMaskForPress) &&
            state == ButtonState_PressByKey)
        {
            SetState(ButtonState_Idle);

#if defined(NNS_SGX_GUI_PUSH_ON_RELEASE)
            onPush(this);
            return true;
#else
            return false;
#endif  // if defined(NNS_SGX_GUI_PUSH_ON_RELEASE)
        }
    }

    return false;
}

bool ButtonBase::UpdateTouchInput() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    auto* tp = input::TouchPanel::GetInstance();

    auto rect  = GetHitArea().ToFloat4();
    auto state = GetState();
    if (tp->IsTouchTriggered(rect))
    {
        if (state == ButtonState_Idle)
        {
            if (IsEnabled())
            {
                // ボタン押下開始
                SetState(ButtonState_PressByTouch);
                PlaySystemSe(SystemSe::Press);
            }
            else
            {
                // 禁止状態
                PlaySystemSe(SystemSe::Buzzer);
            }
        }
    }
    else if (!tp->IsTouched(rect) &&
        state == ButtonState_PressByTouch)
    {
        SetState(ButtonState_Idle);

        // 押下状態からリリースされたらボタン押し判定
        if (tp->IsTouchReleased(rect))
        {
            PlaySystemSe(SystemSe::Decide);
            //m_HandlerForPushEvent.Invoke(this);
            ReservePostUpdateEvent(m_HandlerForPushEvent);
            return true;
        }
    }

    return false;
}

void ButtonBase::CancelTouchInput() NN_NOEXCEPT
{
    NNS_SGX_GUI_SCOPED_LOCK;

    SetState(ButtonState_Idle);
}

}}}  // nns::sgx::gui
