﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>

#include <algorithm>
#include <cmath>
#include <cstdarg>
#include <cstdio>
#include <cstring>
#include <mutex>

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include "../SimpleGfx_Types.h"
#include "../SimpleGfx_TypesInternal.h"
#include "SimpleGfx_ApiInternal.h"

namespace nns { namespace sgx { namespace detail {

namespace
{

// 描画領域スタックの要素数
const int RenderAreaStackCount = 32;

ShaderType  g_CurrentShader  = ShaderType::Simple;
Size        g_CanvasSize     = DefaultCanvasSize;

ShaderPack* g_pSimpleShader  = nullptr;
ShaderPack* g_pTextureShader = nullptr;

// 描画領域スタック
class RenderAreaStack final
{
public:
    RenderAreaStack() NN_NOEXCEPT :
        m_Areas(),
        m_Count(0)
    {}

    void Initialize() NN_NOEXCEPT
    {
        m_Count = 0;
    }

    void Apply(const Rectangle& area) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_LESS(m_Count, NN_ARRAY_SIZE(m_Areas));

        m_Areas[m_Count] = area;
        m_Count++;

        Rectangle finalArea;
        Point2D offset;
        GetFinalRenderArea(&finalArea, &offset);
        SetArea(finalArea, offset);
    }

    void Restore() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_GREATER(m_Count, 0);

        m_Count--;
        if (m_Count > 0)
        {
            Rectangle area;
            Point2D offset;
            GetFinalRenderArea(&area, &offset);
            SetArea(area, offset);
        }
        else
        {
            detail::ResetRenderArea();
        }
    }

    void GetFinalRenderArea(Rectangle* pOutArea, Point2D* pOutOffset) const NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutArea);
        NN_ASSERT_NOT_NULL(pOutOffset);

        auto size = detail::GetCanvasSize();

        if (m_Count <= 0)
        {
            *pOutArea   = { { 0, 0, size.width, size.height } };
            *pOutOffset = { { 0, 0 } };
        }

        Point2D position = { { 0, 0 } };
        Point2D offset   = { { 0, 0 } };

        // 全領域を順にマージする
        for (int i = 0; i < m_Count; i++)
        {
            const auto& rect = m_Areas[i];
            if (rect.x < 0)
            {
                offset.x += rect.x;
            }
            else
            {
                offset.x = std::min(offset.x + rect.x, 0.0f);
            }
            if (rect.y < 0)
            {
                offset.y += rect.y;
            }
            else
            {
                offset.y = std::min(offset.y + rect.y, 0.0f);
            }

            position.x += std::max(rect.x, 0.0f);
            position.y += std::max(rect.y, 0.0f);
            size.width  = std::min(std::max(size.width  - rect.x, 0.0f), rect.width);
            size.height = std::min(std::max(size.height - rect.y, 0.0f), rect.height);
        }

        pOutArea->position = position;
        pOutArea->size     = size;

        *pOutOffset = offset;
    }

private:
    void SetArea(const Rectangle& area, const Point2D& offset) NN_NOEXCEPT
    {
        detail::SetRenderArea(area, offset);
    }

private:
    Rectangle m_Areas[RenderAreaStackCount];
    int       m_Count;
};

RenderAreaStack g_RenderAreaStack;

}  // anonymous

void CreateShader(
    ShaderPack* pOutShader,
    const GLchar* const* vertexProgram,
    const GLchar* const* fragmentProgram) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutShader);
    NN_ASSERT_NOT_NULL(vertexProgram);
    NN_ASSERT_NOT_NULL(fragmentProgram);

    GLint result;
    GLchar shaderLog[1024];
    GLsizei shaderLogSize;

    // Vertex shader
    auto vertexId = glCreateShader(GL_VERTEX_SHADER);
    NN_ASSERT(vertexId != 0, "Failed to create vertex shader\n");
    glShaderSource(vertexId, 1, vertexProgram, 0);
    glCompileShader(vertexId);
    glGetShaderiv(vertexId, GL_COMPILE_STATUS, &result);
    if (!result)
    {
        glGetShaderInfoLog(vertexId, sizeof(shaderLog), &shaderLogSize, shaderLog);
        NN_ASSERT(false, "Failed to compile vertex shader: %s\n", shaderLog);
    }

    // Fragment shader
    auto fragmentId = glCreateShader(GL_FRAGMENT_SHADER);
    NN_ASSERT(fragmentId != 0, "Failed to create fragment shader\n");
    glShaderSource(fragmentId, 1, fragmentProgram, 0);
    glCompileShader(fragmentId);
    glGetShaderiv(fragmentId, GL_COMPILE_STATUS, &result);
    if (!result)
    {
        glGetShaderInfoLog(fragmentId, sizeof(shaderLog), &shaderLogSize, shaderLog);
        NN_ASSERT(false, "Failed to compile fragment shader: %s\n", shaderLog);
    }

    // コンパイル
    auto programId = glCreateProgram();
    NN_ASSERT(programId != 0, "Failed to create shader program\n");

    glAttachShader(programId, vertexId);
    glAttachShader(programId, fragmentId);
    glLinkProgram(programId);

    pOutShader->vertexId = vertexId;
    pOutShader->fragmentId = fragmentId;
    pOutShader->programId = programId;
}

void DeleteShader(const ShaderPack& shader) NN_NOEXCEPT
{
    glDetachShader(shader.programId, shader.vertexId);
    glDetachShader(shader.programId, shader.fragmentId);

    glDeleteShader(shader.vertexId);
    glDeleteShader(shader.fragmentId);

    glDeleteProgram(shader.programId);
}

void RegisterShader(
    ShaderPack* pSimpleShader,
    ShaderPack* pTextureShader) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pSimpleShader);
    NN_ASSERT_NOT_NULL(pTextureShader);

    g_pSimpleShader  = pSimpleShader;
    g_pTextureShader = pTextureShader;
}

bool ChangeShader(ShaderType type) NN_NOEXCEPT
{
    if (g_CurrentShader == type)
    {
        return false;
    }

    switch (type)
    {
    case ShaderType::None:
        glUseProgram(0);
        break;

    case ShaderType::Simple:
        glUseProgram(g_pSimpleShader->programId);
        break;

    case ShaderType::Texture:
        glUseProgram(g_pTextureShader->programId);
        break;

    default:
        NN_UNEXPECTED_DEFAULT;
    }

    bool needsRefresh = g_CurrentShader == ShaderType::None;
    g_CurrentShader = type;

    return needsRefresh;
}

Size GetCanvasSize() NN_NOEXCEPT
{
    return g_CanvasSize;
}

void SetCanvasSize(const Size& size) NN_NOEXCEPT
{
    g_CanvasSize = size;
}

void SetRenderArea(const Rectangle& rect, const Point2D& offset) NN_NOEXCEPT
{
    NN_UNUSED(offset);

    auto canvasSize = GetCanvasSize();

    // 描画範囲を制限
    glEnable(GL_SCISSOR_TEST);
    glScissor(
        static_cast<GLint>(rect.x),
        static_cast<GLint>(canvasSize.height - rect.y - rect.height),
        static_cast<GLsizei>(rect.width),
        static_cast<GLsizei>(rect.height)
    );

    // 縮尺は変えずに原点を変える
    glViewport(
        static_cast<GLint>(rect.x + offset.x),
        static_cast<GLint>(-rect.y - offset.y),
        static_cast<GLsizei>(canvasSize.width),
        static_cast<GLsizei>(canvasSize.height)
    );
}

void SetRenderArea(float x, float y, float width, float height) NN_NOEXCEPT
{
    SetRenderArea({ { x, y, width, height } }, { { 0, 0 } });
}

void ResetRenderArea() NN_NOEXCEPT
{
    auto size = GetCanvasSize();
    SetRenderArea(0.0f, 0.0f, size.width, size.height);
    glDisable(GL_SCISSOR_TEST);
}

void ApplyRenderArea(const Rectangle& rect) NN_NOEXCEPT
{
    g_RenderAreaStack.Apply(rect);
}

void RestoreRenderArea() NN_NOEXCEPT
{
    g_RenderAreaStack.Restore();
}

void GetCurrentRenderArea(Rectangle* pOutArea, Point2D* pOutOffset) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pOutArea);
    NN_ASSERT_NOT_NULL(pOutOffset);

    g_RenderAreaStack.GetFinalRenderArea(pOutArea, pOutOffset);
}

void GetCurrentRenderArea(Rectangle* pOutArea) NN_NOEXCEPT
{
    Point2D offset;
    NN_UNUSED(offset);
    GetCurrentRenderArea(pOutArea, &offset);
}

}}}  // nns::sgx::detail
